package Application::Model::AgreementChecker;

use qbit;
use Application::Model::AgreementChecker::_Agreement;
use QBit::Validator;

use base qw(QBit::Application::Model);

use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

use PiConstants qw($ADINSIDE_CLIENT_ID);

sub accessor {'agreement_checker'}

__PACKAGE__->model_accessors(
    api_balance  => 'Application::Model::API::Yandex::Balance',
    vip_partners => 'Application::Model::VIP_Partners',
    users        => 'Application::Model::Users',
);

=head2 get_all_products

    my $all_products = $app->agreement_checker->get_all_products();

Возвращает полный список продуктов, для которых модель проверяет возможность
запуска. Возвращается структура вида:

    {
        context_on_site_direct => 1,
        context_on_site_rtb    => 1,
        context_on_site_stripe => 1,
        search_on_site_direct  => 1,
        search_on_site_market  => 1,
        search_on_site_mcb     => 1,
        search_on_site_premium => 1,
    }

=cut

sub get_all_products {
    my ($self, @opts) = @_;

    throw Exception::Validation::BadArguments gettext("Got unknown parameters: %s", join(", ", @opts)) if @opts;

    my $products = {
        map {$_ => TRUE}
          qw(
          context_on_site_natural
          context_on_site_content
          context_on_site_direct
          context_on_site_rtb
          context_on_site_adblock
          context_on_site_stripe
          mobile_app_rtb
          search_on_site_direct
          search_on_site_market
          search_on_site_mcb
          search_on_site_premium
          video_an_site_instream
          video_an_site_inpage
          video_an_site_fullscreen
          indoor_block
          outdoor_block
          )
    };

    return $products;
}

=encoding UTF-8

=cut

=head2 get_data_for_client_id

    my $data = $app->agreement_checker->get_data_for_client_id(
        client_id => $client_id,
    );

Возвращает структуру вида:

    {
        today    => {
            context_on_site_direct => 1,
            context_on_site_rtb    => 1,
            context_on_site_stripe => 1,
        },
        tomorrow => {
            context_on_site_direct => 1,
        },
    }

В ответе содержится информация о том какие продукты могут быть запущены
на этом договоре сегодня, а какие продукты могут быть запущены завтра.

=cut

sub get_data_for_client_id {
    my ($self, %opts) = @_;

    my $client_id = delete($opts{'client_id'});
    throw Exception::Validation::BadArguments gettext("Expected 'client_id'") unless defined($client_id);

    throw Exception::Validation::BadArguments gettext("Got unknown parameters: %s", join(", ", keys(%opts))) if %opts;

    my $data = {};

    if ($self->_is_special_client_id_with_all_products($client_id)) {
        $data->{today}    = $self->get_all_products();
        $data->{tomorrow} = $self->get_all_products();
    } elsif (my $available_products = $self->_is_vip_partner($client_id)) {
        $data = {
            today    => $available_products,
            tomorrow => $available_products,
        };
    } elsif ($self->_is_tutby_partner($client_id)) {

        if ($self->_has_tutby_agreement($client_id)) {
            $data = {
                today    => $self->_get_tutby_products(),
                tomorrow => $self->_get_tutby_products(),
            };
        }

    } else {

        my $balance_data = $self->api_balance->get_partner_contracts("ClientID" => $client_id);

        foreach my $element (@$balance_data) {

            next if $element->{'Contract'}{'type'} ne 'PARTNERS';

            my $agreement_ref = Application::Model::AgreementChecker::_Agreement->new(
                contract    => $element->{'Contract'},
                collaterals => $element->{'Collaterals'},
            );

            foreach my $day (qw(today tomorrow)) {
                foreach my $product (keys(%{$self->get_all_products()})) {
                    if ($agreement_ref->can_run(product => $product, day => $day)) {
                        $data->{$day}->{$product} = TRUE;
                    }
                }
            }
        }
    }

    return $data;
}

=head1 has_agreement_for_any_product_for_today

Возвращает true если у указанного client_id есть возможность запускать
сегодня хотя бы один из перечисленных продуктов.

    my $bool = $app->agreement_checker->has_agreement_for_any_product_for_today(
        client_id => $client_id,
        products => ['context_on_site_direct'],
    );

=cut

sub has_agreement_for_any_product_for_today {
    my ($self, %opts) = @_;

    return $self->_has_agreement_for_any_product_for_day(%opts, day => 'today',);
}

=head1 has_agreement_for_any_product_for_tomorrow

То же самое что has_agreement_for_any_product_for_today, но для для tomorrow.

=cut

sub has_agreement_for_any_product_for_tomorrow {
    my ($self, %opts) = @_;

    return $self->_has_agreement_for_any_product_for_day(%opts, day => 'tomorrow',);
}

=begin comment _get_tutby_products

=end comment

=cut

sub _get_tutby_products {
    my ($self, @opts) = @_;

    throw Exception::Validation::BadArguments gettext("Got unknown parameters: %s", join(", ", @opts)) if @opts;

    my $products = {
        map {$_ => TRUE}
          qw(
          context_on_site_content
          context_on_site_direct
          context_on_site_rtb
          mobile_app_rtb
          search_on_site_direct
          search_on_site_premium
          video_an_site_inpage
          video_an_site_instream
          )
    };

    return $products;
}

=begin comment _get_user_by_client_id

Возвращает HASHREF с информацией про пользователя или {} если в системе нет пользователя с таким Client ID.

=end comment

=cut

sub _get_user_by_client_id {
    my ($self, $client_id) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(users_view_all users_view_field__client_id users_view_field__is_tutby users_view_field__has_tutby_agreement)
    );

    my $user = $self->users->get_all(
        fields => [qw(client_id is_tutby has_tutby_agreement is_assessor)],
        filter => {client_id => $client_id},
    );

    if (@$user == 1) {
        return $user->[0];
    } else {
        return {};
    }

}

sub _has_agreement_for_any_product_for_day {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                client_id => {type => 'int_un'},
                products  => {
                    type     => 'array',
                    size_min => 1,
                    check    => sub {
                        my ($qv, $values) = @_;

                        my %known_products = map {$_ => TRUE} keys(%{$self->get_all_products()});
                        foreach my $product (@$values) {
                            throw Exception::Validator::Fields gettext("Product '%s' not found", $product)
                              unless $known_products{$product};
                        }

                        return '';
                      }
                },
                day                => {in   => [qw(today tomorrow)]},
                not_check_assessor => {type => 'boolean', optional => TRUE},
            }
        },
        throw => TRUE,
    );

    my $client_id = delete($opts{'client_id'});
    my $products  = delete($opts{'products'});
    my $day       = delete($opts{'day'});

    if (!$opts{not_check_assessor} and $self->_is_assessor($client_id)) {
        return TRUE;
    } else {
        my $data = $self->get_data_for_client_id(client_id => $client_id);

        foreach my $product (@$products) {
            return TRUE if exists($data->{$day}{$product});
        }
    }

    return FALSE;
}

=begin comment _has_tutby_agreement

=end comment

=cut

sub _has_tutby_agreement {
    my ($self, $client_id) = @_;

    return !!$self->_get_user_by_client_id($client_id)->{'has_tutby_agreement'};
}

sub _is_special_client_id_with_all_products {
    my ($self, $client_id) = @_;

    # https://st.yandex-team.ru/PI-11665
    return TRUE if $client_id == $ADINSIDE_CLIENT_ID;

    # https://st.yandex-team.ru/PI-4675#1439472369000
    return TRUE if $client_id == 5769686;    # login: music-kostya
    return TRUE if $client_id == 7481996;    # login: imho-vivaki-project
    return TRUE if $client_id == 5946594;    # login: vast-rtb-test

    # https://st.yandex-team.ru/PI-5590
    return TRUE if $client_id == 5719912;    # login: yndx-stas
    return TRUE if $client_id == 7382405;    # login: yndx-stas-test

    return FALSE;
}

=begin comment _is_tutby_partner

=end comment

=cut

sub _is_tutby_partner {
    my ($self, $client_id) = @_;

    return !!$self->_get_user_by_client_id($client_id)->{'is_tutby'};
}

=begin comment _is_vip_partner

В том случае если указанный $client_id - это vip партнер, то возвращает
hashref со списком продуктов, кторые доступны этому партнеру.

Если же этот $client_id - это не vip партнер, то возвращает undef.

=end comment

=cut

sub _is_vip_partner {
    my ($self, $client_id) = @_;

    my %vip_client_id_products =
      map {$_->{'client_id'} => $_->{'available_block_models'}} @{$self->vip_partners->get_info()};

    if (exists($vip_client_id_products{$client_id})) {
        my $available_products = {map {$_ => TRUE} @{$vip_client_id_products{$client_id}}};
        return $available_products;
    } else {
        return undef;
    }
}

sub _is_assessor {
    my ($self, $client_id) = @_;

    return !!$self->_get_user_by_client_id($client_id)->{'is_assessor'};
}

TRUE;
__END__
