package Application::Model::AgreementChecker::_Agreement;

=encoding UTF-8

=cut

=head1 DESCRIPTION

Объект, который представляет договор в балансе.

'end_dt' - это дата последнего дня, когда договор действует.

Поля 'is_signed' и 'is_faxed' работает просто как "галочки" - если есть
значнеие, то это значит что договор подписан или получен по факсу (но договор
действует начиная с даты dt).

=cut

use qbit;
use Utils::TSV;

use Exception::Validation::BadArguments;

=head2 can_run

    my $bool = $ref->can_run(
        day => 'today',
        product => 'search_on_site_mcb',
    );

=cut

sub can_run {
    my ($self, %opts) = @_;

    my $day = delete($opts{'day'});
    throw Exception::Validation::BadArguments gettext("Expected 'day'") unless defined($day);

    my $product = delete($opts{'product'});
    throw Exception::Validation::BadArguments gettext("Expected 'product'") unless defined($product);

    my $date = $self->_get_date($day);

    return FALSE if !$self->_is_live($day);

    my $contract_type = $self->get_contract_type_id();

    my @rules = $self->_get_rules_structure();

    my $product_can_run_on_this_type_id_if_signed_or_faxed =
      in_array($contract_type, $self->_get_allowed_type_ids_for_product($product, 'signed_or_faxed'));
    my $product_can_run_on_this_type_id_if_signed =
      in_array($contract_type, $self->_get_allowed_type_ids_for_product($product, 'signed'));

    if (($self->is_signed_or_faxed || $self->is_oferta2_test_mode)
        && $product_can_run_on_this_type_id_if_signed_or_faxed)
    {
        return TRUE;
    } elsif ($self->is_signed && $product_can_run_on_this_type_id_if_signed) {
        return TRUE;
    } else {
        return FALSE;
    }
}

=head2 get_contract_type_id

    my $id = $ref->get_contract_type_id(); # 5

Возвращает id типа договора.

=cut

sub get_contract_type_id {
    my ($self) = @_;

    my $contract_type = $self->{'_contract'}{'contract_type'};

    throw Exception sprintf(
        'Unknown contract_type "%s" (id: "%s", external_id: "%s")',
        $contract_type,
        $self->{'_contract'}->{'id'}          // '',
        $self->{'_contract'}->{'external_id'} // '',
    ) unless in_array($contract_type, [$self->_get_known_contract_types()]);

    return $contract_type;
}

=head2 get_external_id

    my $name = $ref->get_external_id(); # "РС-4085-05/09"

Возвращает имя договора.

=cut

sub get_external_id {
    my ($self) = @_;

    return $self->{'_contract'}{'external_id'};
}

=head2 is_live_today

    my $bool = $ref->is_live_today();

Возвращает true, если договор сегодня действует, иначе возвращет false.

=cut

sub is_live_today {
    my ($self) = @_;

    return $self->_is_live('today');
}

=head2 is_live_tomorrow

    my $bool = $ref->is_live_tomorrow();

Возвращает true, если договор действует завтра, иначе возвращет false.

=cut

sub is_live_tomorrow {
    my ($self) = @_;

    return $self->_is_live('tomorrow');
}

=head2 is_signed

Возвращает true значение если договор имеет флаг is_signed
(Но то что этот метод возращает true еще не означает что договор действует).

=cut

sub is_signed {
    my ($self) = @_;

    return TRUE if $self->{'_contract'}{'is_signed'};
}

sub is_oferta2_test_mode {
    my ($self) = @_;

    return TRUE if $self->{'_contract'}{contract_type} == 9 && $self->{'_contract'}{'test_mode'};
}

sub allows_to_fill_part2 {
    my ($self) = @_;

    return TRUE if $self->is_oferta2_test_mode && !$self->{'_contract'}{'person_id'};
}

=head2 is_signed_or_faxed

Возвращает true значение если договор имеет флаг is_signed или is_faxed.
(Но то что этот метод возращает true еще не означает что договор действует).

=cut

sub is_signed_or_faxed {
    my ($self) = @_;

    return TRUE if $self->{'_contract'}{'is_signed'} || $self->{'_contract'}{'is_faxed'};
}

=head2 new

    my $ref = Application::Model::AgreementChecker::_Agreement->new(
        contract => {
            dt => '2000-01-01',
            is_signed => '2000-01-01',
            is_cancelled => "2014-01-17"
        },
        collaterals => [],
    );

=cut

sub new {
    my ($class, %params) = @_;

    my $self = {};
    bless $self, $class;

    $self->{'_contract'}    = delete($params{'contract'});
    $self->{'_collaterals'} = delete($params{'collaterals'});

    throw Exception::Validation::BadArguments 'No contract'    unless $self->{'_contract'};
    throw Exception::Validation::BadArguments 'No collaterals' unless $self->{'_collaterals'};
    throw Exception::Validation::BadArguments 'Incorrect collaterals' if ref($self->{'_collaterals'}) ne 'ARRAY';

    return $self;
}

=begin comment _get_allowed_type_ids_for_product

На входе - имя продукта и строка 'signed_or_faxed'/'signed'.

На выходе - arrayref со списком type_id договоров на котором возможно
запускать продукт. Пример:

    my $data = $ref->_get_allowed_type_ids_for_product(
        'context_on_site_stripe',
        'signed_or_faxed',
    );

Вернет:

    [
        5,
        6,
        7,
    ]

=end comment

=cut

sub _get_allowed_type_ids_for_product {
    my ($self, $product, $type) = @_;

    throw Exception::Validation::BadArguments gettext("Expected 'product'") unless defined($product);
    throw Exception::Validation::BadArguments gettext("Expected 'type'")    unless defined($type);

    my %type2symbol = (
        signed_or_faxed => 'x',
        signed          => 's',
    );

    throw Exception::Validation::BadArguments gettext('Incorrect type "%s"', $type) unless $type2symbol{$type};

    my @data = grep {$_->{product} eq $product} $self->_get_rules_structure();

    throw Exception::Validation::BadArguments gettext("No info about product '%s'", $product) unless scalar(@data) == 1;

    my @type_ids =
      sort {$a <=> $b} grep {defined($data[0]->{$_}) && $data[0]->{$_} eq $type2symbol{$type}} keys(%{$data[0]});

    return \@type_ids;
}

sub _get_date {
    my ($self, $day) = @_;

    if (in_array($day, [qw(today tomorrow)])) {
        return name2date($day, oformat => 'db');
    } else {
        throw Exception::Validation::BadArguments 'Incorrect usage';
    }

}

sub _get_known_contract_types {
    my ($self) = @_;

    my @rules = $self->_get_rules_structure();

    my @known_contract_types = sort grep {$_ ne 'product'} keys(%{$rules[0]});

    return @known_contract_types;
}

sub _get_rules_structure {
    my ($self) = @_;

    my $rules = $self->_get_rules_table();
    $rules =~ s/ *\| */\t/g;

    return @{parse_tsv($rules)};
}

sub _get_rules_table {
    my ($self) = @_;

    # x - договор получен по факсу или получен оригинал договора
    # s - получен оригинал договора (вариант получения по факсу не подходит)

    # Соответствие айдишников договоров - названиям:
    # 1 - С Юр.лицом
    # 2 - С Аргератором
    # 3 - С Физ.лицом
    # 4 - С нерезидентом
    # 5 - универсальный
    # 6 - РСЯ 2014
    # 7 - Новая оферта РСЯ
    # 8 - SSP
    # 9 - Оферта РСЯ для России (создана в апреле 2018)
    # 10 - Лицензионный договор (создан в ноябре 2018 — BALANCE-25434 )

    my $rules =
        "product                  | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 |10 |\n"
      . "context_on_site_direct   | x |   | x | x | x | x | x | x | x |   |\n"
      . "context_on_site_rtb      |   |   |   | x | x | x | x | x | x |   |\n"
      . "context_on_site_stripe   |   |   |   | x | x | x | x | x | x |   |\n"
      . "mobile_app_rtb           |   |   |   | x | x | x | x | x | x |   |\n"
      . "search_on_site_direct    | x | x | x | x | x | x | x | x | x |   |\n"
      . "search_on_site_market    | x | x | x | x | x | x | x | x | x |   |\n"
      . "search_on_site_mcb       |   |   |   | x | x | x | x | x | x |   |\n"
      . "search_on_site_premium   | x | x | x | x | x | x | x | x | x |   |\n"
      . "video_an_site_instream   |   |   |   |   | x | x | x | x | x | x |\n"
      . "video_an_site_inpage     |   |   |   |   | x | x | x | x | x | x |\n"
      . "video_an_site_fullscreen |   |   |   |   | x | x | x | x | x | x |\n"
      . "context_on_site_adblock  | x | x | x | x | x | x | x | x | x |   |\n"
      . "indoor_block             | x | x | x | x | x | x | x | x | x |   |\n"
      . "outdoor_block            | x | x | x | x | x | x | x | x | x |   |\n"
      . "context_on_site_natural  | x | x | x | x | x | x | x | x | x |   |\n"
      . "context_on_site_content  | x | x | x | x | x | x | x | x | x |   |\n";

    return $rules;
}

sub _is_live {
    my ($self, $day) = @_;

    my $date = $self->_get_date($day);

    # Еще не начал действовать
    return FALSE if $date lt $self->{'_contract'}{'dt'};

    # Уже закрыли
    if ($self->{'_contract'}{'end_dt'}) {
        return FALSE if $date gt $self->{'_contract'}{'end_dt'};
    }

    # Договор аннулирован
    if ($self->{'_contract'}{'is_cancelled'}) {
        return FALSE if $date ge $self->{'_contract'}{'is_cancelled'};
    }

    # Договор не действует, если есть допсоглашение "расторжение договора"
    # (collateral_type_id = 2050)
    # Это допсоглашение действует только в том случае, если пользователь
    # подписал это допсоглашение, т.е. если заполнено поле is_faxed или
    # is_signed)
    return FALSE
      if grep {
             $_->{'collateral_type_id'} == 2050
          && !$_->{'is_cancelled'}
          && defined($_->{'end_dt'})
          && $date gt $_->{'end_dt'}
          && ($_->{'is_faxed'} || $_->{'is_signed'})
      } @{$self->{'_collaterals'}};

    # Договор не действет, если есть допсоглашение "уведомление о закрытии"
    # (collateral_type_id = 2090)
    # Это уведомление не нужно подписывать, оно начининает действовать с даты 'dt'
    return FALSE
      if grep {$_->{'collateral_type_id'} == 2090 && !$_->{'is_cancelled'} && $_->{'end_dt'} && $date gt $_->{'end_dt'}}
          @{$self->{'_collaterals'}};

    # Договор должен быть подписан или отправлен по факсу.
    # Либо договор должен быть офертой в тестовом режиме - так работают партнеры,
    # которые зарегистрировались в ПИ, но еще не оставили свои реквизиты.
    return FALSE
      unless $self->{'_contract'}{'is_signed'}
          || $self->{'_contract'}{'is_faxed'}
          || $self->is_oferta2_test_mode;

    return TRUE;
}

TRUE;
