package Application::Model::AllPages;

=encoding UTF-8

=cut

use qbit;

use base qw(Application::Model::Common);

use PiConstants qw($ADINSIDE_CLIENT_ID $MYSQL_CHUNK);

use Exception;
use Exception::Validation::BadArguments;

sub accessor      {'all_pages'}
sub db_table_name {'all_pages'}

__PACKAGE__->model_accessors(
    partner_db => 'Application::Model::PartnerDB::AllPages',
    queue      => 'Application::Model::Queue',
    users      => 'Application::Model::Users',
    site       => 'Application::Model::Product::AN::Site',
);

my $SSP_MODELS = {};

__PACKAGE__->model_fields(
    id                    => {default => TRUE, db => TRUE, pk => TRUE},
    model                 => {default => TRUE, db => TRUE, pk => TRUE},
    page_id               => {default => TRUE, db => TRUE},
    caption               => {default => TRUE, db => TRUE},
    domain                => {default => TRUE, db => TRUE},
    domain_id             => {db      => TRUE},
    product_type          => {db      => TRUE},
    owner_id              => {db      => TRUE},
    login                 => {db      => TRUE},
    client_id             => {db      => TRUE},
    business_unit         => {db      => TRUE},
    is_balance_registered => {db      => TRUE},
    is_yandex_page        => {
        forced_depends_on => [qw(client_id)],
        get               => sub {
            return $_[1]->{'client_id'} == $ADINSIDE_CLIENT_ID ? 1 : 0;
          }
    },
    is_ssp => {
        forced_depends_on => [qw(model)],
        get               => sub {
            return $SSP_MODELS->{$_[1]->{'model'}} //=
              $_[0]->model->app->{$_[1]->{'model'}}->isa('Application::Model::Page::SSP') ? 1 : 0;
          }
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id                    => {type => 'number', label => d_gettext('Page ID')},
        model                 => {type => 'text',   label => d_gettext('Model')},
        page_id               => {type => 'number', label => d_gettext('Page ID')},
        caption               => {type => 'text',   label => d_gettext('Caption')},
        domain                => {type => 'text',   label => d_gettext('Domain')},
        domain_id             => {type => 'number', label => d_gettext('Domain ID')},
        owner_id              => {type => 'number', label => d_gettext('Owner ID')},
        login                 => {type => 'text',   label => d_gettext('Login')},
        client_id             => {type => 'number', label => d_gettext('Client ID')},
        product_type          => {type => 'text',   label => d_gettext('Product type')},
        is_balance_registered => {type => 'text',   label => d_gettext('Added to Balance')},
        users                 => {
            type           => 'subfilter',
            model_accessor => 'users',
            field          => 'owner_id',
            fk_field       => 'id',
            label          => d_gettext('Users'),
        },
        domains => {
            type           => 'subfilter',
            model_filter   => ['model', 'IN', \['context_on_site_campaign', 'search_on_site_campaign']],
            model_accessor => 'site',
            field          => 'domain_id',
            fk_field       => 'id',
            label          => d_gettext('Domains'),
        },
        user_type => {type => 'alias', path => [qw(users user_type)]},
    }
);

sub query {
    my ($self, %opts) = @_;

    my $filter = $self->partner_db->filter($opts{'filter'});

    return $self->partner_db->query->select(
        table => ($opts{'options'}->{'from_view'} ? $self->partner_db->all_pages_view() : $self->partner_db_table()),
        fields => $opts{'fields'}->get_db_fields(),
        filter => $filter,
    );
}

sub is_only_rtb_product_page {
    my ($self, $page_id) = @_;

    return $self->partner_db->only_rtb_product_page->get($page_id, fields => [qw(page_id)]);
}

=head2 mark_pages_for_async_update
Варианты вызова: ничего не передавать, только пейджи, только айдишники, только модели, айдишники и пейджи.
=cut

sub mark_pages_for_async_update {
    my ($self, %opts) = @_;

    my ($pages, $page_ids, $models, $min_send_date, $chunk_size, $use_separate_queue) =
      @opts{qw( pages  page_ids  models  min_send_date  chunk_size use_separate_queue)};

    $pages      //= [];
    $page_ids   //= [];
    $models     //= [];
    $chunk_size //= $MYSQL_CHUNK;

    my %pages_by_model = ();
    if (exists $opts{pages} || exists $opts{page_ids}) {
        if (@$page_ids) {
            my $objects = $self->get_all(
                fields => ['id', 'model', 'page_id'],
                filter    => {page_id => $page_ids},
                from_view => TRUE,
            );

            if (@$objects != @$page_ids) {
                my %pages_ids_hash = map {$_->{'page_id'} => 1} @$objects;

                throw Exception gettext('Pages with Page ID %s not found',
                    join(', ', grep {!$pages_ids_hash{$_}} @$page_ids));
            }

            push(@$pages, @$objects);
        }

        if (@$pages) {
            foreach my $object (@$pages) {
                push(@{$pages_by_model{$object->{'model'}}}, $object);
            }
        }

        $models = [sort keys %pages_by_model];
    } elsif (!exists $opts{models}) {
        $models = $self->app->product_manager->get_page_model_accessors();
    }

    if ($use_separate_queue) {
        return $self->add_pages_to_separate_queue(pages_by_model => \%pages_by_model, models => $models);
    }

    my $tmp_rights = $self->app->add_all_tmp_rights();

    foreach my $model_name (sort @$models) {
        my $is_mark_all_model_pages = defined $pages_by_model{$model_name} ? 0 : 1;

        my $model           = $self->app->$model_name;
        my $need_update_bit = $model->get_multistate_by_name('need_update');

        my $base_filters = [['multistate', 'IN', \$model->get_multistate_by_action('set_need_update')]];
        push @$base_filters, ['send_time', '<=', \$min_send_date] if $min_send_date;

        my $model_pages = $pages_by_model{$model_name} // [];
        while (@$model_pages || $is_mark_all_model_pages) {

            my $update_data = [map {$_->{'id'}} splice(@$model_pages, 0, $chunk_size)];
            my $curr_filters = @$update_data ? [@$base_filters, ['id', 'IN', \$update_data]] : $base_filters;

            my $filter = $self->partner_db->filter(['AND', $curr_filters]);
            $model->partner_db_table->edit(
                $filter,
                {
                    multistate => ['|', ['multistate', \$need_update_bit]],
                    update_time => curdate(oformat => 'db_time'),
                }
            );

            last if $is_mark_all_model_pages;
        }
    }
}

sub add_pages_to_separate_queue {
    my ($self, %opts) = @_;

    my $pages_by_model = $opts{'pages_by_model'};
    my $models = $opts{'models'} // [keys(%$pages_by_model)];

    my $curdate = curdate(oformat => 'db_time');

    foreach my $model (@$models) {
        my $pages = $pages_by_model->{$model} // [];

        my $is_mark_all_model_pages = !@$pages;

        my $page_accessor = $self->app->$model;

        my $base_filters = ['multistate', 'IN', \$page_accessor->get_multistate_by_action('set_need_update')];

        while ($is_mark_all_model_pages || @$pages) {
            my @chunk    = splice(@$pages, 0, $MYSQL_CHUNK);
            my @ids      = ();
            my @page_ids = ();
            foreach my $page (@chunk) {
                push(@ids,      $page->{'id'});
                push(@page_ids, $page->{'page_id'});
            }

            my $page_id_field_name = $page_accessor->get_page_id_field_name();

            my $filter = [
                'AND',
                [
                    $base_filters,
                    (@chunk ? (['id', 'IN', \\@ids]) : ()),
                    [
                        $page_id_field_name,
                        'NOT IN',
                        $self->partner_db->query->select(
                            table  => $self->partner_db->need_update_pages,
                            fields => [qw(page_id)],
                            filter => [
                                'AND',
                                [
                                    ['worker', 'IS', \undef],
                                    (
                                        @chunk
                                        ? ['page_id', 'IN', \\@page_ids]
                                        : ['model', '=', \$page_accessor->accessor()]
                                    ),
                                ]
                            ],
                        )
                    ]
                ]
            ];

            my $page_table = $page_accessor->partner_db_table();

            my $query = $self->partner_db->query->select(
                table  => $page_table,
                fields => {
                    page_id     => $page_id_field_name,
                    model       => \$model,
                    update_time => \$curdate,
                },
                filter => $filter
            );
            $query->fields_order(qw(page_id model update_time));

            $self->partner_db->need_update_pages->add_multi([$query], fields => [qw(page_id model update_time)]);

            last if $is_mark_all_model_pages;
        }
    }
}

TRUE;
