package Application::Model::Assistants;

use qbit;

use base qw(Application::Model::DBManager::Base);

use Exception::Denied;
use Exception::Validator;
use Exception::Validator::Fields;
use List::MoreUtils qw(part);

use PiConstants qw($SYSTEM_CRON_USER_ID $TABLE_PAGE_MODELS $YAN_PARTNER_ASSISTANT_ROLE_ID);

sub accessor      {'assistants'}
sub db_table_name {'assistants'}

__PACKAGE__->model_accessors(
    partner_db    => 'Application::Model::PartnerDB',
    rbac          => 'Application::Model::RBAC',
    resources     => 'Application::Model::Resources',
    user_features => 'Application::Model::Users::Features',
    users         => 'Application::Model::Users',
    inviter       => 'Application::Model::Inviter',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'assistants',
            description => d_gettext('Rights to use assistants'),
            rights      => {assistants_save => d_gettext('Right to save assistants'),}
        }
    ]
);

__PACKAGE__->model_fields(
    id        => {default => TRUE, db => TRUE, pk          => TRUE, adjust_type => 'str',},
    page_id   => {default => TRUE, db => TRUE, adjust_type => 'str',},
    user_id   => {default => TRUE, db => TRUE, adjust_type => 'str',},
    can_edit  => {default => TRUE, db => TRUE, adjust_type => 'str',},
    dissabled => {default => TRUE, db => TRUE, adjust_type => 'str',},
    login     => {
        depends_on => ['user_id'],
        get        => sub {
            $_[0]->{'__USERS__'}->{$_[1]->{'user_id'}} // '';
        },
    },
    inviter_id => {default => TRUE, db => TRUE, adjust_type => 'str',},
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id       => {type => 'number',  label => d_gettext('ID')},
        page_id  => {type => 'number',  label => d_gettext('Page ID')},
        user_id  => {type => 'number',  label => d_gettext('User ID')},
        can_edit => {type => 'boolean', label => d_gettext('Can edit')},
        user     => {
            type           => 'subfilter',
            model_accessor => 'users',
            field          => 'user_id',
            label          => d_gettext('User'),
        },
    },
);

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('login')) {
        $fields->{'__USERS__'} = {
            map {$_->{'id'} => $_->{'login'}} @{
                $self->users->get_all(
                    fields => [qw(id login)],
                    filter => {id => array_uniq(map {$_->{'user_id'}} @$result)},
                )
              }
        };
    }
}

sub get_assistants {
    my ($self, %opts) = @_;

    my $tmp    = $self->app->add_tmp_rights('users_view_all_an_partner_assistants');
    my $filter = $self->partner_db->filter($opts{'filter'});

    $filter->and(
        [
            'OR',
            [
                [user_id => '='     => \0],
                [user_id => '= ANY' => $self->users->query(fields => $self->users->_get_fields_obj(['id']))]
            ]
        ]
    );

    my $query = $self->query(
        fields => $self->_get_fields_obj([qw(page_id user_id can_edit inviter_id)]),
        filter => $filter
    )->join(table => $self->partner_db->users, fields => ['login'])->get_all();

    my ($defaults, $invites) = part {$_->{user_id} == 0} @$query;

    if (ref($invites) eq 'ARRAY' && scalar @$invites) {
        my $tmp_view_inviter = $self->app->add_tmp_rights('inviter_view');

        my $all_invites = {
            map {$_->{id} => $_} @{
                $self->inviter->get_all(
                    filter => [id => 'IN' => [map {$_->{inviter_id}} @$invites]],
                    fields => [qw(id email is_expired)]
                )
              }
        };
        $invites = [
            map {
                $_->{email}   = $all_invites->{$_->{inviter_id}}->{email};
                $_->{expired} = $all_invites->{$_->{inviter_id}}->{is_expired};
                $_->{login}   = undef;
                $_;
              } @$invites
        ];
    }

    return [map {ref($_) eq 'ARRAY' ? @$_ : ()} ($defaults, $invites)];
}

sub replace {
    my ($self, $page_id, $data, $mode) = @_;

    throw Exception::Denied gettext("You can't save assistants") unless $self->check_short_rights('save');

    my %available_assistants = map {$_->{user_id} => TRUE} @{$self->get_assistants(filter => {page_id => $page_id})};

    my @need_invalidate_resources_cache = ();
    push(@need_invalidate_resources_cache,
        @{arrays_difference([map {$_->{'user_id'}} @$data], [sort {$a <=> $b} keys(%available_assistants)])});
    push(@need_invalidate_resources_cache,
        @{arrays_difference([sort {$a <=> $b} keys(%available_assistants)], [map {$_->{'user_id'}} @$data])});

    my $cur_user_id = $self->get_option('cur_user', {})->{id};
    my $invites = [map {$_->{inviter_id}} grep {0 eq $_->{user_id}} @$data];
    if (@$invites) {
        my %invite_with_partner_id = map {$_->{id} => $_} @{
            $self->partner_db->invites->get_all(
                fields => [qw(id partner_id)],
                filter => [
                    'AND',
                    [
                        [id         => '='      => \$invites],
                        [manager_id => '='      => \$cur_user_id],
                        [partner_id => 'IS NOT' => \undef],
                    ]
                ]
            )
          };
        if (keys %invite_with_partner_id) {
            foreach (@$data) {
                if (exists $_->{inviter_id} && $invite_with_partner_id{$_->{inviter_id}}) {
                    $_->{user_id} = $invite_with_partner_id{$_->{inviter_id}}{partner_id};
                }
            }
        }
    }

    $self->app->validator->check(
        data     => $data,
        template => {
            type => 'array',
            all  => {
                type   => 'hash',
                fields => {
                    inviter_id => {type => 'scalar', optional => 1},
                    user_id    => {type => 'int_un'},
                    can_edit   => {type => 'boolean'},
                }
            },
            check => sub {
                my ($qv, $users) = @_;

                my %count_users;

                $count_users{$_->{'user_id'}}++ for grep {0 ne $_->{user_id}} @$users;
                my @dup_users = sort {$a <=> $b} grep {$count_users{$_} > 1} keys %count_users;

                my %count_inviter;
                $count_inviter{$_->{'inviter_id'}}++ for grep {0 eq $_->{user_id}} @$users;
                push @dup_users, sort {$a cmp $b} grep {$count_inviter{$_} > 1} keys %count_inviter;

                throw Exception::Validator::Fields gettext('The following users are duplicated: %s',
                    join(',', @dup_users))
                  if @dup_users;

                if (keys %count_inviter) {
                    my %invites = map {$_->{'id'} => TRUE} @{
                        $qv->app->partner_db->invites->get_all(
                            fields => ['id'],
                            filter =>
                              ['AND', [[id => '=' => \[keys %count_inviter]], [manager_id => '=' => \$cur_user_id],]]
                        )
                      };
                    my @invalid_invites = grep {!exists $invites{$_}} keys %count_inviter;
                    if (@invalid_invites) {
                        throw Exception::Validator::Fields gettext('The following users do not exist: %s',
                            join(',', sort @invalid_invites));
                    }
                }

                my @user_ids = keys %count_users;
                return '' unless @user_ids;

                my %assistants =
                  map {$_->{'id'} => TRUE}
                  @{$qv->app->users->get_all_assistants(fields => ['id'], filter => {id => \@user_ids})};

                my @not_assistants = grep {not $assistants{$_}} @user_ids;

                return '' if @not_assistants == 0;

                my %exist =
                  map {$_->{'id'} => $_}
                  @{$qv->app->users->get_all(fields => [qw(id login)], filter => {id => \@not_assistants})};

                my @not_exist = grep {not $exist{$_}} @not_assistants;

                throw Exception::Validator::Fields gettext('The following users do not exist: %s',
                    join(',', @not_exist))
                  if @not_exist;

                throw Exception::Validator::Fields gettext(
                    'The following users have not role "assistant": %s',
                    join(',', map {$exist{$_}->{'login'}} @not_assistants)
                );
            },
        },
        app   => $self,
        throw => TRUE
    );

    my $all_assistants = $self->get_all(
        fields => [qw(user_id can_edit dissabled)],
        filter => {page_id => $page_id}
    );

    push(@$data, grep {!$available_assistants{$_->{'user_id'}}} @$all_assistants);

    $self->partner_db->transaction(
        sub {
            $self->partner_db_table()->delete($self->partner_db->filter({page_id => $page_id})) if $mode eq 'edit';

            $self->partner_db_table()->add_multi(
                [
                    map {
                        {%$_, page_id => $page_id}
                      } @$data
                ]
            );
        }
    );

    $self->resources->invalidate_cache($_) foreach @need_invalidate_resources_cache;
}

sub get_rights_for_assistant {
    my ($self, $user_id, $cur_user_roles, $only_feature_rights) = @_;

    my $is_assistant = 0;
    if ($user_id != $SYSTEM_CRON_USER_ID) {
        $is_assistant = exists $cur_user_roles->{$YAN_PARTNER_ASSISTANT_ROLE_ID};
    }

    my %other_rights = ();
    if ($is_assistant) {

        my @user_ids = map {$_->{'owner_id'}} @{
            $self->partner_db->all_pages->get_all(
                fields   => ['owner_id'],
                distinct => TRUE,
                filter   => [
                    'page_id' => 'IN' => $self->partner_db->query->select(
                        table  => $self->partner_db->assistants,
                        fields => ['page_id'],
                        filter => {user_id => $user_id}
                      )->distinct()
                ],
            )
          };

        if (@user_ids) {
            %other_rights = (
                $only_feature_rights ? () : %{$self->rbac->get_rights_by_user_id(\@user_ids)},
                map {
                    map {$_ => 1}
                      $self->user_features->get_user_rights($_)
                  } @user_ids,
            );
        }
    }

    return \%other_rights;
}

TRUE;
