package Application::Model::AutoStart;

=encoding UTF-8

=cut

use qbit;

use base qw(QBit::Application::Model);

use Utils::Logger qw(INFO INFOF ERROR);
use PiConstants qw($START_STOP_MAIL);

use Utils::MonitoringUtils qw(send_to_graphite);

use Exception;

sub accessor {'auto_start'}

__PACKAGE__->model_accessors(
    agreement_checker => 'Application::Model::AgreementChecker',
    exception_dumper  => 'Application::Model::ExceptionDumper',
    mail_notification => 'Application::Model::MailNotification',
    users             => 'Application::Model::Users',
);

my $RETRY_COUNT_MAX = 100;

=head2 do_auto_start

=cut

sub do_auto_start {
    my ($self, %opts) = @_;

    my $model                          = delete($opts{'model'});
    my $products_for_agreement_checker = delete($opts{'products_for_agreement_checker'});
    my $login                          = delete($opts{'login'});
    my $from                           = delete($opts{'from'}) // 'testing';
    my $starting_email_subject         = delete($opts{'starting_email_subject'});
    my $not_starting_email_subject     = delete($opts{'not_starting_email_subject'});
    my $additional_field               = delete($opts{'additional_field'}) // 'domain';
    my $page_ids_to_ignore             = delete($opts{'page_ids_to_ignore'}) // [];
    my $page_ids                       = [split(',', delete($opts{'page_ids'}) // '')];
    my $heartbeat                      = delete($opts{'heartbeat'});

    throw Exception gettext("Got unknown parameters: %s", join(", ", keys(%opts))) if %opts;

    INFO "model " . ref($model);
    INFO "page_ids_to_ignore " . join(', ', @$page_ids_to_ignore);

    # see https://st.yandex-team.ru/PI-16824
    unless (defined($model->get_multistate_by_name('testing'))) {
        INFOF 'no [testing] status in model [%s], skipping', ref($model);
        return;
    }

    my @filter;

    my %from2multistate_query = (
        testing            => 'testing',
        stopped_or_testing => 'stopped or testing',
    );

    if ($from2multistate_query{$from}) {
        push(@filter, {multistate => $from2multistate_query{$from}});
    } else {
        throw Exception "Uknown value for the 'from' parameter";
    }

    if (scalar @$page_ids) {
        push(@filter, {$model->get_page_id_field_name() => $page_ids});
    }

    if ($from eq 'stopped_or_testing') {

        unless (scalar @$page_ids) {
            throw Exception "Option [--from=stopped_or_testing] can only be used with option [--page_ids=]";
        }

        push(
            @filter,
            [
                OR => [
                    {multistate => $model->get_multistate_by_action('start')},
                    {multistate => $model->get_multistate_by_action('start_testing')},
                ]
            ]
        );

    } else {
        push(@filter, {multistate => $model->get_multistate_by_action('start')});
    }

    if ($login) {
        my $user = $self->users->get_by_login($login, fields => [qw(id)],);

        throw Exception "Can't find login $login in database" unless $user;
        push(@filter, {owner_id => $user->{id}});
    }

    my $campaigns = $model->get_all(
        fields => [qw(id multistate_name client_id login caption page_id owner), $additional_field],
        filter => ['AND' => \@filter],
    );

    INFOF(
        "number of Page IDs found: %s (%s)",
        scalar(@$campaigns), join(', ', sort {$a <=> $b} map {$_->{page_id}} @$campaigns),
    );

    my %client_ids = map {$_->{'client_id'} => undef} @$campaigns;
    my @client_ids = sort {$a <=> $b} keys %client_ids;

    INFOF("number of Client IDs found: %s (%s)", scalar(@client_ids), join(', ', @client_ids),);

    my %graphite_data = (
        error_campaigns   => 0,
        error_clients     => 0,
        started_campaigns => 0,
        started_clients   => 0,
    );

    my $starting_message;
    my $not_starting_message;

    foreach my $client_id (@client_ids) {
        $self->app->send_heartbeat() if $heartbeat && $self->app->can('send_heartbeat');

        try {
            my $can_run_campaign = $self->agreement_checker->has_agreement_for_any_product_for_today(
                client_id => $client_id,
                products  => $products_for_agreement_checker,
            );

            INFOF(
                "Client ID %s has_agreement_for_any_product_for_today (%s): %s",
                $client_id,
                join(', ', @$products_for_agreement_checker),
                ($can_run_campaign ? 'true' : 'false'),
            );

            if ($can_run_campaign) {
                foreach my $campaign (grep {$_->{'client_id'} eq $client_id} @$campaigns) {

                    if (@$page_ids_to_ignore && in_array($campaign->{'page_id'}, $page_ids_to_ignore)) {
                        INFO "Ignoring Page ID " . $campaign->{'page_id'};
                        next;
                    }

                    if ($from eq 'testing' && !$model->check_action($campaign->{'id'}, 'start')) {
                        INFO "Can't make action 'start'. Skipping Page ID " . $campaign->{'page_id'};
                        next;
                    }

                    my $error_message = '';

                    my $success = FALSE;
                    try {
                        INFO "Starting Page ID " . $campaign->{'page_id'};
                        if ($from eq 'stopped_or_testing' && $model->check_action($campaign->{'id'}, 'start_testing')) {
                            $model->do_action($campaign->{'id'}, 'start_testing');
                        }
                        $model->do_action($campaign->{'id'}, 'start');
                        $success = TRUE;
                        $graphite_data{started_campaigns}++;
                    }
                    catch {
                        $error_message = $_[0]->message;
                        $graphite_data{error_campaigns}++;
                        INFO $error_message;
                    };

                    if ($success) {
                        # Главное сообщение в логе про автозапуск. На это сообщение
                        # будет заведен мониторинг, который проверят что
                        # за последние N дней был хотя бы один успешный автозапуск.
                        INFOF('Page ID %s successfully autostarted. Model: %s', $campaign->{'page_id'}, ref($model));
                    }

                    $starting_message .= gettext(
                        "Autostarting %s (Page ID: %s - %s), login: %s\n",
                        $campaign->{$additional_field},
                        $campaign->{'page_id'},
                        $campaign->{'caption'},
                        $campaign->{'login'} . ($error_message ? " - ERROR $error_message" : ''),
                    );
                }

                $graphite_data{started_clients}++;
            } else {

                foreach my $campaign (grep {$_->{'client_id'} eq $client_id} @$campaigns) {

                    INFO "Not starting Page ID " . $campaign->{'page_id'};

                    $not_starting_message .= gettext(
                        "Not autostarting %s (Page ID: %s - %s), login: %s\n",
                        $campaign->{$additional_field},
                        $campaign->{'page_id'},
                        $campaign->{'caption'},
                        $campaign->{'login'},
                    );
                }
            }
        }
        catch {
            my ($exception) = @_;
            $graphite_data{error_clients}++;

            $self->exception_dumper->dump_as_html_file($exception);

            my $tmp_message = gettext("Error when starting: %s", $exception->message());
            $tmp_message .=
              " while requesting balance.has_agreement_for_any_product_for_today for Client ID $client_id\n";
            INFO $tmp_message;
            $starting_message .= $tmp_message;

            if ($RETRY_COUNT_MAX >= $graphite_data{error_clients}) {
                push @client_ids, $client_id;
                INFO "Will retry Client ID $client_id";
            }
        };
        last if ($RETRY_COUNT_MAX < $graphite_data{error_clients});
    }

    my $prefix = sprintf('auto_start.%s.', $model->accessor);
    foreach my $metric (keys %graphite_data) {
        send_to_graphite(
            interval => 'one_day',
            path     => $prefix . $metric,
            value    => $graphite_data{$metric},
            solomon  => {
                model  => $model->accessor,
                sensor => 'auto_start.' . $metric,
            }
        );
    }

    if ($starting_message) {
        $self->mail_notification->add(
            type    => 0,
            user_id => 0,
            opts    => {
                check_alive => FALSE,
                subject     => $starting_email_subject,
                to          => $START_STOP_MAIL,
                values      => {
                    message_body       => $starting_message,
                    plain_text_wrapper => TRUE,
                },
            },
        );
    }

    if ($not_starting_message) {
        $self->mail_notification->add(
            type    => 0,
            user_id => 0,
            opts    => {
                check_alive => FALSE,
                subject     => $not_starting_email_subject,
                to          => $START_STOP_MAIL,
                values      => {
                    message_body       => $not_starting_message,
                    plain_text_wrapper => TRUE,
                },
            },
        );
    }

    return TRUE;
}

1;
