package Application::Model::AutoStop;

=encoding UTF-8

=cut

use qbit;

use base qw(QBit::Application::Model);

sub accessor {'auto_stop'}

use Utils::Logger qw(INFO INFOF ERROR);
use Utils::MonitoringUtils qw(send_to_graphite);
use PiConstants qw($START_STOP_MAIL);

__PACKAGE__->model_accessors(
    agreement_checker => 'Application::Model::AgreementChecker',
    exception_dumper  => 'Application::Model::ExceptionDumper',
    mail_notification => 'Application::Model::MailNotification',
    partner_db        => 'Application::Model::PartnerDB',
);

my $RETRY_COUNT_MAX = 100;

=head2 do_auto_stop

=cut

sub prepare_auto_stop {
    my ($self, %opts) = @_;

    my $model                          = delete($opts{'model'});
    my $products_for_agreement_checker = delete($opts{'products_for_agreement_checker'});
    my $additional_filter              = delete($opts{'additional_filter'});

    INFO "model " . ref($model);

    my $curdate = curdate(oformat => 'db');
    my $tomorrow = date_add($curdate, iformat => 'db', oformat => 'db', day => 1);
    my $already_checked = $self->partner_db->cron_auto_stop->get_all(
        fields => ['user_id'],
        filter => [AND => [[dt => '>=' => \curdate(oformat => 'db')], [model => '=' => \$model->accessor()],]],
    );
    my $filter = $model->get_db_filter({multistate => 'working'});
    $filter->and(
        [
            owner_id => 'IN' => $self->partner_db->query->select(
                table  => $self->partner_db->users,
                alias  => 'u',
                fields => ['id'],
              )->left_join(    # есть завершившиесы договоры
                table  => $self->partner_db->contracts,
                alias  => 'sc',                                             # stopped contracts
                fields => [],
                filter => [AND => [[client_id => 'IS NOT' => \undef],],],
                join_on => [AND => [[{client_id => 'sc'} => '=' => {client_id => 'u'}], [end_dt => '<=' => \$curdate]]],
              )->left_join(                                                 # нет рабочих договоров
                table   => $self->partner_db->contracts,
                alias   => 'wc',                                            # worked contracts
                fields  => [],
                filter  => [AND => [[client_id => IS => \undef],],],
                join_on => [
                    AND => [
                        [{client_id => 'wc'} => '=' => {client_id => 'u'}],
                        [dt => '<=' => \$tomorrow],
                        [OR => [[end_dt => '>' => \$curdate], [end_dt => IS => \undef],]],
                    ]
                ],
              )->distinct()
        ]
    );
    $filter->and([owner_id => 'NOT IN' => \[map {$_->{user_id}} @$already_checked]]) if @$already_checked;
    $filter->and($additional_filter) if ($additional_filter);
    my $has_is_assessor = $model->get_model_fields->{is_assessor} ? 1 : 0;
    my $campaigns = $model->get_all(
        fields => [qw(client_id owner_id page_id), ($has_is_assessor ? 'is_assessor' : ())],
        filter => $filter,
    );

    INFO "number of working campaigns found: " . scalar(@$campaigns);

    my %client_ids;
    $client_ids{$_->{client_id}} = $_ foreach (@$campaigns);
    my @client_id_list = sort {$a <=> $b} keys(%client_ids);

    INFO "number of client ids found: " . scalar(@client_id_list);

    my $error_clients = 0;
    foreach my $client_id (@client_id_list) {
        try {
            my $need_to_stop_campaigns = not(
                $client_ids{$client_id}{is_assessor}
                || $self->agreement_checker->has_agreement_for_any_product_for_tomorrow(
                    client_id          => $client_id,
                    products           => $products_for_agreement_checker,
                    not_check_assessor => $has_is_assessor,
                )
            );

            INFO "Client ID $client_id need to stop campaings: " . ($need_to_stop_campaigns ? 'true' : 'false');

            $self->partner_db->cron_auto_stop->add(
                {
                    dt      => curdate(oformat => 'db'),
                    model   => $model->accessor(),
                    user_id => $client_ids{$client_id}{owner_id},
                    keep_working => ($need_to_stop_campaigns ? 0 : 1),
                }
            );
        }
        catch {
            my ($exception) = @_;

            $self->exception_dumper->dump_as_html_file($exception);
            $error_clients++;

            INFO gettext("Error when preparing to stop: %s\n", $exception->message());

            if ($RETRY_COUNT_MAX >= $error_clients) {
                push @client_id_list, $client_id;
                INFO "Will retry Client ID $client_id";
            }
        };
        last if ($RETRY_COUNT_MAX < $error_clients);
    }

    my $prefix = sprintf('auto_stop.%s.', $model->accessor);
    send_to_graphite(
        interval => 'one_day',
        path     => $prefix . 'error_clients',
        value    => $error_clients,
        solomon  => {
            model  => $model->accessor,
            sensor => 'auto_stop.' . 'error_clients',
        }
    );

    return TRUE;
}

sub do_auto_stop {
    my ($self, %opts) = @_;

    my $model                          = delete($opts{'model'});
    my $email_subject                  = delete($opts{'email_subject'});
    my $additional_field               = delete($opts{'additional_field'}) // 'domain';
    my $additional_filter              = delete($opts{'additional_filter'});
    my $email                          = delete($opts{'email'});
    my $products_for_agreement_checker = delete($opts{'products_for_agreement_checker'});
    my $heartbeat                      = delete($opts{'heartbeat'});

    INFO "model " . ref($model);

    my $filter = $model->get_db_filter({multistate => 'working'});
    my $neeed_to_stop = $self->partner_db->cron_auto_stop->get_all(
        fields => ['user_id'],
        filter => [
            AND => [
                [dt           => '=' => \date_sub(curdate(), hour => 23, oformat => 'db',)],
                [model        => '=' => \$model->accessor()],
                [keep_working => '=' => \0],
            ]
        ],
    );
    $filter->and([owner_id => 'IN' => \[map {$_->{user_id}} @$neeed_to_stop]]);
    $filter->and($additional_filter) if ($additional_filter);
    my $has_is_assessor = $model->get_model_fields->{is_assessor} ? 1 : 0;
    my $campaigns = $model->get_all(
        fields => [
            qw(id multistate_name client_id login caption page_id),
            $additional_field,
            ($has_is_assessor ? 'is_assessor' : ())
        ],
        filter => $filter,
    );

    INFO "number of working campaigns found: " . scalar(@$campaigns);

    my $message;

    my %client_ids;
    push(@{$client_ids{$_->{client_id}}}, $_) foreach (@$campaigns);
    my @client_id_list = sort {$a <=> $b} keys(%client_ids);

    INFO "number of client ids found: " . scalar(@client_id_list);

    my %graphite_data = (
        error_campaigns   => 0,
        stopped_campaigns => 0,
        stopped_clients   => 0,
    );
    foreach my $client_id (@client_id_list) {
        $self->app->send_heartbeat() if $heartbeat && $self->app->can('send_heartbeat');

        # Rechek contract
        if (
            $self->agreement_checker->has_agreement_for_any_product_for_tomorrow(
                client_id          => $client_id,
                products           => $products_for_agreement_checker,
                not_check_assessor => $has_is_assessor,
            )
           )
        {
            INFO "Client ID $client_id need to stop campaings: false";
            next;
        }
        INFO "Client ID $client_id need to stop campaings: true";

        foreach my $campaign (@{$client_ids{$client_id}}) {

            $message .= gettext(
                "Stopping %s (Page ID: %s - %s), login: %s\n",
                $campaign->{$additional_field},
                $campaign->{'page_id'},
                $campaign->{'caption'},
                $campaign->{'login'},
            );

            my $success = FALSE;
            try {
                INFO "Stopping Page ID " . $campaign->{'page_id'};
                $model->do_action($campaign->{'id'}, 'stop');
                $success = TRUE;
                $graphite_data{stopped_campaigns}++;
            }
            catch {
                my $tmp_message = gettext("Error when stopping: %s\n", $@->message());
                INFO $tmp_message;
                $message .= $tmp_message;
                $graphite_data{error_campaigns}++;
            };

            if ($success) {
                # Главное сообщение в логе про автостоп. На это сообщение
                # будет заведен мониторинг, который проверят что
                # за последние N дней был хотя бы один успешный автостоп.
                INFOF('Page ID %s successfully autostopped. Model: %s', $campaign->{'page_id'}, ref($model));
            }
        }
        $graphite_data{stopped_clients}++;
    }

    my $prefix = sprintf('auto_stop.%s.', $model->accessor);
    foreach my $metric (sort keys %graphite_data) {
        send_to_graphite(
            interval => 'one_day',
            path     => $prefix . $metric,
            value    => $graphite_data{$metric},
            solomon  => {
                model  => $model->accessor,
                sensor => 'auto_stop.' . $metric,
            }
        );
    }

    if ($message) {
        $self->mail_notification->add(
            type    => 0,
            user_id => 0,
            opts    => {
                check_alive => FALSE,
                subject     => $email_subject,
                to          => [$START_STOP_MAIL, (defined($email) ? $email : ())],
                values      => {
                    message_body       => $message,
                    plain_text_wrapper => TRUE,
                },
            },
        );
    }

    return TRUE;
}

sub check_soon_auto_stop {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                after_date      => {type => 'date'},
                warn_date       => {type => 'date'},
                crit_date       => {type => 'date'},
                money_last_days => {type => 'int_un'},
                money_per_month => {type => 'int_un'},
            }
        },
        app   => $self,
        throw => TRUE,
    );

    my $after_date = $opts{after_date};
    my $warn_date  = $opts{warn_date};
    my $crit_date  = $opts{crit_date};

    my $rows = $self->partner_db->contracts->get_all(
        filter => [AND => [[end_dt => '>=' => \$after_date], [end_dt => '<=' => \$warn_date]]],
        fields => [qw(client_id end_dt)],
    );

    my @clients         = map {$_->{client_id}} @$rows;
    my $money_date_to   = $after_date;
    my $money_date_from = date_sub($money_date_to, iformat => 'db', oformat => 'db', day => $opts{money_last_days});
    my $money_limit =
      int dates_delta_days($money_date_from, $money_date_to, iformat => 'db') / 30 * $opts{money_per_month};

    my $money = $self->app->bk_statistics->get_money_for_clients_on_dates(
        clients   => \@clients,
        date_to   => $money_date_to,
        date_from => $money_date_from,
    );

    my (@warn, @crit);
    for my $row (@$rows) {
        my $client_money = $money->{$row->{client_id}};
        next unless $client_money && $client_money >= $money_limit;
        if ($row->{end_dt} le $crit_date) {
            push @crit, $row->{client_id};
        } else {
            push @warn, $row->{client_id};
        }
    }

    return \(@warn, @crit);
}

1;
