package Application::Model::BKStatistics;

use qbit;

use CHI;
use Hash::Util qw(lock_keys);
use Utils::PublicID;
use Utils::Logger qw(WARN WARNF INFOF);

use base qw(
  QBit::Application::Model
  Application::Model::BKStatistics::Utils
  );

use Application::Model::BKStatistics::Fields;
use Exception;
use Exception::API::HTTPMOL::TooManyGroupBy;
use Exception::Denied;
use Exception::Validation::BadArguments;

use PiConstants qw(
  $CATEGORY_NAMES
  $CATEGORY_COUNTER

  $MOL_STAT_TYPES
  $MOL_REPORT_TYPE_MAIN
  $MOL_REPORT_TYPE_DSP
  $MOL_REPORT_TYPE_MM

  $CATEGORY_MOL_BASIC

  $YNDX_PARTNER_INTAPI_USER_ID
  $VIDEO_BLOCK_TYPES
  $SITE_VERSIONS
  $STAT_AVAILABLE_CURRENCIES
  );

use Application::Model::Product::AN::MobileApp::BlockTypes::Settings qw(
  @BLOCK_TYPES
  );

sub accessor {'bk_statistics'}

__PACKAGE__->model_accessors(
    geo_base     => 'Application::Model::GeoBase',
    cur_user     => 'Application::Model::CurUser',
    api_http_mol => 'Application::Model::API::Yandex::HTTPMOL',
    rbac         => 'Application::Model::RBAC',
    resources    => 'Application::Model::Resources',
    crimea       => 'Application::Model::Crimea',
    memcached    => 'QBit::Application::Model::Memcached',
    users        => 'Application::Model::Users',
    partner_db   => 'Application::Model::PartnerDB',
);

my $CACHE = CHI->new(driver => 'Memory', global => 1);

my $COMPARE_SPECIFIC_FIELDS_RE = '^(\w+?)(?:_(a|b|delta|absdelta))?$';

our @FIELDS = @Application::Model::BKStatistics::Fields::FIELDS;

foreach my $i (0 .. $#FIELDS) {
    $FIELDS[$i]->{index} = $i;
}

my %PARTNER2_TO_MOL;
my %PARTNER2_FIELDS;
my %MOL_TO_PARTNER2;
my %MOL_FIELDS;
my %GROUP_BY_PARTNER2_FIELDS;

foreach my $field (@FIELDS) {
    foreach
      my $stat_type (@{$field->{'report_types'} // die "$field->{'mol_id'} $field->{'partner2_id'} no report type"})
    {
        $MOL_TO_PARTNER2{$stat_type}{$field->{'mol_id'}}      = $field->{'partner2_id'};
        $MOL_FIELDS{$stat_type}{$field->{'mol_id'}}           = $field;
        $PARTNER2_TO_MOL{$stat_type}{$field->{'partner2_id'}} = $field->{'mol_id'};
        $PARTNER2_FIELDS{$stat_type}{$field->{'partner2_id'}} = $field;
        if (exists($field->{'group_by'}) && $field->{'group_by'}) {
            $GROUP_BY_PARTNER2_FIELDS{$stat_type}{$field->{'partner2_id'}} = 1;
        }
    }
}

lock_keys(%MOL_TO_PARTNER2);
map {lock_keys(%{$MOL_TO_PARTNER2{$_}})} keys %MOL_TO_PARTNER2;

lock_keys(%PARTNER2_TO_MOL);
map {lock_keys(%{$PARTNER2_TO_MOL{$_}})} keys %PARTNER2_TO_MOL;

lock_keys(%PARTNER2_FIELDS);
map {lock_keys(%{$PARTNER2_FIELDS{$_}})} keys %PARTNER2_FIELDS;

lock_keys(%MOL_FIELDS);
map {lock_keys(%{$MOL_FIELDS{$_}})} keys %MOL_FIELDS;

for my $stat_type (@$MOL_STAT_TYPES) {
    $GROUP_BY_PARTNER2_FIELDS{$stat_type}{'date'} = 1;
    $GROUP_BY_PARTNER2_FIELDS{$stat_type}{'geo'}  = 1;
}

lock_keys(%GROUP_BY_PARTNER2_FIELDS);
map {lock_keys(%{$GROUP_BY_PARTNER2_FIELDS{$_}})} keys %GROUP_BY_PARTNER2_FIELDS;

=head2 get_statistics2

=cut

sub get_statistics2 {
    my ($self, %opts) = @_;

    $self->_check_access();

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            extra  => TRUE,
            fields => {
                stat_type => {
                    optional => TRUE,
                    'in'     => $MOL_STAT_TYPES,
                    'type'   => 'scalar',
                },
            },
        },
        throw => TRUE,
    );

    my $stat_type = delete $opts{stat_type} // $MOL_REPORT_TYPE_MAIN;

    my @fields          = $self->_get_fields($stat_type);
    my @select_fields   = map {$_->{partner2_id}} grep {$_->{select}} @fields;
    my @group_by_fields = map {$_->{partner2_id}} grep {$_->{group_by}} @fields;

    my %available_for_order_by = map {$_ => 1} (@{$opts{fields}}, @{$opts{entity_fields}});
    for my $dimension (qw(date geo)) {
        if ($self->_get_group($opts{'dimension_fields'}, $dimension, $self->_get_groups($dimension))) {
            $available_for_order_by{$dimension} = 1;
        }
    }

    my $limit_min = 1;
    my $limit_max = 500;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                currency => {
                    optional => TRUE,
                    in       => $STAT_AVAILABLE_CURRENCIES,
                },
                limits => {
                    optional => TRUE,
                    type     => 'hash',
                    fields   => {
                        limit => {
                            type => 'int_un',
                            (
                                $self->check_rights('bk_statistics_allow_any_limit')
                                ? ()
                                : (
                                    min => $limit_min,
                                    max => $limit_max,
                                    msg => gettext('limit must be in the range from %s to %s', $limit_min, $limit_max),
                                  ),
                            ),
                        },
                        offset => {type => 'int_un'},
                    }
                },
                fields => {
                    type     => 'array',
                    size_min => 1,
                    all      => {
                        in  => \@select_fields,
                        msg => gettext('Incorrect value in fields'),
                    },
                },
                entity_fields => {
                    type => 'array',
                    all  => {
                        in  => \@group_by_fields,
                        msg => gettext('Incorrect value in entity_fields'),
                    },
                },
                order_by => {
                    optional => TRUE,
                    type     => 'array',
                    all      => {
                        type   => 'hash',
                        fields => {
                            dir   => {in => [qw(asc desc)],},
                            field => {in => [sort keys %available_for_order_by],},
                        },
                    }
                }
            },
            extra => 1,
        },
        throw => TRUE,
    );

    my $top_keys          = delete($opts{'top_keys'});
    my $top_keys_order_by = delete($opts{'top_keys_order_by'});

    my $answer;

    my ($date_from, $date_to) = $self->_get_dates_from_period($opts{'period'});
    _validate_comparing_periods($opts{'period'}) if _is_comparing_periods($date_from);

    my %params = (
        (exists $opts{currency} ? (currency => $opts{currency}) : ()),
        stat_type => $stat_type,
        date_from => $date_from,
        date_to   => $date_to,
        (
            defined($opts{'limits'})
            ? (
                limit  => $opts{'limits'}{'limit'},
                offset => $opts{'limits'}{'offset'}
              )
            : ()
        ),
    );
    # Параметр top_keys имеет смысл, только в том случае если указаны
    # какие-то поля для group_by
    if ($top_keys && (@{$opts{'entity_fields'}} || exists $available_for_order_by{geo})) {
        lock_keys(%opts);
        throw Exception::Validation::BadArguments gettext("Incorrect top_keys '%s'", $top_keys) if $top_keys > 20;

        my $first_mol_request = $self->_convert_get_statistics2_to_mol_request(
            stat_type => $stat_type,
            get_statistics2 => {%opts, order_by => $top_keys_order_by,},
        );

        return $self->_get_empty_get_statistics2_answer(%params) unless defined($first_mol_request);

        delete($first_mol_request->{'group_by_date'});
        $first_mol_request->{'limits'} = {
            limit  => $top_keys,
            offset => 0,
        };

        my $first_mol_answer = $self->_safe_get_data(%$first_mol_request);

        my $is_top_keys_cropped;
        if ($first_mol_answer->{total_rows} > $top_keys) {
            $is_top_keys_cropped = TRUE;
        }

        my $second_mol_request = $self->_convert_get_statistics2_to_mol_request(
            stat_type       => $stat_type,
            get_statistics2 => \%opts,
        );

        return $self->_get_empty_get_statistics2_answer(%params) unless defined($second_mol_request);
        my $additional_where = $self->_get_additional_where_from_mol_answer($first_mol_answer, $stat_type);
        if (@$additional_where) {
            my @where_sum;
            if (defined $second_mol_request->{'filters_pre'}->{'-and'}) {
                foreach my $orig_where (@{$second_mol_request->{'filters_pre'}->{'-and'}}) {
                    foreach my $add_where (@$additional_where) {
                        push @where_sum, {%$orig_where, %$add_where};
                    }
                }
            } else {
                @where_sum = @$additional_where;
            }
            $second_mol_request->{'filters_pre'}->{'-and'} = \@where_sum;
        }

        my $second_mol_response = $self->_safe_get_data(%$second_mol_request);

        $answer = $self->_convert_mol_response_to_get_statistics2(%params, mol_response => $second_mol_response);
        if ($is_top_keys_cropped) {
            $answer->{top_keys_cropped} = $is_top_keys_cropped;
        }
    } else {
        my $mol_request = $self->_convert_get_statistics2_to_mol_request(
            stat_type       => $stat_type,
            get_statistics2 => \%opts,
        );

        if (defined($mol_request)) {
            my $mol_response = $self->_safe_get_data(%$mol_request);
            $answer = $self->_convert_mol_response_to_get_statistics2(%params, mol_response => $mol_response);
        } else {
            $answer = $self->_get_empty_get_statistics2_answer(%params);
        }
    }

    return $answer;
}

=head2 get_tree2

Возвращает стуктуру данных, которую фронт использует для отрисовки
констуктора статистики.

=cut

sub get_tree2 {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                stat_type => {
                    optional => TRUE,
                    'in'     => $MOL_STAT_TYPES,
                    'type'   => 'scalar',
                },
            },
        },
        throw => TRUE,
    );

    my $stat_type = delete($opts{stat_type}) // $MOL_REPORT_TYPE_MAIN;

    $self->_check_access();

    my $date_groups   = $self->_get_date_groups();
    my $region_levels = $self->_get_region_levels();

    my $tree = [
        {
            id    => "payment",
            title => gettext('Payment'),

            has_product  => "",
            has_raw_stat => "",

            conflict_fields => [],

            dimension_fields => [
                {
                    id     => "date",
                    title  => gettext('mol_date'),
                    type   => "select",
                    values => [map {[$_ => $date_groups->{$_}]} sort keys %$date_groups],
                },
                (
                    $stat_type eq $MOL_REPORT_TYPE_MM || $stat_type eq $MOL_REPORT_TYPE_DSP
                    ? ()
                    : (
                        {
                            id     => 'geo',
                            title  => gettext('mol_geo'),
                            type   => "select",
                            values => [map {[$_ => $region_levels->{$_}]} sort keys %$region_levels],
                        }
                      )
                ),
            ],

            # Фильтры - where
            entity_filter_simple_fields => $self->_get_tree2__entity_filter_simple_fields($stat_type, for_tree => TRUE),

            # Для того чтобы работали фильтры
            entity_filter_fields => $self->_get_tree2__entity_filter_fields($stat_type, for_tree => TRUE),

            # Группировки данных - group by
            entity_fields => $self->_get_tree2__entity_fields($stat_type, for_tree => TRUE),

            # Исходные данные
            # Денежные показатели
            fields => $self->_get_tree2__fields($stat_type, for_tree => TRUE),
        },
    ];

    return {tree => $tree,};
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->register_rights($self->get_structure_rights_to_register());
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name   => $self->accessor(),
            rights => {
                bk_statistics_always_view => d_gettext('Right to view bk_statistics'),
                bk_statistics_view        => d_gettext('Right to view bk_statistics'),
                bk_statistics_view_dsp    => d_gettext('Right to view bk_statistics'),
                map {
                        $self->accessor()
                      . '_view_field__'
                      . $_ => d_gettext('Right to view field "%s" for %s', $_, $self->accessor())
                  } qw(
                  application_type
                  block_level
                  block_type
                  client_id
                  deal_caption
                  deal_id
                  domain
                  dsp_caption
                  dsp_id
                  login
                  multistate
                  page_level
                  site_version

                  all_wo_nds
                  clicks_direct
                  cpmv_all_wo_nds
                  ctr_direct
                  ecpm_all_wo_nds
                  rpm_all_wo_nds
                  rec_widget_hits

                  hits_dsp
                  price_dsp
                  price_dsp_average

                  shows_direct
                  shows_direct_banners

                  rec_widget_ctr_total
                  rec_widget_visibility
                  video_in_rtb

                  owner_id
                  application_id

                  partner_wo_nds_bk
                  revenue_original_mm_eur
                  revenue_original_mm_rub
                  revenue_original_mm_usd
                  ),
            }
        }
    ];
}

sub is_available {
    my ($self) = @_;

    if ($self->check_short_rights('always_view')) {
        return TRUE;
    } elsif ($self->check_short_rights('view_dsp')) {
        return TRUE;
    } else {
        my $cur_user = $self->app->get_option('cur_user') or throw Exception gettext('Current user is undefined');
        my $cache_key = $cur_user->{'id'} . '_' . ref($self);

        my $is_available = $self->memcached->get(available_stat_levels => $cache_key);
        unless (defined $is_available) {
            my $res = $self->get_statistics2(
                'entity_fields' => ['page_id'],
                'period'        => [
                    date_sub(curdate(oformat => 'db'), day => 365 * 2, iformat => 'db', oformat => 'db'),
                    curdate(oformat => 'db')
                ],
                'dimension_fields' => [],
                'total'            => 0,
                'fields'           => ['shows'],
                'limits'           => {
                    limit  => 1,
                    offset => 0,
                },
            );

            $is_available = exists $res->{total_rows} && $res->{total_rows} > 0;
            $self->memcached->set(available_stat_levels => $cache_key, $is_available, 30 * 60);
        }
        return $is_available;
    }
}

sub get_available_statistics_levels {
    my ($self) = @_;
    return [qw(payment), (map {$_->{id}} (@{$self->_get_block_model_values()}, @{$self->_get_product_id_values()}))];
}

sub _check_access {
    my ($self) = @_;

    my $resources = $self->resources->get_get_resources(['bk_statistics']);

    throw Exception::Denied unless $resources->{'bk_statistics'};

    return 1;
}

=begin comment _convert_get_statistics2_to_mol_request

Возвращает undef в том случае если не нужно отправлять запрос в МОЛ.

=end comment

=cut

sub _convert_get_statistics2_to_mol_request {
    my ($self, %params) = @_;

    my $stat_type = delete $params{stat_type};

    my $get_statistics2 = delete($params{'get_statistics2'});
    throw Exception gettext("Internal error no get_statistics2") unless $get_statistics2;

    my %opts = %{clone($get_statistics2)};

    my $group_by_date = $self->_get_group($opts{'dimension_fields'}, 'date', $self->_get_date_groups());
    my $region_level  = $self->_get_group($opts{'dimension_fields'}, 'geo',  $self->_get_region_levels());

    my ($date_from, $date_to) = $self->_get_dates_from_period($opts{'period'});
    my $is_comparing_periods = _is_comparing_periods($date_from);

    my $limit;
    my $offset;

    if (defined $opts{'limits'}) {
        $limit = $opts{'limits'}{'limit'};
        throw Exception::Validation::BadArguments "Must specify limits.limit" unless defined($limit);

        $offset = $opts{'limits'}{'offset'};
        throw Exception::Validation::BadArguments "Must specify limits.offset" unless defined($offset);
    }

    my $order_by = $opts{'order_by'};
    if ($order_by) {
        foreach my $el (@{$order_by}) {
            # if sorting is on countable field with comparing periods we have to pass
            # specific field with postfix, the original field is not present in countable_fields
            if ($is_comparing_periods) {
                if (grep {$el->{'field'} eq $_} @{$opts{'fields'}}) {
                    $el->{'field'} = $PARTNER2_TO_MOL{$stat_type}{$el->{'field'}} . '_a';
                } else {
                    $el->{'field'} = $PARTNER2_TO_MOL{$stat_type}{$el->{'field'}};
                }
            } else {
                $el->{'field'} = $PARTNER2_TO_MOL{$stat_type}{$el->{'field'}};
            }
        }
    }

    foreach my $el (@{$opts{'fields'}}) {
        $el = $PARTNER2_TO_MOL{$stat_type}{$el};
    }

    foreach my $el (@{$opts{'entity_fields'}}) {
        $el = $PARTNER2_TO_MOL{$stat_type}{$el};
    }

    my $filters_pre;

    my %operations = (
        'LIKE'     => 'contains',
        'NOT LIKE' => 'not_contains',
        '='        => 'eq',
        '<>'       => 'ne',
        'IN'       => 'eq',
        'NOT IN'   => 'ne',
    );

    my %where_fields = map {$_->{partner2_id} => 1} grep {$_->{where}} $self->_get_fields($stat_type);
    my %filter_ids;
    my $filter_ids_operator = 'eq';

    if ($opts{'levels'} && defined($opts{'levels'}[0]{'filter'})) {
        throw gettext("%s is unsupported", $opts{'levels'}[0]{'filter'}[0])
          if $opts{'levels'}[0]{'filter'}[0] ne 'AND';

        my %additional_where;
        foreach my $el (@{$opts{'levels'}[0]{'filter'}[1]}) {

            my $partner2_field_id = $el->[0];
            my $operation         = $el->[1];
            my $value             = $el->[2];

            unless ($where_fields{$partner2_field_id}) {
                throw Exception::Validation::BadArguments sprintf('Unknown fields "%s"', $partner2_field_id);
            }

            my $field = $PARTNER2_FIELDS{$stat_type}{$partner2_field_id};
            my $field_label = ref($field->{'label'}) eq 'CODE' ? $field->{'label'}->() : $field->{'label'};

            my $mol_field_id = $PARTNER2_TO_MOL{$stat_type}{$partner2_field_id};
            throw Exception::Validation::BadArguments gettext("Operation %s is unsupported", $operation)
              unless exists($operations{$operation});

            if ($field->{'json_type'} eq 'number') {
                my $number_re =
                  exists $field->{'is_number_signed'} && $field->{'is_number_signed'} ? '^-?[0-9]+\z' : '^[0-9]+\z';
                if (ref($value) eq '') {
                    throw Exception::Validation::BadArguments gettext('Incorrect value for field "%s"', $field_label)
                      if $value !~ /$number_re/;
                } elsif (ref($value) eq 'ARRAY') {
                    foreach my $part_value (@{$value}) {
                        throw Exception::Validation::BadArguments gettext('Incorrect value for field "%s"',
                            $field_label)
                          if $part_value !~ /$number_re/;
                    }
                } else {
                    throw Exception::Validation::BadArguments gettext('Incorrect value for field "%s"', $field_label);
                }
            }

            if ($field->{'json_type'} eq 'boolean') {
                $value = $value ? 1 : 0;
            }

            if (ref($value) ne 'ARRAY') {
                $value = [$value];
            }
            $value = [map {"$_"} @$value];

            # Поля Public ID в сервере МОЛ БК нет, поэтому из Public ID делаем набор Page ID + Номер блока и формируем фильтр из этих пар
            if (in_array($partner2_field_id, ['public_id', 'complex_block_id'])) {
                foreach my $public_id (@$value) {
                    my ($prefix, $page_id, $block_id) = split_block_public_id($public_id);
                    # добавляем площадку в фильтр, так МОЛ быстрее работает
                    # https://st.yandex-team.ru/PI-25104#612f4fd25c54a3159ed3fcd5
                    push @{$filters_pre->{PageID}{eq}}, $page_id unless $operations{$operation} eq 'ne';
                    if (exists $additional_where{$page_id}) {
                        unless (exists $additional_where{$page_id}{ImpID}{$operations{$operation}}) {
                            $additional_where{$page_id}{PageID} = {eq => [$page_id]};
                        }
                        push @{$additional_where{$page_id}{ImpID}{$operations{$operation}}}, $block_id;
                    } else {
                        $additional_where{$page_id} = {
                            PageID => {eq                      => [$page_id]},
                            ImpID  => {$operations{$operation} => [$block_id]},
                        };
                        if ('ne' eq $operations{$operation}) {
                            $additional_where{force_include_page_ids}{PageID}{ne} //= [];
                            push @{$additional_where{force_include_page_ids}{PageID}{ne}}, $page_id;
                        }
                    }
                }
            } elsif ($partner2_field_id eq 'login') {
                $filter_ids_operator = $operations{$operation};

                if ($stat_type eq $MOL_REPORT_TYPE_DSP) {
                    my $dsp_id_list = $self->partner_db->query->select(
                        table  => $self->partner_db->dsp,
                        fields => [qw(id)],
                      )->join(
                        table   => $self->partner_db->users,
                        fields  => [qw(login)],
                        filter  => ['AND', [['login', '=', \$value]]],
                        join_on => [id => '=' => {owner_id => $self->partner_db->dsp}],
                      )->get_all;
                    # если был фильтр по логину, то нужно что-то положить
                    $dsp_id_list = [{id => 0}] unless @$dsp_id_list;
                    # только накапливаем фильтр, применение ниже
                    @filter_ids{map {$_->{id}} @$dsp_id_list} = ();
                } else {
                    my $page_id_list = $self->partner_db->query->select(
                        table  => $self->partner_db->all_pages,
                        fields => [qw(page_id)],
                        filter => ['AND', [['login', '=', \$value]]],
                    )->get_all;
                    # если был фильтр по логину, то нужно что-то положить
                    $page_id_list = [{page_id => 0}] unless @$page_id_list;
                    # только накапливаем фильтр, применение ниже
                    @filter_ids{map {$_->{page_id}} @$page_id_list} = ();
                }
            } else {
                if (exists $filters_pre->{$mol_field_id}) {
                    $filters_pre->{$mol_field_id}{$operations{$operation}} //= [];
                    push @{$filters_pre->{$mol_field_id}{$operations{$operation}}}, @$value;

                } else {
                    $filters_pre->{$mol_field_id} = {$operations{$operation} => $value};
                }
            }
        }

        if (keys %additional_where) {
            $filters_pre->{'-and'} //= [];
            push @{$filters_pre->{'-and'}}, map {$additional_where{$_}}
              sort {$b cmp $a} keys %additional_where;
        }
    }

    my $can_view_all_page_ids =
         $self->rbac->cur_user_is_internal()
      || $self->_cur_user_is_intapi()
      || (0 == $self->cur_user->get_cur_user_id());

    my $id_field_name;
    # в зависимости от типа отчета выбираем поле и делаем настройки запроса
    if ($stat_type eq $MOL_REPORT_TYPE_DSP) {
        $id_field_name = 'DSPID';
    } else {
        $id_field_name = 'PageID';
        $filters_pre->{isPi2} = {eq => [1]};
    }
    my ($date_from1, $date_to1) =
      _is_comparing_periods($date_from) ? ($date_from->[0], $date_from->[1]) : ($date_from, $date_to);
    if ((my $delta = dates_delta_days($date_from1, $date_to1, iformat => 'date_only_numbers')) > 366 * 2) {
        throw Exception::Validation::BadArguments gettext('Too big period');
    } elsif (!$can_view_all_page_ids || %filter_ids) {
        my %strict_ids;
        unless ($can_view_all_page_ids) {
            # если нельзя смотреть все - ограничиваем доступными
            my @available_ids;
            if ($stat_type eq $MOL_REPORT_TYPE_DSP) {
                @available_ids = $self->cur_user->get_all_dsp_ids_available_for_cur_user();
            } else {
                @available_ids = $self->cur_user->get_all_page_ids_available_for_cur_user();
            }
            return undef unless @available_ids;
            @strict_ids{@available_ids} = ();
        }

        if (exists $filters_pre->{$id_field_name} && exists $filters_pre->{$id_field_name}{$filter_ids_operator}) {
            my %strict = %filter_ids;
            %filter_ids = ();
            if (%strict) {
                # если фильтр указан - то накопленное выше используем как ограничение, если есть
                @filter_ids{grep {exists $strict{$_}} @{$filters_pre->{$id_field_name}{$filter_ids_operator}}} = ();
                return undef unless %filter_ids;
            } else {
                @filter_ids{@{$filters_pre->{$id_field_name}{$filter_ids_operator}}} = ();
            }
            if (%strict_ids) {
                # применяем ограничения
                delete @filter_ids{grep {!exists $strict_ids{$_}} keys %filter_ids};
                return undef unless %filter_ids;
            }
        }
        return undef if %filter_ids && exists $filter_ids{0};
        if (%filter_ids) {
            # фильтр есть - применяем его
            $filters_pre->{$id_field_name}{$filter_ids_operator} = [sort keys %filter_ids];
        } elsif (%strict_ids) {
            # фильтра нет, но есть ограничения - применяем их как фильтр
            $filter_ids_operator = 'eq';
            $filters_pre->{$id_field_name}{$filter_ids_operator} = [sort keys %strict_ids];
        }
    } elsif ($delta > 365) {
        throw Exception::Validation::BadArguments gettext(
            'For big period, filters by login, page, or block are required')
          unless exists $filters_pre->{PageID}
              || exists $filters_pre->{DSPID}
              || exists $filters_pre->{DSPLogin}
              || exists $filters_pre->{Login}
              || (exists $filters_pre->{'-and'} && grep {exists $_->{ImpID}} @{$filters_pre->{'-and'}});
    }

    if (exists $filters_pre->{$id_field_name} && exists $filters_pre->{$id_field_name}{$filter_ids_operator}) {
        # оставляем только уникальные номера в фильтре
        my %uniq_id = map {$_ => undef} @{$filters_pre->{$id_field_name}{$filter_ids_operator}};
        $filters_pre->{$id_field_name}{$filter_ids_operator} = [sort keys %uniq_id];
    }

    my ($date_from_a, $date_to_a, $date_from_b, $date_to_b);
    if (_is_comparing_periods($date_from)) {
        ($date_from_a, $date_to_a, $date_from_b, $date_to_b) =
          dates_delta_days($date_from->[0], $date_to->[0], iformat => 'date_only_numbers') < 0
          ? ($date_from->[0], $date_from->[1], $date_to->[0], $date_to->[1])
          : ($date_to->[0], $date_to->[1], $date_from->[0], $date_from->[1]);
    }

    my $group_by = $opts{'entity_fields'};
    if (defined($region_level)) {
        push(@$group_by, 'RegionName');
    }

    my $currency = $opts{'currency'} // "RUB";
    my %mol_request = (
        currency         => $currency,
        countable_fields => $opts{'fields'},
        (
            _is_comparing_periods($date_from)
            ? (
                date_from_a => $date_from_a,
                date_to_a   => $date_to_a,
                date_from_b => $date_from_b,
                date_to_b   => $date_to_b,
              )
            : (
                date_from => $date_from,
                date_to   => $date_to,
              )
        ),
        group_by => $group_by,
        (defined($group_by_date) ? (group_by_date => $group_by_date) : ()),
        (defined($region_level)  ? (region_level  => $region_level)  : ()),
        (
            defined($opts{limits})
            ? (
                limits => {
                    limit  => $limit,
                    offset => $offset,
                }
              )
            : ()
        ),
        (defined($order_by)    ? (order_by    => $order_by)    : ()),
        (defined($filters_pre) ? (filters_pre => $filters_pre) : ()),
        lang => $self->_get_cur_user_lang(),
    );

    my $ukraine_id = 187;
    my $country_id = $self->crimea->get_country_id_with_crimea($self->cur_user->get_cur_user_id());

    if ($country_id == $ukraine_id) {
        $mol_request{'disputed_borders'} = 'UA';
    }

    return \%mol_request;
}

sub _convert_mol_response_to_get_statistics2 {
    my ($self, %opts) = @_;

    my $stat_type = delete $opts{stat_type} // $MOL_REPORT_TYPE_MAIN;

    my $mol_response = $opts{mol_response};

    my $dates = [$opts{date_from}, $opts{date_to}];
    my $is_comparing_periods = _is_comparing_periods($dates->[0]);

    my $limit  = $opts{limit};
    my $offset = $opts{offset};

    my @mol_header = @{$mol_response->{'header'}};
    my @header;
    for my $header_name (@mol_header) {
        if (exists $MOL_TO_PARTNER2{$stat_type}->{$header_name}) {
            push @header, $MOL_TO_PARTNER2{$stat_type}->{$header_name};
        } elsif ($header_name =~ /$COMPARE_SPECIFIC_FIELDS_RE/ && exists $MOL_TO_PARTNER2{$stat_type}->{$1}) {
            push @header, $MOL_TO_PARTNER2{$stat_type}->{$1} . '_' . $2;
        } else {
            throw "Got unknown field from MOL: $header_name";
        }
    }

    my $dimensions = {};
    my $measures   = {};

    my %group_by_positions;
    my %json_type_by_positions;

    my %indexes;
    my $j = 1;
    foreach my $name (
        sort {$PARTNER2_FIELDS{$stat_type}{$a}{'index'} <=> $PARTNER2_FIELDS{$stat_type}{$b}{'index'}}
        map {$_ =~ /$COMPARE_SPECIFIC_FIELDS_RE/; $1;} @header
      )
    {
        $indexes{$name} = $j++;
    }

    my $entity_fields = $self->_get_tree2__entity_filter_fields($stat_type);
    my %values_by_positions;

    foreach my $i (0 .. $#header) {
        my ($name) = ($header[$i] =~ /$COMPARE_SPECIFIC_FIELDS_RE/);
        my $field  = $PARTNER2_FIELDS{$stat_type}{$name};
        my $l      = $field->{'label'};

        my $currency = $field->{currency};
        if ($opts{currency} && $field->{frontend_type} && $field->{frontend_type} eq 'money') {
            $currency = $opts{currency};
        }
        if (exists $entity_fields->{$name} && $entity_fields->{$name}{values}) {
            $values_by_positions{$i} = {map {$_->{id} => $_->{label}} @{$entity_fields->{$name}{values}}};
        }

        my $description = {
            index => $indexes{$name},
            title => ref($l) eq 'CODE' ? $l->() : $l,
            type  => $field->{'frontend_type'},
            ($currency ? (currency => $currency) : ()),
        };

        if (exists($GROUP_BY_PARTNER2_FIELDS{$stat_type}{$name})) {
            $group_by_positions{$i} = 1;
            $dimensions->{$name} = $description;
        } else {
            $description->{'unit'} = $field->{'frontend_unit'};
            $measures->{$name} = $description;
        }

        $json_type_by_positions{$i} = $field->{'json_type'};
    }

    my %product_id2label;
    my %product_id2prefix;

    my $points = [];
    foreach my $element (@{$mol_response->{'data'}}) {
        my %dimensions;
        my %measures;

        foreach my $i (0 .. scalar(@$element) - 1) {

            # Для того чтобы frontend получал типы "числа", "строки", "boolean" в json
            if ($json_type_by_positions{$i} eq 'number') {
                $element->[$i] += 0;
            } elsif ($json_type_by_positions{$i} eq 'string') {
                $element->[$i] .= '';
            } elsif ($json_type_by_positions{$i} eq 'boolean') {
                $element->[$i] = $element->[$i] ? JSON::XS::true : JSON::XS::false;
            }

            if ($group_by_positions{$i}) {
                if ($header[$i] eq 'date') {
                    $dimensions{$header[$i]} = [$element->[$i]];
                } elsif ($header[$i] eq 'page_level' || $header[$i] eq 'block_level') {
                    my $model = $element->[$i];
                    $product_id2label{$model} //= $model ? $self->app->$model->get_product_name() : 'NA';

                    $dimensions{$header[$i]} = $product_id2label{$model};
                } elsif ($header[$i] eq 'complex_block_id') {
                    # Сервер МОЛ отдает строки вида video_an_site_instream-189455-10, context_on_site_rtb-99551-8
                    my ($model, $page_id_block_id) = $element->[$i] =~ /^(.*)-([0-9]+-[0-9]+)\z/;

                    if ($model) {
                        $product_id2prefix{$model} //= $self->app->$model->public_id_prefix();
                        $dimensions{$header[$i]} = $product_id2prefix{$model} . $page_id_block_id;
                    } else {
                        INFOF "No accessor in complex_block_id '%s'", $element->[$i];
                        $dimensions{$header[$i]} = $page_id_block_id;
                    }
                } else {
                    if ($values_by_positions{$i} && $values_by_positions{$i}{$element->[$i]}) {
                        $dimensions{$header[$i]} = $values_by_positions{$i}{$element->[$i]};
                    } else {
                        $dimensions{$header[$i]} = $element->[$i];
                    }
                }
            } else {

                my ($t) = ($mol_header[$i] =~ /$COMPARE_SPECIFIC_FIELDS_RE/);
                if ($MOL_FIELDS{$stat_type}->{$t}{'is_float'}) {
                    $element->[$i] = $self->_get_float_from_mol_number($element->[$i]);
                }

                $measures{$header[$i]} = $element->[$i];
            }
        }

        if (exists $dimensions{date_a} && exists $dimensions{date_b}) {
            $dimensions{date} = [delete $dimensions{date_a}, delete $dimensions{date_b},];
        }

        push @$points,
          {
            dimensions => \%dimensions,
            measures   => $is_comparing_periods
            ? [_split_hash_by_key_postfix($COMPARE_SPECIFIC_FIELDS_RE, %measures)]
            : [\%measures],
          };
    }

    my $totals;
    foreach my $mol_id (keys(%{$mol_response->{'totals'}})) {

        my ($mol_field, $postfix) = ($mol_id =~ /$COMPARE_SPECIFIC_FIELDS_RE/);
        my $partner2_id = $MOL_TO_PARTNER2{$stat_type}->{$mol_field};
        my $value;

        if ($MOL_FIELDS{$stat_type}->{$mol_field}{'is_float'}) {
            $value = $self->_get_float_from_mol_number($mol_response->{'totals'}->{$mol_id});
        } elsif ($PARTNER2_FIELDS{$stat_type}{$partner2_id}->{'json_type'} eq 'number') {
            $value = ($mol_response->{'totals'}->{$mol_id} // 0) + 0;
        } else {
            $value = $mol_response->{'totals'}->{$mol_id} . '';
        }

        $totals->{$partner2_id . ($postfix ? "_$postfix" : '')} = $value;
    }

    my $dates_period =
      _is_comparing_periods($dates->[0])
      ? join(' - ', map {format_date($_, gettext('%d.%m.%Y'), iformat => 'date_only_numbers')} @{$dates->[0]}) . ' / '
      . join(' - ', map {format_date($_, gettext('%d.%m.%Y'), iformat => 'date_only_numbers')} @{$dates->[1]})
      : join(' - ', map {format_date($_, gettext('%d.%m.%Y'), iformat => 'date_only_numbers')} @$dates);

    foreach my $date (@$dates) {
        if (_is_comparing_periods($dates->[0])) {
            $date = [map {format_date($_, '%Y-%m-%d', iformat => 'date_only_numbers')} @$date];
        } else {
            $date = format_date($date, '%Y-%m-%d', iformat => 'date_only_numbers');
        }
    }

    my $currencies = [
        map {$_->{id} = "$_->{id}"; $_;} @{
            $self->app->currency->get_all(
                fields => ['id', 'code'],
                filter   => [code => IN => $STAT_AVAILABLE_CURRENCIES],
                order_by => ['id']
            )
          }
    ];

    my $answer = {
        points     => $points,
        dimensions => $dimensions,
        measures   => $measures,
        totals     => {
            2 => $is_comparing_periods
            ? [_split_hash_by_key_postfix($COMPARE_SPECIFIC_FIELDS_RE, %$totals)]
            : [$totals]
        },
        periods => _is_comparing_periods($dates->[0]) ? $dates : [$dates],
        report_title => gettext('Report for period %s', $dates_period),
        is_last_page => (
            defined($limit)
            ? (
                $self->_is_last_page(
                    limit      => $limit,
                    offset     => $offset,
                    total_rows => $mol_response->{total_rows},
                  ) ? JSON::XS::true : JSON::XS::false
              )
            : JSON::XS::true
        ),
        currencies => $currencies,
        total_rows => $mol_response->{total_rows},
    };

    return $answer;
}

=begin comment _get_additional_where_from_mol_answer

Возвращате ARRAYREF cо структйой дання для поля filters_pre МОЛ.

=end comment

=cut

sub _get_additional_where_from_mol_answer {
    my ($self, $mol_answer, $stat_type) = @_;

    my %all_group_by_fields =
      map {$_->{'mol_id'} => 1} grep {$_->{'group_by'} || $_->{'_group_by'}} $self->_get_fields($stat_type);

    my $i = 0;
    my %ids2positions = map {$_ => $i++} @{$mol_answer->{'header'}};

    my @field_ids       = grep {$all_group_by_fields{$_}} @{$mol_answer->{'header'}};
    my @field_positions = map  {$ids2positions{$_}} @field_ids;

    my $where = [];

    foreach my $element (@{$mol_answer->{'data'}}) {

        my $tmp = {};

        my $i = 0;
        foreach my $f (@field_ids) {
            my $value;

            $value = $element->[$field_positions[$i]];

            $tmp->{$f} = {eq => [$value]};
            $i++;
        }
        push @$where, $tmp;
    }

    return $where;
}

sub _get_cur_user_lang {
    my ($self) = @_;

    my $lang = $self->get_option('locale');

    if (!defined($lang) || !in_array($lang, [qw(ru en)])) {
        throw "Lang is not known: " . ($lang // 'undef');
    }

    return $lang;
}

=begin comment _get_dates_from_period

На входе - данные в формате get_statitsitcs, на выходе - данные в формате
для мастера отчетов.

На входе может быть либо массив с датами вида:

    [
        '2016-07-01',
        '2016-07-03',
    ]

Либо строка '7days', 'yesterday', 'pastyear' и т.д.

На выходе всегда массив с датами вида:

    [
        '20160701',
        '20160703',
    ]

=end comment

=cut

sub _get_dates_from_period {
    my ($self, $period) = @_;

    my ($date_from, $date_to);

    if (
        (ref($period) eq '')
        && (
            in_array(
                $period,
                [
                    qw(
                      10days
                      14days
                      30days
                      365days
                      7days
                      90days
                      lastmonth
                      lastweek
                      pastyear
                      thismonth
                      thisweek
                      thisyear
                      today
                      yesterday
                      )
                ]
            )
           )
       )
    {
        ($date_from, $date_to) = name2dates($period, '', '', iformat => 'db', oformat => 'db');
    } elsif (ref($period) eq 'ARRAY') {
        ($date_from, $date_to) = @$period;
    } else {
        throw Exception::Validation::BadArguments gettext("Incorrect period '%s'", $period);
    }

    my @r;
    unless (_is_comparing_periods($date_from)) {
        for ($date_from, $date_to) {
            (my $d = $_) =~ s/[^0-9]//g;
            push @r, $d;
        }
    } else {
        for ($date_from, $date_to) {
            push @r, [map {(my $d = $_) =~ s/[^0-9]//g; $d;} @$_];
        }
    }

    return @r;
}

sub _get_empty_get_statistics2_answer {
    my ($self, %opts) = @_;

    $opts{mol_response} = {
        header     => [],
        total_rows => 0,
    };
    my $answer = $self->_convert_mol_response_to_get_statistics2(%opts);

    return $answer;
}

=begin comment _get_fields

Возвращает ARRAY с теми элементами массива @FIELDS, которые доступны
текущему пользователю.

=end comment

=cut

sub _get_fields {
    my ($self, $stat_type) = @_;

    $stat_type //= $MOL_REPORT_TYPE_MAIN;

    my $user_id   = $self->cur_user->get_cur_user_id();
    my $cache_key = $self->get_option('locale') . '_' . $user_id . '_' . $stat_type;

    my $fields = $CACHE->get($cache_key);

    if (!defined $fields) {
        no strict 'refs';

        $fields = [];
        foreach my $f (@FIELDS) {

            unless (grep {$_ eq $stat_type} @{$f->{'report_types'}}) {
                next;
            }

            my $right = 'bk_statistics_view_field__' . $f->{partner2_id};
            my $is_available = $f->{'sub_is_available'} ? $f->{'sub_is_available'}->($self) : FALSE;
            if (   $f->{'available_for_everyone'}
                || $self->check_rights($right)
                || $is_available)
            {
                my %f;
                for my $key (keys %$f) {
                    if (ref($f->{$key}) eq 'CODE') {
                        $f{$key} = $f->{$key}();
                    } else {
                        $f{$key} = $f->{$key};
                    }
                }
                push @$fields, \%f;
            }
        }

        $CACHE->set($cache_key, $fields, "10 minutes");
    }

    return @{$fields};
}

sub _get_geobase_values {
    my ($self) = @_;
    return $self->geo_base->get_geobase(user_id => $self->cur_user->get_cur_user_id());
}

sub _get_float_from_mol_number {
    my ($self, $number) = @_;

    $number //= 0;

    my $float = sprintf('%0.2f', $number / 1_000_000);

    # Для того чтобы frontend получал тип "число" в json
    $float += 0;

    return $float;
}

sub _get_product_id_values {
    my ($self) = @_;

    my @values;
    my $app = $self->app;
    foreach my $model (sort @{$app->product_manager->get_accessors_by_tag('has_bk_stat')}) {
        my $app_model = $app->$model;
        if ($app_model->check_short_rights('view_all') || $app_model->check_short_rights('view')) {
            push @values,
              {
                label     => $app_model->get_product_name(),
                id        => $model,
                key       => $model,
                parent_id => '',
              };
        }
    }

    return \@values;
}

sub _get_block_model_values {
    my ($self) = @_;

    my @values;
    my $app = $self->app;
    foreach my $model (sort @{$self->app->product_manager->get_accessors_by_tag('has_bk_stat')}) {
        my $app_model = $app->$model;
        for my $block_model (sort @{$app_model->get_block_model_names}) {
            if ($app_model->check_short_rights('view_all') || $app_model->check_short_rights('view')) {
                push @values,
                  {
                    label     => $app->$block_model->get_product_name(),
                    id        => $block_model,
                    key       => $block_model,
                    parent_id => $model,
                  };
            }
        }
    }

    return [sort {$a->{label} cmp $b->{label}} @values];
}

sub _get_block_type_values {
    my ($self) = @_;

    my @values = (@{$self->_get_common_block_type_values}, @{$self->_get_video_block_type_values});

    return \@values;
}

sub _get_common_block_type_values {
    my ($self) = @_;

    my @values;

    push @values, map {{label => $_->{label_stat}->(), id => "app-$_->{id}", key => "app-$_->{id}", parent_id => '',}}
      sort {$a->{id} cmp $b->{id}} @BLOCK_TYPES;

    return \@values;
}

sub _get_video_block_type_values {
    my ($self) = @_;

    my @values;

    push @values,
      map {{label => $_->{label_stat}->(), id => "video-$_->{bk}", key => "video-$_->{bk}", parent_id => '',}}
      sort {$a->{bk} cmp $b->{bk}} values %$VIDEO_BLOCK_TYPES;

    return \@values;
}

sub _get_site_version_values {
    my ($self) = @_;

    my @values;
    push @values, map {
        {
            label => $SITE_VERSIONS->{$_}->{label_stat}
            ? $SITE_VERSIONS->{$_}->{label_stat}->()
            : $SITE_VERSIONS->{$_}->{label}->(),
            id        => $_,
            key       => $_,
            parent_id => '',
        }
      }
      sort keys %$SITE_VERSIONS;
    return \@values;
}

=begin comment _get_tree2__entity_fields

Возваращет ARRAYREF, который содержит HASHREF.

Пример возвращаемых значений:

    [
        {
            id => 'client_id',
            index => 1,
            label => 'Client ID',
            type => 'publicid',
        },
        {
            id => 'login',
            index => 2,
            label => 'Логин',
            type => 'login',
        },
    ]

=end comment

=cut

sub _get_tree2__entity_fields {
    my ($self, $stat_type, %opts) = @_;

    my $data = [
        map {
            {
                (
                    $_->{'frontend_category'}
                    ? (
                        category => $_->{'frontend_category'} + 0,
                        category_name =>
                          ($CATEGORY_NAMES->{$_->{'frontend_category'}} // sub {$_->{'frontend_category'}})->(),
                      )
                    : ()
                ),
                  id    => $_->{'partner2_id'},
                  label => $_->{'label'},
                  ($_->{'hint'} ? (hint => $_->{'hint'}) : ()),
                  type  => $_->{'get_tree2_type'},
                  index => $_->{'index'},
                  ($_->{currency} ? (currency => $_->{currency}) : ()),
            }
          }
          grep {
            $_->{'group_by'} && !($opts{for_tree} && $_->{hide_in_tree})
          } $self->_get_fields($stat_type)
    ];

    return $data;
}

=begin comment _get_tree2__entity_filter_fields

Возвращает HASHREF с данными.

Пример возвращаемого значения:

    {
        client_id => {
            index => 1,
            type => 'publicid',
        },
        dsp_id => {
            index => 4,
            type => 'publicid',
        },
        login => {
            index => 2,
            type => 'login',
        },
    }

Список type с которым может работать frontend:

 * number
 * text
 * login - специальный тип, который добавляет саджест по логину
 * multistate - тип с неправильным названием. Изначально он создавался для
   поля multistate и поэтому был так назван. На самом деле этот тип подходит
   для любого поля значнеие которого — это хеш.
 * geo

=end comment

=cut

sub _get_tree2__entity_filter_fields {
    my ($self, $stat_type, %opts) = @_;

    my $data = {};

    foreach my $element ($self->_get_fields($stat_type)) {
        next unless $element->{'where'} && !($opts{for_tree} && $element->{hide_in_tree});

        my $values;

        if (exists($element->{'get_tree2_values_sub'})) {
            my $sub_name = $element->{'get_tree2_values_sub'};
            $values = $self->$sub_name();
        } else {
            $values = $element->{'get_tree2_values'};
        }

        $data->{$element->{'partner2_id'}} = {
            type  => $element->{'get_tree2_type'},
            index => $element->{'index'},
            (defined($values) ? (values => $values) : ()),
            ($element->{currency} ? (currency => $element->{currency}) : ()),
        };
    }

    return $data;
}

=begin comment _get_tree2__entity_filter_simple_fields

Возвращает ARRAYREF с одним элементом ARRAYREF в котором содержатся HASHREF.

Прмер возвращаемоего значения:

    [
        [
            {
                index => 1,
                label => 'Client ID',
                name => 'client_id',
            },
            {
                index => 2,
                label => 'Логин',
                name => 'login',
            },
        ]
    ]

=end comment

=cut

sub _get_tree2__entity_filter_simple_fields {
    my ($self, $stat_type, %opts) = @_;

    my $data = [
        [
            map {
                {
                    (
                        $_->{'frontend_category'}
                        ? (
                            category => $_->{'frontend_category'} + 0,
                            category_name =>
                              ($CATEGORY_NAMES->{$_->{'frontend_category'}} // sub {$_->{'frontend_category'}})->(),
                          )
                        : ()
                    ),
                      name  => $_->{'partner2_id'},
                      label => $_->{'label'},
                      ($_->{'hint'} ? (hint => $_->{'hint'}) : ()),
                      index => $_->{'index'},
                      ($_->{currency} ? (currency => $_->{currency}) : ()),
                }
              }
              grep {
                $_->{'where'} && !($opts{for_tree} && $_->{hide_in_tree})
              } $self->_get_fields($stat_type)
        ]
    ];

    return $data;
}

=begin comment _get_tree2__fields

Возаращает ARRAYREF, который содержит набор HASHREF.

Пример возвращаемого значения:

    [
        {
            category => 1,
            id => 'rtb_block_shows',
            index => 21,
            title => 'Показы RTB-блоков',
            type => 'text',
        },
        {
            category => 1,
            id => 'rtb_block_hits_own_adv',
            index => 22,
            title => 'Запросы своей рекламы в RTB-блоках',
            type => 'text',
        },
    ]

=end comment

=cut

sub _get_tree2__fields {
    my ($self, $stat_type, %opts) = @_;

    my $data = [
        map {
            my $c = $_->{'frontend_category'} // $CATEGORY_MOL_BASIC;
            my $cn = ($CATEGORY_NAMES->{$c} // sub {$c})->();
            {
                category      => $c + 0,
                category_name => $cn,
                id            => $_->{'partner2_id'},
                title         => $_->{'label'},
                ($_->{'hint'} ? (hint => $_->{'hint'}) : ()),
                type  => $_->{'get_tree2_type'},
                index => $_->{'index'},
                ($_->{currency}      ? (currency => $_->{currency})      : ()),
                ($_->{frontend_unit} ? (unit     => $_->{frontend_unit}) : ()),
            }
          }
          grep {
            $_->{'select'} && !($opts{for_tree} && $_->{hide_in_tree})
          } $self->_get_fields($stat_type)
    ];

    return $data;
}

sub _safe_get_data {
    my ($self, %hash) = @_;

    my $mol_answer;

    try {
        $mol_answer = $self->api_http_mol->get_data(%hash);
    }
    catch Exception::API::HTTPMOL::TooManyGroupBy with {
        throw Exception::Validation::BadArguments gettext('Too many fields in group by');
    };

    return $mol_answer;
}

=begin comment _is_last_page

    my $bool = $self->_is_last_page(
        limit => 50,
        offset => 0,
        total_rows => 860,
    );

=end comment

=cut

sub _is_last_page {
    my ($self, %opts) = @_;

    return $opts{total_rows} <= ($opts{limit} + $opts{offset});
}

sub _get_date_groups {
    my ($self) = @_;
    return $self->_get_groups('date');
}

sub _get_region_levels {
    my ($self) = @_;
    return $self->_get_groups('geo');
}

sub _get_groups {
    my ($self, $dimension) = @_;

    my $BY_DIMENSION_GROUPS = {
        date => {
            day   => gettext('by day'),
            week  => gettext('by week'),
            month => gettext('by month'),
            year  => gettext('by year'),
        },
        geo => {
            country => gettext('by county'),
            area    => gettext('by area'),
            city    => gettext('by city'),
        }
    };
    return $BY_DIMENSION_GROUPS->{$dimension};
}

sub _get_group {
    my ($self, $dimension_fields, $field, $groups) = @_;

    return undef if ref($dimension_fields) ne 'ARRAY';
    return undef if @$dimension_fields == 0;

    my $group;

    my $regexp = join('|', keys(%{$groups}));

    foreach my $el (@$dimension_fields) {
        if (ref($el) eq '' && $el =~ /^$field\|($regexp)\z/) {
            $group = $1;
            last;
        }
    }

    return $group;
}

sub _recommendation_widget_available {
    return $_[0]->check_rights('context_on_site_content_edit')
      || $_[0]->check_rights('internal_context_on_site_content_edit');
}

sub _dsp_fields_available {
    $_[0]->check_rights('bk_statistics_view_dsp');
}

sub _user_without_new_mobmed_feature {
    my ($self) = @_;
    return !$self->users->has_feature($self->get_option('cur_user', {})->{id}, 'new_mobmed');
}

sub _is_shows_direct_banners_available {
    my ($self) = @_;

    return TRUE if $self->check_rights('bk_statistics_view_field__shows_direct_banners');

    my $cur_user_id = $self->get_option('cur_user')->{id};
    if ($self->check_rights('is_assistant')) {
        my $search_on_site_campaign_assistant_query = $self->partner_db->query->select(
            table  => $self->partner_db->search_on_site_campaign,
            fields => [qw(page_id)],
          )->join(
            table   => $self->partner_db->assistants,
            fields  => [qw(user_id)],
            filter  => ['AND', [['user_id', '=', \$cur_user_id]]],
            join_on => [page_id => '=' => {page_id => $self->partner_db->search_on_site_campaign}],
          );
        return TRUE if scalar @{$search_on_site_campaign_assistant_query->get_all()};
    }
    my $search_on_site_campaign_owner_query = $self->partner_db->query->select(
        table  => $self->partner_db->search_on_site_campaign,
        fields => [qw(page_id)],
        filter => ['AND', [['owner_id', '=', \$cur_user_id]]],
    );
    return TRUE if scalar @{$search_on_site_campaign_owner_query->get_all()};

    return FALSE;
}

sub _old_statistics_available {
    return $_[0]->rbac->has_role_from_group_dev([keys %{$_[0]->rbac->get_cur_user_roles()}])
      || !$_[0]->check_rights('statistics_mol_view');
}

sub _cur_user_is_intapi {
    my ($self) = @_;
    my $cur_user = $self->get_option('cur_user', {});
    return $cur_user->{id} && ($YNDX_PARTNER_INTAPI_USER_ID eq $cur_user->{id});
}

sub _is_comparing_periods {
    my ($date) = @_;
    return 'ARRAY' eq ref($date);
}

sub _validate_comparing_periods {
    my ($period) = @_;

    throw Exception::Validation::BadArguments gettext("Wrong period")
      unless $period
          && (ref($period) eq 'ARRAY')
          && @$period == 2
          && (ref($period->[0]) eq 'ARRAY')
          && (ref($period->[1]) eq 'ARRAY')
          && @{$period->[0]} == 2
          && @{$period->[1]} == 2;

    foreach (@{$period->[0]}, @{$period->[1]}) {
        throw Exception::Validation::BadArguments gettext('Wrong date format: "%s", expected format "yyyy-mm-dd"', $_)
          unless $_ && check_date($_, iformat => 'db');
    }

    throw Exception::Validation::BadArguments gettext('Wrong period starts in "%s" and end in "%s"', $period->[0][0],
        $period->[0][1])
      unless $period->[0][0] le $period->[0][1];

    throw Exception::Validation::BadArguments gettext('Wrong period starts in "%s" and end in "%s"', $period->[1][0],
        $period->[1][1])
      unless $period->[1][0] le $period->[1][1];

    my $fst_interval_map  = _get_interval_map_for_period(@{$period->[0]});
    my $snd_interval_map  = _get_interval_map_for_period(@{$period->[1]});
    my $fst_period_length = $fst_interval_map->{'day'}{'end'} - $fst_interval_map->{'day'}{'start'};
    my $snd_period_length = $snd_interval_map->{'day'}{'end'} - $snd_interval_map->{'day'}{'start'};

    throw Exception::Validation::BadArguments gettext("Can't compare periods of different lengths")
      unless $fst_period_length == $snd_period_length;
}

sub _get_interval_map_for_period {
    my ($start, $end) = @_;

    $start = trdate(db_time => sec => $start . ' 00:00:00');
    $end   = trdate(db_time => sec => $end . ' 23:59:59');

    my %interval_map = map {
        (
            $_ => {
                start => trdate($_ => sec => trdate(sec => $_ => $start)),
                end   => trdate($_ => sec => trdate(sec => $_ => $end))
            }
        )
    } qw(week month year);

    $interval_map{day}{start} = $start;
    $interval_map{day}{end}   = $end;

    return \%interval_map;
}

sub _split_hash_by_key_postfix {
    my ($postfix_re, %hash) = @_;

    my @splitted;
    my @postfix = ('_a', '_b');
    for (my $i = 0; $i < 2; $i++) {
        my @keys = grep {$_ =~ /$postfix[$i]$/} keys %hash;
        @{$splitted[$i]}{map {$_ =~ /$postfix_re/; $1;} @keys} = @hash{@keys};
    }
    return @splitted;
}
1;
