package Application::Model::BkPiComparison;

use qbit;

use base qw(QBit::Application::Model);

sub accessor {'devel_bk_pi_comparison'}

__PACKAGE__->model_accessors(
    internal_context_on_site_campaign => 'Application::Model::Product::InternalAN::InternalContextOnSite::Campaign',
    internal_search_on_site_campaign  => 'Application::Model::Product::InternalAN::InternalSearchOnSite::Campaign',
    partner_db                        => 'Application::Model::PartnerDB',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'news',
            description => d_gettext('Rights to use BK PI comparison'),
            rights      => {devel_bk_pi_comparison => d_gettext('Right to use tool BK PI comparison'),},
        }
    ]
);

=head1 get_bk_data_for_comparison

=cut

sub get_bk_data_for_comparison {
    my ($self, %opts) = @_;

    my $type = delete($opts{'type'});

    my $fields;

    if (($type eq 'context') || ($type eq 'search')) {
        $fields = {
            'page_id' => '',
            ($opts{'domain'}               ? ('domain'               => '') : ()),
            ($opts{'state'}                ? ('state'                => '') : ()),
            ($opts{'opt_readonly'}         ? ('opt_readonly'         => '') : ()),
            ($opts{'opt_dontshowsex'}      ? ('opt_dontshowsex'      => '') : ()),
            ($opts{'opt_dontshowbehavior'} ? ('opt_dontshowbehavior' => '') : ()),
            ($opts{'mirrors'}              ? ('mirrors'              => '') : ()),
            ($opts{'excludeddomains'}      ? ('excludeddomains'      => '') : ()),
        };
    } else {
        throw "Error. Unknown type.";
    }

    my $data = $self->partner_db->pagesbs->get_all(
        fields => $fields,
        filter => {page_id => $opts{'page_ids'},},
    );

    foreach (@$data) {
        if (defined($_->{'mirrors'}) && length($_->{'mirrors'}) > 0) {
            $_->{'mirrors'} = join(" ", sort(split(',', $_->{'mirrors'})));
        }

        if (defined($_->{'excludeddomains'}) && length($_->{'excludeddomains'}) > 0) {
            $_->{'excludeddomains'} = join(" ", sort(split(',', $_->{'excludeddomains'})));
        }
    }

    my $data2 = [
        map {
            {$_->{'page_id'} => $_}
          } @$data
    ];

    return $data2;
}

=head1 get_comparison_diff

=cut

sub get_comparison_diff {
    my ($self, %opts) = @_;

    my $result;

    my $bk = delete($opts{'bk'});
    my $pi = delete($opts{'pi'});

    my $pi_hash = {map {my ($page_id) = keys %$_; $page_id => $_->{$page_id}} @$pi};
    my $bk_hash = {map {my ($page_id) = keys %$_; $page_id => $_->{$page_id}} @$bk};

    # Возможна ситуация, когда в БК нет данных о площадке
    # (напирмер, площадку только что добавили и мы еще не получили из БК
    # данные для сверки)
    # Поэтому мы считаем нормальным ситуацию, когда в ПИ есть данные о
    # площадке, а в БК нет.
    # Но ситуация, когда в БК есть данные о площадки, а в ПИ нет, в данной
    # реализации является ошибкой.
    die 'Incorrect data' if @$bk > @$pi;

    foreach my $page_id (sort {$a <=> $b} keys %$pi_hash) {

        my $bk_element = $bk_hash->{$page_id};
        my $pi_element = $pi_hash->{$page_id};

        if (defined($bk_element)) {
            die 'Incorrect data' if join('', sort(keys(%$bk_element))) ne join('', sort(keys(%$pi_element)));
        }

        foreach my $key (sort(keys(%$pi_element))) {
            if (!defined($bk_element->{$key})) {
                $bk_element->{$key} = '';
            }
            if ($bk_element->{$key} ne $pi_element->{$key}) {
                push @$result,
                  {
                    page_id => $page_id,
                    bk      => "$key: $bk_element->{$key}",
                    pi      => "$key: $pi_element->{$key}",
                  };
            }
        }
    }

    return $result;
}

=head1 get_diff

=cut

sub get_diff {
    my ($self, %opts) = @_;

    my %opts2 = (
        type            => $opts{'type'},
        state           => $opts{'state'},
        domain          => $opts{'domain'},
        opt_readonly    => $opts{'opt_readonly'},
        opt_dontshowsex => $opts{'opt_dontshowsex'},
        mirrors         => $opts{'mirrors'},
        excludeddomains => $opts{'excludeddomains'},

        # opt_dontshowbehavior не используется на поисковых площадках
        (
            $opts{'type'} eq 'context'
            ? (opt_dontshowbehavior => $opts{'opt_dontshowbehavior'},)
            : ()
        ),
    );

    my $pi_data = $self->get_pi_data_for_comparison(%opts2,);

    my @page_ids = $self->get_page_ids_from_comparison($pi_data);

    my $bk_data = $self->get_bk_data_for_comparison(
        page_ids => \@page_ids,
        %opts2,
    );

    my $diff = $self->get_comparison_diff(
        pi => $pi_data,
        bk => $bk_data,
    );

    return $diff;
}

=head1 get_page_ids_from_comparison

=cut

sub get_page_ids_from_comparison {
    my ($self, $comparison_data) = @_;

    my @page_ids;

    foreach my $tmp (@$comparison_data) {
        my ($page_id) = keys %$tmp;
        push @page_ids, $page_id;
    }

    return @page_ids;
}

=head1 get_pi_data_for_comparison

=cut

sub get_pi_data_for_comparison {
    my ($self, %opts) = @_;

    my $data;

    my $type = delete($opts{'type'});

    if ($type eq 'context') {

        $data = $self->internal_context_on_site_campaign->get_all(
            fields => [sort keys(%{$self->internal_context_on_site_campaign->get_model_fields()})],);

    } elsif ($type eq 'search') {

        $data = $self->internal_search_on_site_campaign->get_all(
            fields => [sort keys(%{$self->internal_search_on_site_campaign->get_model_fields()})],);

    } else {
        throw "Error. Unknown type.";
    }

    foreach (@$data) {

        $_->{'page_id'} = $_->{'id'};

        if ($opts{'state'}) {
            $_->{'state'} = $_->{'bk_state_id'};
        }

        if ($opts{'opt_readonly'}) {
            $_->{'opt_readonly'} = $_->{'is_read_only'} ? 1 : 0;
        }

        if ($opts{'opt_dontshowsex'}) {
            $_->{'opt_dontshowsex'} = $_->{'family_filter'};
        }

        if ($opts{'opt_dontshowbehavior'}) {
            $_->{'opt_dontshowbehavior'} = $_->{'behavioral_targeting'} ? 0 : 1;
        }

        if ($opts{'excludeddomains'}) {
            $_->{'excludeddomains'} = join(
                " ",
                sort((map {get_domain($_, ascii => TRUE)} @{$_->{'excluded_domains'}}),
                    (map {s/[^0-9]//g; "$_.phone"} @{$_->{'excluded_phones'}}),
                    $_->{'domain'},
                    @{$_->{'mirrors'}},
                )
            );
        }

        if ($opts{'mirrors'}) {
            $_->{'mirrors'} = join(" ", sort(@{$_->{'mirrors'}}));
        }

        foreach my $field (keys(%$_)) {
            next if $field eq 'page_id';

            delete($_->{$field}) unless $opts{$field};
        }
    }

    my $data2 = [
        map {
            {$_->{'page_id'} => $_}
          } @$data
    ];
    return $data2;
}

1;
