package Application::Model::Block;

=encoding UTF-8

=cut

=head1 DESCRIPTION

Базовый класс для всех блоков.

=cut

use qbit;

use base qw(Application::Model::Product);

use QBit::Application::Model::DBManager::Filter::text;

use PiConstants qw(
  $TECHNICAL_RTB_BLOCK_ID
  $GOMAILRU_ADS_USER_ID
  $M_RU_2015_USER_ID
  $M_RU_TEXT_USER_ID
  $RAMBLER_P_USER_ID
  $VK_RSYA_APPS_USER_ID
  $VK_RSYA_USER_ID
  $WWWODNOKLASSNIKI_2015_USER_ID
  $WWWODNOKLASSNIKI_USER_ID
  $INTERNAL_DSP_TAGS
  $DSP_AWAPS_ID
  $DSP_DIRECT_ID
  $DSP_OWN_ADV_ID
  $DSP_UNSOLD_ID
  @VIDEO_DEFAULT_DSPS
  $LIMIT_CAMPAIGN_FOR_ADDING
  $ADINSIDE_USER_ID
  $DESIGN_TYPES
  $MAX_REVENUE_STRATEGY_ID
  $MIN_CPM_STRATEGY_ID
  $SEPARATE_CPM_STRATEGY_ID
  );

use Exception::Validator::Fields;
use Exception::Validation::BadArguments;

use Partner::DSP::Rule;
use Partner::DSP::Rule::Part;
use Partner::DSP::Rule::Filter;
use Partner::DSP::RuleSet;

__PACKAGE__->abstract_methods(
    qw(get_page_id_field_name get_campaign_model_name get_need_update_in_bk_fields get_statistics_level));

sub get_campaign_model {
    my ($self) = @_;

    my $page_model_name = $self->get_campaign_model_name();
    my $page_model      = $self->app->$page_model_name;

    return $page_model;
}

=head2 public_id

L<https://wiki.yandex-team.ru/partner/w/partner2-public-id/>

=cut

sub public_id {
    my ($self, $id) = @_;

    my $prefix = $self->public_id_prefix();

    return "$prefix$id->{$self->get_page_id_field_name()}-$id->{'id'}";
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        partner_db       => 'Application::Model::PartnerDB',
        rbac             => 'Application::Model::RBAC',
        statistics       => 'Application::Model::Statistics',
        page             => 'QBit::Application::Model',
        tns_dict_brand   => 'Application::Model::TNSDict::Brand',
        tns_dict_article => 'Application::Model::TNSDict::Article',
        geo_base         => 'Application::Model::GeoBase',
        dsp              => 'Application::Model::DSP',
        all_pages        => 'Application::Model::AllPages',
        product_manager  => 'Application::Model::ProductManager',
        users            => 'Application::Model::Users',
        design_templates => 'Application::Model::DesignTemplates',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        #@{$self->SUPER::get_structure_rights_to_register()},
        {
            name        => $self->accessor(),
            description => d_gettext('Right to manage ' . $self->accessor()),
            rights      => {
                (
                    map {$self->get_description_right($_)}
                      qw(
                      view
                      view_all
                      view_action_log
                      edit
                      ),
                ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $FIELDS_DEPENDS;

    return {
        # Оverride any of these fields by a constant in blocks that don't need them.
        #
        page_id => {
            # To be overridden in some child blocks while we have different field names for page_id
            default => TRUE,
            db      => TRUE,
            (
                $self->get_page_id_field_name() ne 'page_id'
                ? (db_expr => $self->get_page_id_field_name())
                : (pk => TRUE,)
            ),
            label      => d_gettext('Page ID'),
            hint       => d_gettext('The unique page identifier (as in BK)'),
            type       => 'number',
            api        => 1,
            need_check => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $page_id) = @_;
                    $qv->app->validate_page_reachable($page_id);
                  }
            },
            adjust_type => 'str',
        },
        id => {
            default     => TRUE,
            db          => TRUE,
            pk          => TRUE,
            label       => d_gettext('Block ID (raw)'),
            type        => 'number',
            need_check  => {type => 'int_un',},
            api         => 1,
            adjust_type => 'str',
        },
        public_id => {
            depends_on => [qw(page_id id)],
            label      => d_gettext('Block ID'),
            get        => sub {
                $_[0]->model->public_id_prefix() . "$_[1]->{'page_id'}-$_[1]->{'id'}";
            },
        },
        page => {
            depends_on => ['page_id', 'pages.id', 'pages.multistate', 'pages.owner_id'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}};
            },
            type => $self->get_campaign_model_name(),
        },
        # NOTE! В события кладутся внутренние ID площадок (#PI-7985)
        internal_campaign_id => {
            depends_on => ['page_id', 'pages.id'],
            label      => d_gettext('Internal Campaign ID'),
            get        => sub {
                my ($self, $obj) = @_;

                return $self->{'pages'}{$obj->{'page_id'}}{'id'};
            },
            type => 'number',
        },
        caption => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Block caption'),
            type       => 'string',
            need_check => {len_min => 1, len_max => 255,},
            need_trim  => TRUE,
            api        => 1,
        },
        multistate => {
            db          => TRUE,
            label       => d_gettext('Status'),
            type        => 'number',
            need_check  => {type => 'int_un'},
            api         => 1,
            adjust_type => 'str',
        },
        multistate_name => {
            depends_on => ['multistate'],
            label      => d_gettext('Multistate name'),
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1,
        },
        is_deleted => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'deleted');
            },
            type => 'boolean',
            api  => 1,
        },
        is_protected => {
            depends_on => ['page_id', 'pages.is_protected'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'is_protected'};
            },
            type => 'boolean',
        },
        is_updating => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'need_update')
                  || $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'updating');
            },
            type => 'boolean',
            api  => 1,
        },
        statistics_exists => {
            depends_on => [qw(multistate)],
            label      => d_gettext('Statistics exists'),
            get        => sub {
                return !!1;
            },
            type => 'boolean',
            api  => 1,
        },
        is_page_deleted => {
            depends_on => ['page_id', 'pages.multistate'],
            get        => sub {
                return $_[0]
                  ->model->page->check_multistate_flag($_[0]->{'pages'}{$_[1]->{'page_id'}}{'multistate'}, 'deleted');
            },
            type => 'boolean',
            api  => 1
        },
        comment => {
            db           => TRUE,
            label        => d_gettext('Block\'s comment'),
            check_rights => $self->get_right('view_field__comment'),
            type         => 'string',
            need_check   => {optional => TRUE,},
            api          => 1,
            need_trim    => TRUE,
        },
        status => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'need_update')
                  || $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'updating') ? 'pending' : 'sync';
            },
            type => 'string',
            api  => 1,
        },
        fields_depends => {
            get => sub {
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
            api  => 1,
        }
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $accessor         = $self->accessor();
    my $page_id_name     = $self->get_page_id_field_name();
    my $public_id_prefix = $self->public_id_prefix();

    return {
        db_accessor => 'partner_db',
        fields      => {
            id => {
                type   => 'publicid',
                label  => d_gettext('Block ID'),
                regexp => {
                    '^([0-9]+)\z' => [qw(id)],
                    '^' . $public_id_prefix . '([0-9]+)-([0-9]+)\z' => [$page_id_name, 'id'],
                    '^([0-9]+)-([0-9]+)\z'                          => [$page_id_name, 'id'],
                }
            },
            page_id => {
                type      => 'number',
                label     => d_gettext('Page ID'),
                db_filter => sub {
                    return [$page_id_name => $_[1]->[1] => \$_[1]->[2]];
                },
            },
            multistate => {type => 'multistate', label => d_gettext('Status')},
            caption    => {type => 'text',       label => d_gettext('Block\'s caption')},
            resource   => {
                type      => 'dictionary',
                label     => d_gettext('Resource'),
                db_filter => sub {
                    my ($model, $filter, $filter_obj, %opts) = @_;

                    my $type = ref($filter->[2]);

                    if (defined($filter->[2]) && (($type eq 'ARRAY' && @{$filter->[2]}) || !$type)) {
                        return [\$accessor => $filter->[1] => \$filter->[2]];
                    } else {
                        my %opr = ('=' => undef, 'IS' => undef, '<>' => 1, 'IS NOT' => 1);

                        return \$opr{$filter->[1]};
                    }
                },
                values => sub {
                    my ($model) = @_;

                    return [
                        map {{id => $_, label => $model->app->$_->get_product_name}}
                        sort @{$model->app->product_manager->get_block_model_names()}
                    ];
                  }
            },
            public_id => {
                type      => 'text',
                db_filter => sub {
                    my ($model, $filter, $filter_obj, %opts) = @_;

                    return [
                        {CONCAT => [\$public_id_prefix, $page_id_name, \'-', 'id']},
                        $filter->[1],
                        \(
                            $filter->[1] =~ /LIKE$/
                            ? QBit::Application::Model::DBManager::Filter::text::__like_str($filter->[2])
                            : $filter->[2]
                         )
                    ];
                },
            },
            page => {
                type           => 'subfilter',
                model_accessor => 'page',
                field          => $page_id_name,
                fk_field       => $self->page->get_page_id_field_name(),
                label          => d_gettext('Campaign'),
            },
        },
    };
}

# TODO: использовать базовый
sub get_structure_multistates_graph {
    my ($self) = @_;

    return {};
}

sub get_editable_fields {
    my ($self, $data) = @_;

    return {} unless $self->check_action($data, 'edit');

    my %res = ();

    if (   $self->DOES('Application::Model::Role::Has::CustomBkData')
        && $self->check_short_rights('edit_field__bk_data'))
    {

        $res{'is_custom_bk_data'} = TRUE;

        if ($data->{'is_custom_bk_data'}) {
            $res{'bk_data'} = TRUE;

            return \%res;
        }
    }

    %res = (%res, %{$self->collect_editable_fields($data)});

    return \%res;
}

sub is_mobile_block {
    my ($self) = @_;
    return ($self->accessor() eq 'mobile_app_rtb' || $self->accessor() eq 'internal_mobile_app_rtb');
}

sub can_action_delete {return TRUE;}

sub on_action_delete {
    my ($self, $block, %opts) = @_;

    $self->maybe_do_action($block, 'stop');

    $self->update_in_bk($block) unless $opts{'do_not_update_in_bk'};

    return TRUE;
}

sub duplicate {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {public_id => {type => 'publicid'}}
        },
        app   => $self,
        throw => TRUE
    );

    my $result = $self->do_action_with_result($opts{'public_id'}, 'duplicate');

    return $result->{'public_id'};
}

sub can_action_restore {
    my ($self, $block) = @_;

    my $page = $self->_get_object_fields($block, [qw(page)])->{'page'};

    return $self->page->check_action($page, 'restore_block');
}

sub on_action_restore_with_page {
    my ($self, $block) = @_;

    $self->maybe_do_action($block, 'start');

    return TRUE;
}

sub on_action_set_need_update {
    my ($self, $block) = @_;

    my $page = $self->page;

    my $tmp_rights = $self->app->add_tmp_rights($page->get_rights_by_actions('set_need_update'));

    my $fields = $self->_get_object_fields($block, ['internal_campaign_id']);    # page primary key

    $page->do_action($fields->{'internal_campaign_id'}, 'set_need_update')
      if ($page->check_action($fields->{'internal_campaign_id'}, 'set_need_update'));
}

sub can_action_start {
    my ($self, $block) = @_;

    my $page = $self->_get_object_fields($block, [qw(page)])->{'page'};

    return $self->page->check_action($page, 'start_block');
}

sub get_bk_data {
    my ($self, $page_id, @opts) = @_;

    my %opts = @opts unless @opts % 2;

    my $tmp_rights = $self->app->add_all_tmp_rights();    # ??? also set in update_in_bk

    my $page_id_name = $self->get_page_id_field_name();

    my @blocks = @{
        $self->get_all(
            fields => ['*'],
            filter => {$page_id_name => $page_id, multistate => $self->get_multistate_filter_for_update_in_bk()},
            use_java_json_api => FALSE,
        )
      };

    my %res_blocks;

    foreach my $block (@blocks) {
        $self->do_action($block, 'start_update') if !$opts{read_only} and $self->check_action($block, 'start_update');

        next if $block->{'id'} == 0;    # do not put 0-blocks to res_blocks

        my $bk_block = $self->get_bk_block_data($block);
        if ($bk_block) {
            $bk_block->{BlockModel} = $self->accessor();

            # TODO: move to get_bk_block_data in some base block class
            if (exists($bk_block->{'PageImpOptions'})) {
                foreach (qw(Enable Disable)) {
                    $bk_block->{'PageImpOptions'}{$_} = [] if !exists($bk_block->{'PageImpOptions'}{$_});

                    $bk_block->{'PageImpOptions'}{$_} = [sort @{$bk_block->{'PageImpOptions'}{$_}}];
                }
            }
        }

        if ($self->DOES('Application::Model::Role::Has::CustomBkData')) {
            if ($block->{'is_custom_bk_data'}) {
                $bk_block = from_json($block->{'bk_data'});
            } elsif (!$opts{read_only}) {
                $self->partner_db_table()->edit(
                    $self->partner_db->filter(
                        {
                            $page_id_name     => $block->{$page_id_name},
                            id                => $block->{'id'},
                            is_custom_bk_data => 0,
                        }
                    ),
                    {bk_data => to_json($bk_block, pretty => TRUE)}
                );
            }
        }

        $res_blocks{$block->{'id'}} = $bk_block if $bk_block;
    }

    return ($self->get_bk_common_data_wrapper(\@blocks),
        (%res_blocks ? ($self->get_bk_block_data_key() => \%res_blocks) : ()));
}

sub get_bk_common_data_wrapper {
    my ($self, $blocks) = @_;

    return (places => {}) unless @$blocks;

    return $self->get_bk_common_data($blocks);
}

sub get_multistate_filter_for_update_in_bk {'not deleted'}

sub update_in_bk {
    my ($self, $block) = @_;

    # $block here, depending on the context can be:
    #
    # 1. scalar public_id as 'prefix-page_id-block_id'
    # 2. block_id as an ARRAYREF if called from add()
    # 3. block object as a HASHREF if called from inside other actions

    my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('set_need_update'));

    $self->do_action($block, 'set_need_update');
}

sub api_check_public_id {
    my ($self, $id) = @_;

    my $prefix = $self->public_id_prefix();

    return $id =~ /^$prefix(?:\d+)-(?:\d+)$/;
}

sub get_bk_block_data {
    return ();
}

sub get_bk_block_data_key {'rtb_blocks'}

sub get_bk_context_rtb_settings {
    my ($self, $block) = @_;

    my %bk_data;

    $self->_set_direct_limit(\%bk_data, $block);

    $bk_data{'DSPType'} = 1;    #2**0 media

    my $percent;
    ($bk_data{'Width'}, $percent, $bk_data{'Height'}) = $block->{'media_block'} =~ m/^([0-9]+)(%?)x([0-9]+)$/
      if $block->{'media_block'};
    $bk_data{'Width'} = 0 if $percent;

    my $blocks_for_section_sizes = {map {$_ => TRUE} @{$block->{'dsp_blocks'}}};
    $blocks_for_section_sizes->{$block->{'media_block'}} = TRUE
      if $block->{'media_block'};

    my $already_has_0x0;
    my ($width, $height);

    foreach my $type (sort keys %$blocks_for_section_sizes) {
        next unless $type =~ m/^([0-9]+)(%?)x([0-9]+)$/;
        $width = $2 ? 0 : $1;
        $height = $3;

        push(
            @{$bk_data{'Sizes'}},
            {
                Width  => $width,
                Height => $height,
            }
        );
        $already_has_0x0 = TRUE if $width == 0 && $height == 0;
    }

    # For video direct (PI-8892)
    push(
        @{$bk_data{'Sizes'}},
        {
            Width  => 0,
            Height => 0,
        }
    ) unless $already_has_0x0;

    # Технический блок - отправляем доп.поле на уровне блоков: OverrideSizes=1 (#PI-7517, PI-8684)
    $bk_data{'OverrideSizes'} = 1
      if $block->{'id'} == $TECHNICAL_RTB_BLOCK_ID;

    my $strategy = $block->{'strategy'};
    $bk_data{'OptimizeType'} = $strategy == 3 ? 0 : $strategy;

    if (defined($block->{'alternative_code'}))
    {    # should not be in adfox but exists as undef in model (override of RTBMixin)
        $bk_data{'AlternativeCode'} = $block->{'alternative_code'};
        $bk_data{'AltWidth'} = $block->{'alt_width'} || 0
          if exists($block->{'alt_width'});    # not exists in adfox and internal_context_rtb
        $bk_data{'AltHeight'} = $block->{'alt_height'} || 0
          if exists($block->{'alt_width'});    # not exists in adfox and internal_context_rtb
    }

    $bk_data{'BlindLevel'}   = $block->{'blind'};
    $bk_data{'BlockCaption'} = $block->{'caption'};
    $bk_data{'MultiState'}   = $block->{'multistate'};

    $self->get_ad_type_set(\%bk_data, $block, allow_media_performance => TRUE);

    $bk_data{'AdFoxBlock'} = 1 if $self->accessor =~ /adfox/;    # only in adfox, to be merged with RTB

    $bk_data{'Video'}->{MaxDuration} = 60
      if (!$self->DOES('Application::Model::Role::Has::ContextShowVideo')
        || (grep {$self->accessor eq $_} qw(context_on_site_rtb internal_context_on_site_rtb))
        && $block->{show_video}
        && (grep {$_->{type} eq 'video'} @{$block->{design_templates}}));

    return \%bk_data;
}

sub _get_preferred_media_format {
    my ($self, $block, $selected_formats) = @_;

    if ($block->{'block_type'} eq 'interstitial' || $block->{'block_type'} eq 'rewarded') {
        my @preferred_formats = qw(300x250 300x300 240x400 400x240 320x480 480x320);
        my @result_formats = grep {$selected_formats->{$_}} @preferred_formats;

        return $result_formats[0] // '320x480';
    }

    my $media_blocks = $self->media_blocks();

    if (exists($media_blocks->{$block->{'media_block'}}{'size'})) {
        return $media_blocks->{$block->{'media_block'}}{'size'};
    }

    return $block->{'media_block'};
}

sub get_bk_mobile_rtb_settings {
    my ($self, $block) = @_;

    my $accessor = $self->accessor();
    my $is_ssp_blocks = in_array($accessor, [qw(ssp_context_on_site_rtb ssp_mobile_app_rtb ssp_video_an_site_rtb)]);

    my %bk_data;

    my $direct_blocks = $self->direct_blocks();

    my $type_name = $block->{'direct_block'}     // '';
    my $type_obj  = $direct_blocks->{$type_name} // {};

    my %skip_fields = $type_obj->{'skip_fields'} ? (map {$_ => TRUE} @{$type_obj->{'skip_fields'}}) : ();

    $bk_data{'DSPType'} = 4;    #2**2 mobile

    if (
        $self->app->user_features->page_owner_has_feature_simple_inapp($block)
        || (   $accessor eq 'internal_mobile_app_rtb'
            && $self->app->user_features->page_owner_has_feature_internal_simple_inapp($block))
       )
    {
        $bk_data{'Sizes'} = [
            {
                "Height" => "50",
                "Width"  => "320"
            },
            {
                "Height" => "90",
                "Width"  => "728"
            },
            {
                "Height" => "90",
                "Width"  => "970"
            },
            {
                "Height" => "100",
                "Width"  => "320"
            },
            {
                "Height" => "120",
                "Width"  => "1000"
            },
            {
                "Height" => "250",
                "Width"  => "300"
            },
            {
                "Height" => "250",
                "Width"  => "970"
            },
            {
                "Height" => "280",
                "Width"  => "336"
            },
            {
                "Height" => "300",
                "Width"  => "300"
            },
            {
                "Height" => "400",
                "Width"  => "240"
            },
            {
                "Height" => "500",
                "Width"  => "300"
            },
            {
                "Height" => "600",
                "Width"  => "160"
            },
            {
                "Height" => "600",
                "Width"  => "240"
            },
            {
                "Height" => "600",
                "Width"  => "300"
            }
        ];
    } else {
        my $dsp_blocks               = $self->dsp_blocks();
        my $blocks_for_section_sizes = {
            map {$_ => TRUE}
            map {exists($dsp_blocks->{$_}{'size'}) ? $dsp_blocks->{$_}{'size'} : $_} @{$block->{'dsp_blocks'}}
        };

        my $size = $self->_get_preferred_media_format($block, $blocks_for_section_sizes);

        $blocks_for_section_sizes->{$size} = TRUE;

        ($bk_data{'Width'}, $bk_data{'Height'}) = $size =~ m/^([0-9]+)x([0-9]+)$/;

        my $already_has_0x0;
        my ($width, $height);

        foreach my $type (sort keys %$blocks_for_section_sizes) {
            next unless $type =~ m/^([0-9]+)x([0-9]+)$/;
            $width  = $1;
            $height = $2;

            push(
                @{$bk_data{'Sizes'}},
                {
                    Width  => $width,
                    Height => $height,
                }
            );
            $already_has_0x0 = TRUE if $width == 0 && $height == 0;
        }

        # For video direct (PI-8892)
        push(
            @{$bk_data{'Sizes'}},
            {
                Width  => 0,
                Height => 0,
            }
        ) unless $already_has_0x0;

        if (exists($block->{'block_type'}) && $block->{'block_type'} eq 'adaptive_banner') {
            ($bk_data{'Width'}, $bk_data{'Height'}) = (0, 0);
            $bk_data{'Sizes'} = [];
        }
    }

    my $strategy = $block->{'strategy'};
    $bk_data{'OptimizeType'} = $strategy == 3 ? 0 : $strategy;

    if ($is_ssp_blocks) {
        #логика дублируется в get_bk_direct_design
        $bk_data{'DirectLimit'} = $block->{'limit'} = 1;
    } elsif (defined($block->{'limit'}) && !$skip_fields{'limit'}) {
        $bk_data{'DirectLimit'} = $block->{'limit'};
    } else {
        throw gettext('direct_block = "%s" not found', $block->{'direct_block'})
          unless $direct_blocks->{$block->{'direct_block'}};

        throw gettext('Limit not specified and "banners" default not found for block %s, direct_block = "%s"',
            $block->{'blockId'}, $block->{'direct_block'})
          if ( !$direct_blocks->{$block->{'direct_block'}}{'banners'}
            && !$direct_blocks->{$block->{'direct_block'}}{'max_banners'});
        $bk_data{'DirectLimit'} = $direct_blocks->{$block->{'direct_block'}}{'banners'}
          // $direct_blocks->{$block->{'direct_block'}}{'max_banners'};
    }

    $self->get_ad_type_set(\%bk_data, $block,
        allow_media_performance => $is_ssp_blocks || $accessor eq 'mobile_app_rtb');

    $bk_data{'InterstitialBlock'} = 1 if $block->{'block_type'} eq 'interstitial';

    $bk_data{'BlockCaption'} = $block->{'caption'};
    $bk_data{'MultiState'}   = $block->{'multistate'};

    $bk_data{'BlockType'} = $block->{'block_type'};

    return \%bk_data;
}

sub get_ad_type_set {
    my ($self, $bk_data, $block, %opts) = @_;

    my $is_simple_inapp = $self->app->user_features->page_owner_has_feature_simple_inapp($block);
    if (exists($block->{'block_type'})) {
        # в эту ветку зайдут только ВНЕШНИЕ мобильные блоки. Причем с любым owner, так как фича есть у всех
        if ($is_simple_inapp) {
            if ($block->{'block_type'} eq 'interstitial') {
                $bk_data->{'AdTypeSet'} = {
                    'media'             => 1,
                    'media-performance' => 1,
                    'text'              => 1,
                    'video'             => 1,
                    'video-motion'      => 1,
                    'video-performance' => 1,
                };

                $bk_data->{'AdType'} = {};

                return TRUE;
            } elsif ($block->{'block_type'} eq 'rewarded') {
                $bk_data->{'AdTypeSet'} = {
                    'media'             => 0,
                    'media-performance' => 0,
                    'text'              => 0,
                    'video'             => 1,
                    'video-motion'      => 1,
                    'video-performance' => 1,
                };

                $bk_data->{'AdType'} = {};

                return TRUE;
            } elsif ($block->{'block_type'} eq 'native') {
                $bk_data->{'AdTypeSet'} = {
                    'media'             => $block->{'rich_media'} // 0,
                    'media-performance' => $block->{'rich_media'} // 0,
                    'text'              => 1,
                    'video'             => $block->{'show_video'} // 0,
                    # todo (satsakul) PI-29220. Why we don't specify ad types for video-motion and video-performance products?
                };
                $bk_data->{'AdType'} = {};

                return TRUE;
            } elsif ($block->{'block_type'} eq 'banner') {
                $bk_data->{'AdTypeSet'} = {
                    'media'             => 1,
                    'media-performance' => 1,
                    'text'              => 1,
                    'video'             => $block->{'show_video'} // 0,
                    # todo (satsakul) PI-29220. Why we don't specify ad types for video-motion and video-performance products?
                };
                $bk_data->{'AdType'} = {};

                return TRUE;
            }
            # а в эту ветку зайдут только внутренние мобильные
        } else {
            if ($block->{'block_type'} eq 'interstitial') {
                $bk_data->{'AdTypeSet'} = {
                    'media' => $opts{'media'} // 1,
                    'media-performance' => $opts{'allow_media_performance'} ? 1 : 0,
                    'text'              => 1,
                    'video'             => $block->{'show_video'}           ? 1 : 0,
                    'video-performance' => $block->{'show_video'}           ? 1 : 0,
                    'video-motion'      => $block->{'show_video'}           ? 1 : 0,
                };

                $bk_data->{'AdType'} = {};

                return TRUE;
            }
            if ($block->{'block_type'} eq 'rewarded') {
                $bk_data->{'AdTypeSet'} = {
                    'media'             => 0,
                    'media-performance' => 0,
                    'text'              => 0,
                    'video'             => 1,
                    'video-motion'      => 1,
                    'video-performance' => 1,
                };

                $bk_data->{'AdType'} = {};

                return TRUE;
            } elsif ($block->{'block_type'} eq 'native') {
                $bk_data->{'AdTypeSet'} = {
                    # смотрим на rich_media, чтобы при переключении формирования бкдаты в порядке
                    # perl->java->perl на любом этапе логика была одинаковая PI-29691
                    'media'             => $block->{'rich_media'} ? 1 : 0,
                    'media-performance' => $block->{'rich_media'} ? 1 : 0,
                    'text'              => 1,
                    'video'             => $block->{'show_video'} ? 1 : 0,
                    'video-performance' => $block->{'show_video'} ? 1 : 0,
                    'video-motion'      => $block->{'show_video'} ? 1 : 0,
                };
                $bk_data->{'AdType'} = {};

                return TRUE;
            } elsif ($block->{'block_type'} eq 'adaptive_banner') {
                $bk_data->{'AdTypeSet'} = {
                    'media'             => 0,
                    'media-performance' => 0,
                    'text'              => 1,
                    'video'             => 0,
                    'video-performance' => 0,
                    'video-motion'      => 0,
                };

                $bk_data->{'AdType'} = {};

                return TRUE;
            }
        }
    }
    if ($block->{site_version}) {
        my $ad_type_set = {
            mobile_fullscreen => {
                'media'             => 1,
                'media-performance' => 1,
                'text'              => 1,
                'video'             => 1,
                'video-performance' => 1,
                'video-motion'      => 0,
            },
            mobile_rewarded => {
                'media'             => 0,
                'media-performance' => 0,
                'text'              => 1,
                'video'             => 1,
                'video-performance' => 1,
                'video-motion'      => 1,
            },
            mobile_floorad => {
                'media'             => 1,
                'media-performance' => 0,
                'text'              => 1,
                'video'             => 0,
                'video-performance' => 0,
                'video-motion'      => 0,
            },
        }->{$block->{site_version}};
        if ($ad_type_set) {
            $bk_data->{'AdTypeSet'} = $ad_type_set;
            return TRUE;
        }
    }

    my $ad_type_set = $bk_data->{'AdTypeSet'} = {
        'text'              => 1,
        'media'             => $opts{'media'} // 1,
        'video'             => 0,
        'media-performance' => $opts{'allow_media_performance'} ? 1 : 0,
        'video-performance' => 0,
    };

    my $ad_type = $bk_data->{'AdType'} = {};

    if (grep {$self->accessor eq $_}
        qw(mobile_app_rtb internal_mobile_app_rtb context_on_site_rtb internal_context_on_site_rtb))
    {
        foreach (qw(video video-performance video-motion)) {
            $ad_type_set->{$_} = $block->{'show_video'} ? 1 : 0;
        }
    } elsif ($self->accessor =~ /^ssp_/) {
        foreach (qw(video video-performance)) {
            $ad_type_set->{$_} = $block->{'show_video'} ? 1 : 0;
        }
        $ad_type_set->{'video-motion'} = 0;
    } elsif (exists($block->{'show_video'})) {
        $ad_type_set->{'video'} = $block->{'show_video'} ? 1 : 0;
    }

    my $adtype_depends = {
        media => [qw(media media-performance)],
        text  => [qw(text)],
        video => [qw(video video-performance), ($self->accessor =~ /^ssp_/ ? () : ('video-motion'))],
    };
    my @adtype_list = sort keys %{$adtype_depends};

    throw gettext('Do not turn off media and text ads')
      unless scalar grep {!$_} map {($block->{$_ . '_active'}, $block->{$_ . '_blocked'})} @adtype_list;

    foreach my $adtype (@adtype_list) {
        my $adtype_active  = $block->{$adtype . '_active'};
        my $adtype_blocked = $block->{$adtype . '_blocked'};
        my $adtype_cpm     = $block->{$adtype . '_cpm'};

        if ($adtype_active) {
            if ($adtype_blocked) {
                $ad_type_set->{$_} = 0 for @{$adtype_depends->{$adtype}};
            } else {
                throw Exception::Validation::BadArguments gettext($adtype . '_cpm must be defined')
                  unless $adtype_cpm;

                $ad_type->{$_} = {
                    'currency' => 'RUB',
                    'value'    => $adtype_cpm * 1000,
                  }
                  for @{$adtype_depends->{$adtype}};

                unless ($opts{'allow_media_performance'}) {
                    delete $ad_type->{'media-performance'};
                }
            }
        }
    }

    my $accessor = $self->accessor;
    my $block_id = $block->{'id'};
    if (($accessor eq 'context_on_site_rtb' || $accessor eq 'internal_context_on_site_rtb')
        && $block_id == $TECHNICAL_RTB_BLOCK_ID)
    {
        #PI-11113 PI-11346
        foreach (qw(media-performance media)) {
            $bk_data->{'AdTypeSet'}{$_} = 1;
        }
    }
}

sub get_bk_direct_design {
    my ($self, $block) = @_;

    my %design;
    $design{'blockId'} = $self->public_id($block)
      if defined $block->{$self->get_page_id_field_name()} && defined $block->{'id'};

    if ($self->app->user_features->page_owner_has_feature_simple_inapp($block)
        && in_array($self->accessor, [qw(mobile_app_rtb)]))
    {
        my %block_type_to_design_name = (
            banner       => 'adaptive0418',
            interstitial => 'interstitial',
            rewarded     => 'rewarded',
            native       => 'native',
        );

        $design{limit} = ($block->{'block_type'} eq 'native') ? ($block->{'limit'} // 1) : 1;

        $design{name} = $block_type_to_design_name{$block->{'block_type'}};
        return \%design;
    }

    # ours => bk's
    # not all fields are available for all blocks
    my $field_map = {
        type                 => 'name',                 # block name in old direct
        direct_block         => 'name',                 # block name in RTB
        favicon              => 'favicon',
        limit                => 'limit',
        border_radius        => 'borderRadius',
        border_type          => 'borderType',
        no_sitelinks         => 'noSitelinks',
        site_bg_color        => 'siteBgColor',
        bg_color             => 'bgColor',
        border_color         => 'borderColor',
        header_bg_color      => 'headerBgColor',
        title_color          => 'titleColor',
        text_color           => 'textColor',
        url_color            => 'urlColor',
        url_background_color => 'urlBackgroundColor',
        hover_color          => 'hoverColor',
        font_family          => 'fontFamily',
        font_size            => 'fontSize',
        title_font_size      => 'titleFontSize',
        links_underline      => 'linksUnderline',
        sitelinks_color      => 'sitelinksColor',
        adaptive_width       => 'width',
        adaptive_height      => 'height',
        adaptive_type        => 'adaptiveType',
        images_first         => 'imagesFirst',
        horizontal_align     => 'horizontalAlign',
    };

    my $direct_blocks  = $self->direct_blocks($block);
    my $boolean_fields = $self->boolean_fields();

    my $type_name = $block->{'direct_block'} // $block->{'type'} // '';
    my $type_obj = $direct_blocks->{$type_name} // {};

    my %skip_fields = $type_obj->{'skip_fields'} ? (map {$_ => TRUE} @{$type_obj->{'skip_fields'}}) : ();

    foreach my $key (keys(%$field_map)) {
        next if $skip_fields{$key};

        if ($boolean_fields->{$key}) {
            $design{$field_map->{$key}} = $block->{$key} ? JSON::XS::true : JSON::XS::false;
            next;
        }
        if (!exists($block->{$key}) || !defined($block->{$key}) || $block->{$key} eq '') {
            next;
        }

        $design{$field_map->{$key}} = $block->{$key};
    }

    if (in_array($self->accessor, [qw(ssp_context_on_site_rtb ssp_mobile_app_rtb ssp_video_an_site_rtb)])) {
        $design{'limit'} = 1;
    } else {
        $design{'limit'} = $type_obj->{'banners'}
          if ($skip_fields{'limit'} || !exists($block->{'limit'}) || !defined($block->{'limit'}))
          && defined($type_obj->{'banners'});
    }

    # https://st.yandex-team.ru/PI-12177#1524839795000
    $design{'horizontalAlign'} = $type_obj->{'horizontal_align'} ? JSON::XS::true : JSON::XS::false
      if defined($type_obj->{'horizontal_align'});

    $design{'name'} = "0x$1" if $design{'name'} =~ m/^[0-9]+%x([0-9]+)$/;

    # mobile
    $design{'name'} = 'interstitial'
      if $block->{'block_type'} && ($block->{'block_type'} eq 'interstitial' || $block->{'block_type'} eq 'rewarded');

    # internal context
    $design{'c11n'}{'textClickable'} = JSON::XS::true
      if defined $block->{id}
          && defined $block->{$self->get_page_id_field_name()}
          && $block->{$self->get_page_id_field_name()} == 129744
          && $block->{'id'} == 1;

    return \%design;
}

sub get_cnt {
    my ($self, $page_ids) = @_;

    my $page_field_name = $self->get_page_id_field_name();

    my $data = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => {
            $page_field_name => '',
            cnt => {count => ['id']}
        },
        filter => [
            AND => [
                [$page_field_name => 'IN'     => \$page_ids],
                [id               => '<>'     => \0],
                [multistate       => 'NOT IN' => \$self->get_multistates_by_filter('deleted')],
                ($self->is_block_table_with_multiple_models() ? ['model', '=', \$self->accessor()] : ()),
            ]
        ],
    )->group_by($page_field_name)->get_all();

    return {map {$_->{$page_field_name} => $_->{'cnt'}} @$data};
}

sub get_page_model {
    my ($self) = @_;

    my $page_model_name = $self->get_campaign_model_name();

    return $self->$page_model_name;
}

sub multi_do_action {
    my ($self, $page_id, $action, %opts) = @_;

    my $page_field_name = $self->get_page_id_field_name();

    my $blocks = $self->get_all(
        fields => $self->get_depends_for_field('actions'),
        filter => {
            $page_field_name => $page_id,
            multistate       => $self->get_multistate_by_action($action),
            (defined $opts{filter} ? %{$opts{filter}} : ())
        }
    );

    foreach my $block (@$blocks) {
        $self->do_action($block, $action, %opts)
          if $self->check_action($block, $action);
    }
}

sub need_update_in_bk {
    my ($self, $fields) = @_;

    my $fields_need_update = $self->get_need_update_in_bk_fields();

    foreach (@$fields) {
        return TRUE if $fields_need_update->{$_};
    }

    return FALSE;
}

sub related_models {
    my ($self) = @_;

    my $page_id_name = $self->page->get_page_id_field_name();

    return {
        pages => {
            accessor => 'page',
            filter   => sub {
                +{$page_id_name => array_uniq(map {$_->{'page_id'} // ()} @{$_[1]})};
            },
            key_fields => [$page_id_name],
        }
    };
}

sub throw_error_by_action {
    my ($self, $object, $action) = @_;

    if ($action eq 'edit' && $self->check_multistate_flag($object->{'multistate'}, 'deleted')) {
        throw Exception::Validation::BadArguments gettext('You cannot edit this block because the block archived');
    }

    if ($action eq 'edit') {
        unless (exists $object->{is_protected}) {
            my $obj = $self->_get_object_fields($object, [qw(is_protected)]);
            $object->{is_protected} = $obj->{is_protected};
        }

        Exception::Validation::BadArguments->throw(gettext('You can not edit block on protected page'))
          if $object->{is_protected} && !$self->check_rights('edit_protected_pages');
    }

    return FALSE;
}

sub hook_owner_processing { }

sub hook_post_validation {
    my ($self, $obj) = @_;

    $self->SUPER::hook_post_validation($obj);

    if ($self->hook_stash->inited && $self->hook_stash->mode('add')) {
        my $id         = $self->hook_stash->get('id');
        my $tmp_rights = $self->app->add_all_tmp_rights();
        my $full       = $self->get($id, fields => ['*']);
        undef $tmp_rights;

        my $model              = $self->accessor();
        my $page_id_field_name = $self->app->$model->get_page_id_field_name;

        my $page_bk_data = $self->get_bk_data($obj->{$page_id_field_name}, read_only => TRUE);
        return TRUE if ref($page_bk_data) ne 'HASH';

        if (my $block_bk_data = $page_bk_data->{$id->{id}}) {
            $self->app->validator->check(
                data => {%$block_bk_data, %{$self->hook_stash->get('id')},},
                template => {type => 'bk_data_hash',},
                app      => $self->app->$model,
                throw    => TRUE,
            );
        }
    }

    return TRUE;
}

our $STRATEGIES = [
    {id => 1, name => d_gettext('Maximum revenue'), fields => []},
    {id => 0, name => d_gettext('Minimum CPM'),     fields => ['mincpm']},
    {
        id     => 3,
        name   => d_gettext('Separate CPM'),
        fields => [
            qw(
              media_active
              media_blocked
              media_cpm
              text_active
              text_blocked
              text_cpm
              video_active
              video_blocked
              video_cpm
              video_performance_active
              video_performance_blocked
              video_performance_cpm
              )
        ]
    },
];

# TODO: It's wrong to have this method in Block.pm,
# because not all blocks has dsps settings.
# Move it to role (Has::DSPS?)
sub get_dsp_rules {
    my ($self) = @_;

    return [
        # dsp_type
        Partner::DSP::Rule->new(
            available_dsps => Partner::DSP::Rule::Part->new(
                depends_on => $self->get_dsp_type_depends(),
                sub        => sub {
                    my ($dsp) = @_;

                    limit => [types => '=' => $self->get_dsp_type($dsp)];
                },
            ),
            turn_on_dsp => Partner::DSP::Rule::Part->new(
                depends_on => {dsp => [qw(types)],},
                sub        => sub {
                    my ($dsp) = @_;

                    my %DSP_TYPES = map {$_ => TRUE} @{$self->get_dsp_type()};
                    if (grep {$DSP_TYPES{$_}} @{$dsp->{types}}) {
                        return limit => Partner::DSP::Rule::Filter->identity();
                    } else {
                        return limit => Partner::DSP::Rule::Filter->zero();
                    }
                },
            ),
        ),
        # sane multistate
        Partner::DSP::Rule->new(
            available_dsps => Partner::DSP::Rule::Part->new(
                sub => sub {
                    limit => [
                        AND => [
                            [id => '<>' => [$DSP_OWN_ADV_ID, $DSP_UNSOLD_ID]],
                            [
                                multistate => '=' =>
'created_in_bk and (created_in_pi or not_need_create_in_pi) and linked_in_balance and not deleted'
                            ]
                        ],
                    ];
                },
            ),
            turn_on_dsp => Partner::DSP::Rule::Part->new(
                depends_on => {dsp => [qw(id multistate)],},
                sub        => sub {
                    my ($dsp) = @_;
                    if (
                           $dsp->{id} != $DSP_OWN_ADV_ID
                        && $dsp->{id} != $DSP_UNSOLD_ID
                        && grep {$_ == $dsp->{multistate}} @{
                            $self->dsp->get_multistates_by_filter(
'created_in_bk and (created_in_pi or not_need_create_in_pi) and linked_in_balance and not deleted'
                            )
                        }
                       )
                    {
                        return limit => Partner::DSP::Rule::Filter->identity();
                    } else {
                        return limit => Partner::DSP::Rule::Filter->zero();
                    }
                },
            ),
        ),
    ];
}

sub get_dsp_type_depends {+{}}

sub _set_direct_limit {
    my ($self, $bk_data, $block) = @_;

    if (exists($block->{'design_templates'})
        && defined($block->{'design_templates'}))
    {
        my %designs_with_directlimit = map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA NATIVE);
        foreach (grep {$designs_with_directlimit{$_->{type}}} @{$block->{'design_templates'}}) {
            my $limit = $self->design_templates->get_direct_limit($_) // 0;
            $bk_data->{'DirectLimit'} = $limit if ($bk_data->{'DirectLimit'} // 0) < $limit;
        }
    }

    my $accessor = $self->accessor();
    if ($accessor eq 'context_on_site_rtb' || $accessor eq 'internal_context_on_site_rtb') {
        #PI-9811
        my $is_rtb_shadow = $block->{'id'} == $TECHNICAL_RTB_BLOCK_ID
          || ($block->{'ex_direct'} && !$block->{'active'}) ? 1 : 0;

        if ($is_rtb_shadow) {
            $bk_data->{'DirectLimit'} = 9;

            $bk_data->{'PageImpOptions'} = {
                Enable  => ['rtbshadow'],
                Disable => [],
            };
        } else {
            $bk_data->{'PageImpOptions'} = {
                Enable  => [],
                Disable => ['rtbshadow'],
            };
        }
    }
}

sub _split_id {
    my ($self, $id) = @_;

    return $id if not defined($id) or ref($id);

    my $prefix = $self->public_id_prefix();

    return $id =~ /^(?:$prefix)?(\d+)-(\d+)\z/
      ? {$self->get_page_id_field_name() => $1, id => $2}
      : $id;
}

sub design_field {'RtbDesign'}

# Просто чтобы показаь что эти ф-ции суть одно и тоже
sub get_all_applications_for_adding {
    my ($self, %opts) = @_;
    $self->get_all_campaigns_for_adding(%opts);
}

sub get_all_sites_for_adding {
    my ($self, %opts) = @_;
    $self->get_all_campaigns_for_adding(%opts);
}

sub get_all_campaigns_for_adding {
    my ($self, %opts) = @_;

    my $model              = $self->get_campaign_model_name();
    my $page_id_field_name = $self->$model->get_page_id_field_name();
    my $page_statuses      = $self->$model->get_multistate_names();

    my @filters = ();
    push(@filters, $opts{'filter'}) if defined($opts{'filter'});

    my @bad_statuses =
      grep {exists $page_statuses->{$_}} $opts{exclude_multistate}
      ? @{$opts{exclude_multistate}}
      : (qw(deleted rejected blocked), ($self->check_rights('edit_protected_pages') ? () : 'protected'));

    push(@filters, {multistate => 'not (' . join(' or ', @bad_statuses) . ')'})
      if @bad_statuses;

    my $model_fields = $self->$model->get_model_fields();
    my @fields =
      ($page_id_field_name, grep {exists($model_fields->{$_})} qw(caption domain page_id), @{$opts{'fields'} // []});

    push(@filters, [partner_read_only => '<>' => TRUE])
      if (exists($model_fields->{partner_read_only})
        && !$self->$model->check_short_rights('edit_field__partner_read_only'));

    # This code gets pages for assistants only if 'is_assistant' right is set for the current user.
    my $campaigns = $self->$model->get_all(
        fields => \@fields,
        (@filters ? (filter => [AND => \@filters]) : ()),
        ($self->$model->is_external_page() ? (query_opts => {assistant_can_edit => 1},) : ()),
        ($opts{limit}    ? (limit    => $opts{limit})    : ()),
        ($opts{order_by} ? (order_by => $opts{order_by}) : ()),
    );

    return $campaigns;
}

sub validate_page_reachable {
    my ($self, $page_id) = @_;

    my $page = $self->get_all_campaigns_for_adding(
        filter             => ['page_id' => '=' => $page_id],
        fields             => [qw(is_deleted is_blocked is_protected )],
        exclude_multistate => [],
    )->[0];

    if ($page) {
        throw Exception::Validation::BadArguments(gettext('You can not edit deleted page'))
          if $page->{is_deleted};

        Exception::Validation::BadArguments->throw(gettext('You can not edit block on protected page'))
          if $page->{is_protected} && !$self->check_rights('edit_protected_pages');

        Exception::Validation::BadArguments->throw(gettext('You can not edit blocked page'))
          if $page->{is_blocked};

    } else {
        throw Exception::Validator::Fields gettext('Can not find page: %s', $page_id // 'UNDEF');
    }

    return 1;
}

sub is_page_reachable {
    my ($self, $page_id_value) = @_;
    return 0 < @{$self->get_all_campaigns_for_adding(filter => ['page_id' => '=' => $page_id_value])};
}

sub check_data_from_adfox {
    my ($self, %opts) = @_;

    return TRUE;
}

sub get_extra_params_for_defaults {
    my ($self, $params) = @_;

    $params->{page_filter} = {
        from_json => TRUE,
        ref_type  => 'ARRAY',
    };

    $params->{page_limit} = {};

    $params->{page_sort} = {
        from_json => TRUE,
        ref_type  => 'ARRAY',
    };
}

sub fix_extra_params_for_defaults {
    my ($self, $opts) = @_;

    $opts->{page_limit} //= $LIMIT_CAMPAIGN_FOR_ADDING if $opts->{page_filter};

    if (!$opts->{page_filter} and !in_array($self->app->get_option('stage', ''), [qw(preprod production)])) {
        if (exists $opts->{attributes}{page_id}) {
            $opts->{page_filter} = ['page_id', '=', $opts->{attributes}{page_id}];
        }
        $opts->{page_limit} //= $LIMIT_CAMPAIGN_FOR_ADDING;
    }
}

sub check_feature_on_page {
    my ($self, $page_id, $feature) = @_;

    my $features_and_options = $self->get_features_and_options_on_page($page_id);

    return $features_and_options->{$feature};
}

sub get_features_and_options_on_page {
    my ($self, $page_id) = @_;
    my $owner;
    my $tmp_rights    = $self->app->add_all_tmp_rights();
    my $page_accessor = $self->get_campaign_model_name;
    if (in_array($page_accessor, $self->product_manager->get_internal_product_accessors())) {
        $owner = {owner_id => $ADINSIDE_USER_ID};
    } else {
        $owner = $self->page->get_all(filter => [page_id => '=' => $page_id], fields => [qw(owner_id)])->[0];
        return {} unless defined($owner);
    }

    my $user = $self->users->get($owner->{'owner_id'}, fields => [qw(allowed_design_auction_native_only features)])
      // {features => []};
    my $features = {map {$_ => TRUE} @{delete($user->{'features'})}};

    return {%$user, %$features};
}

sub get_strategies_available {
    my ($self) = @_;
    return {map {$_ => 1} ($MAX_REVENUE_STRATEGY_ID, $MIN_CPM_STRATEGY_ID, $SEPARATE_CPM_STRATEGY_ID)};
}

TRUE;
