package Application::Model::Block::Content::FastBan;

use qbit;

use Digest::MD5 qw(md5_hex);

use base qw(
  Application::Model::DBManager::Base
  RestApi::DBModel
  );

use Exception::Validator::Fields;
use Utils::JSON qw (fix_type_for_complex);

my %MODELS = map {$_ => 1} qw(
  context_on_site_content
  internal_context_on_site_content
  );

sub accessor      {'fast_ban'}
sub db_table_name {'fast_ban'}

sub get_product_name {gettext('fast_ban')}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {partner_db => 'Application::Model::PartnerDB',};
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => 'fast_ban',
            description => d_gettext('Right to manage fast ban'),
            rights      => {
                do_fast_ban_add    => d_gettext('Right to add fast ban'),
                do_fast_ban_delete => d_gettext('Right to delete fast ban'),
                fast_ban_view      => d_gettext('Right to view fast ban'),
                fast_ban_view_all  => d_gettext('Right to view all fast ban'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        id => {
            default => TRUE,
            db      => TRUE,
            pk      => TRUE,
            label   => d_gettext('ID'),
            type    => 'number',
            api     => 1,
        },
        public_id => {
            db_expr => 'id',
            db      => TRUE,
            default => TRUE,
            label   => d_gettext('Public ID'),
            type    => 'string',
            api     => 1,
        },
        model            => {db => TRUE, default => TRUE, type => 'string', api => 1,},
        page_id          => {db => TRUE, default => TRUE, type => 'number', api => 1,},
        block_id         => {db => TRUE, default => TRUE, type => 'number', api => 1,},
        url              => {db => TRUE, default => TRUE, type => 'string', api => 1,},
        url_md5_hex      => {db => TRUE, default => TRUE, type => 'string', api => 1,},
        create_date      => {db => TRUE, default => TRUE, type => 'string', api => 1,},
        available_fields => {
            depends_on => [qw(id)],
            get        => sub {
                return $_[0]->model->get_available_fields();
            },
            api      => 1,
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
        },
        editable_fields => {
            depends_on => [qw(id)],
            get        => sub {
                return $_[0]->model->get_editable_fields($_[1]);
            },
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
            api      => 1,
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            id       => {type => 'number', label => d_gettext('ID')},
            page_id  => {type => 'number', label => d_gettext('Page ID')},
            block_id => {type => 'number', label => d_gettext('Block ID')},
            url      => {type => 'text',   label => d_gettext('URL')},
        },
    };
}

sub get_available_fields {
    my ($self) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} qw(model page_id block_id url);

    return \%fields;
}

sub get_editable_fields {
    my ($self) = @_;

    return {};
}

sub add {
    my ($self, %opts) = @_;

    $self->app->validator->check(
        app      => $self,
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                model => {
                    type => 'scalar',
                    in   => [sort keys %MODELS],
                },
                page_id  => {type => 'int_un',},
                block_id => {type => 'int_un',},
                url      => {
                    type    => 'scalar',
                    len_min => 1,
                },
            },
            check => sub {
                my ($qv, $data) = @_;

                my $model = $data->{'model'};

                my $block = $qv->app->app->$model->get(
                    {campaign_id => $data->{'page_id'}, id => $data->{'block_id'}},
                    fields     => [qw(public_id actions)],
                    query_opts => {assistant_can_edit => 1},
                ) // throw Exception::Validator::Fields gettext('Block not found');
            },
        },
        throw => TRUE,
    );

    $opts{create_date} = curdate(oformat => 'db_time');
    $opts{url_md5_hex} = md5_hex(unset_utf($opts{url}));

    my $id = $self->partner_db_table()->add(\%opts);

    return $id;
}

sub delete {
    my ($self, $filter) = @_;

    $self->partner_db_table()->delete($self->partner_db->filter($filter));
}

sub api_available_actions {qw()}

sub api_can_add {TRUE}

sub api_can_delete {TRUE}

sub query_filter {
    my ($self, $filter, $opts) = @_;

    my @filters = ();
    foreach my $model (sort grep {$self->app->$_->check_short_rights('view')} keys(%MODELS)) {
        push(
            @filters,
            [
                'AND',
                [
                    ['model', '=', \$model],
                    (
                        $self->app->$model->check_short_rights('view_all')
                        ? ()
                        : [
                            {'' => [qw(page_id block_id)]},
                            'IN',
                            $self->app->$model->query(
                                fields  => $self->app->$model->_get_fields_obj([qw(campaign_id id)]),
                                options => $opts->{'options'},
                            )
                          ]
                    )
                ]
            ]
        );
    }

    if (@filters) {
        $filter->and(['OR', \@filters]);
    } else {
        $filter->and(['AND', [\undef]]);
    }

    return $filter;
}

TRUE;
