package Application::Model::Block::Dooh;
# Dooh == Digital out-of-home

use qbit;

use base qw(
  Application::Model::Block::External
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

use PiConstants qw(
  $DSP_VIDEO_TYPE_ID
  $MIN_CPM_STRATEGY_ID
  $DSP_DIRECT_ID
  );

consume qw(
  Application::Model::Role::Block::Has::BlocksLimit
  Application::Model::Role::Block::Has::FieldsModeration
  Application::Model::Role::Block::Has::IsHidden
  Application::Model::Role::Has::Categories
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::CustomBkData
  Application::Model::Role::Has::DSPS::WithoutModeWithSettings
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::Strategies
  Application::Model::Role::Has::Readonly
  );

use Exception;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

__PACKAGE__->abstract_methods(
    qw(get_campaign_model_name get_product_name get_opts_schema_name public_id_prefix db_table_name));

my $DOOH_BLOCK_PHOTOS_MAX = 6;

sub get_page_id_field_name {'page_id'}

sub get_dsp_type {[$DSP_VIDEO_TYPE_ID]}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {mds_avatars => 'Application::Model::MDSAvatars',};
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          edit_field__comment
          view_field__comment
          edit_field__photolink_list
          view_field__photolink_list
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $page_accessor = $self->get_campaign_model_name();
    return {
        hash_transform(
            $self->SUPER::get_structure_model_fields(),
            [
                qw(
                  id
                  page_id
                  public_id
                  login
                  owner_id
                  owner_client_id
                  multistate
                  multistate_name
                  internal_campaign_id
                  is_protected
                  is_page_deleted
                  is_deleted
                  fields_depends
                  status
                  statistics_exists
                  caption
                  comment
                  )
            ]
        ),
        page => {
            depends_on => [
                'page_id',            'pages.id',               'pages.multistate', 'pages.owner_id',
                'pages.address',      'pages.gps',              'pages.domain',     'pages.facility_type',
                'pages.business_oid', 'pages.address_approved', 'pages.gps_approved'
            ],
            get => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}};
            },
            type => $self->get_campaign_model_name(),
        },
        domain => {
            depends_on => ['page_id', 'pages.domain'],
            label      => d_gettext('Domain'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'domain'};
            },
            type => 'string',
            api  => 1
        },
        facility_type => {
            depends_on => ['page_id', 'pages.facility_type'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'facility_type'};
            },
            type => 'number',
            api  => 1
        },
        # overwrite
        caption => {
            from_opts  => 'db_generated',
            default    => TRUE,
            need_trim  => TRUE,
            type       => 'string',
            need_check => {
                type    => 'scalar',
                len_max => 255,
            },
            api => 1
        },
        resolution => {
            from_opts  => 'db_generated',
            need_check => {type => 'resolution',},
            need_trim  => TRUE,
            type       => 'string',
            api        => 1
        },
        photo_id_list => {
            from_opts  => 'from_hash',
            type       => 'array',
            sub_type   => 'number',
            need_check => {
                optional => TRUE,
                type     => 'array',
                all      => {type => 'scalar'},
                check    => sub {
                    my ($qv, $photo_id_list) = @_;

                    if ($DOOH_BLOCK_PHOTOS_MAX < @$photo_id_list) {
                        throw Exception::Validator::Fields gettext("No more than %d photoes allowed for one block",
                            $DOOH_BLOCK_PHOTOS_MAX);
                    }

                    my $KEEP_HOURS     = $qv->app->mds_avatars->hours_keep();
                    my $dt_n_hours_ago = date_sub(
                        curdate(oformat => 'db_time'),
                        hour    => $KEEP_HOURS,
                        iformat => 'db_time',
                        oformat => 'db_time'
                    );

                    # making sure that all the images are present in mds_avatars now
                    # and wont be removed by MDSAvatarsCleanup cron for some time
                    my $user_photos = $qv->app->mds_avatars->get_all(
                        fields => ['id'],
                        filter => [
                            'AND' => [
                                ['id', 'IN', $photo_id_list],
                                ['OR' => [['block_cnt', '>', 0], ['update_dt', '>=', $dt_n_hours_ago],]]
                            ]
                        ]
                    );

                    unless (@$photo_id_list == @$user_photos) {
                        my %hh = map {$_->{id} => 1} @$user_photos;
                        throw Exception::Validator::Fields gettext("Photo link [%s] is unacceptable",
                            join(',', grep {!exists $hh{$_}} @$photo_id_list));
                    }
                  }
            },
            api => 1,
        },
        photo_link_list => {
            type       => 'array',
            sub_type   => 'complex',
            depends_on => ['photo_id_list', 'photo_links.md5', 'photo_links.links'],
            get        => sub {
                my ($self, $obj) = @_;
                return $obj->{'photo_id_list'} ? [@{$self->{'photo_links'}}{@{$obj->{'photo_id_list'}}}] : [];
            },
            api => 1,
        },
        bk_state_name => {
            depends_on => ['page_id', 'pages.bk_state_name'],
            label      => d_gettext('BK status'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'bk_state_name'};
            },
        },
        #redefine
        cur_user_is_read_assistant => {
            depends_on => ['id'],
            get        => sub {
                0
              }
        },
        photo_link_list_approved => {
            type       => 'array',
            sub_type   => 'complex',
            depends_on => ['photo_id_list_approved', 'photo_links.md5', 'photo_links.links'],
            get        => sub {
                my ($self, $obj) = @_;
                return $obj->{'photo_id_list_approved'}
                  ? [@{$self->{'photo_links'}}{@{$obj->{'photo_id_list_approved'}}}]
                  : [];
            },
        },
    };
}

sub related_models {
    my ($self) = @_;

    return {
        %{$self->SUPER::related_models()},
        photo_links => {
            accessor => 'mds_avatars',
            filter   => sub {
                +{id => array_uniq(map {@{from_json($_->{opts})->{photo_id_list} // []}} @{$_[1]})};
            },
            key_fields => ['id'],
        }
    };
}

sub storage_method_of_strategies {'opts'}

sub get_strategies {
    return {$MIN_CPM_STRATEGY_ID => {}};
}

sub get_actions_depends {
    [
        qw(
          cur_user_is_read_assistant
          id
          is_protected
          multistate
          page
          page_id
          )
    ];
}

sub get_editable_fields_depends {
    [qw(cur_user_is_read_assistant id is_protected multistate page page_id)];
}

sub get_available_fields_depends {[qw(id)]}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {
            %{$parent_filter->{'fields'}},
            page => {
                type           => 'subfilter',
                model_accessor => $self->get_campaign_model_name(),
                field          => 'page_id',
                fk_field       => 'page_id',
                label          => d_gettext('Site'),
            },
            login   => {type => 'alias', path => [qw(page owner login)]},
            caption => {
                type       => 'json',
                value_type => 'text',
                label      => d_gettext('Caption')
            },
            resolution => {
                type       => 'json',
                value_type => 'text',
                label      => d_gettext('Resolution')
            },
            touch_screen => {
                type       => 'json',
                value_type => 'boolean',
                label      => d_gettext('Touch Screen')
            },
            sound => {
                type       => 'json',
                value_type => 'boolean',
                label      => d_gettext('Sound')
            },
            zone_category => {
                type       => 'json',
                value_type => 'number',
                label      => d_gettext('Zone Category')
            },
        }
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    my $product = $self->get_campaign_model_name();
    return [
        (
            $self->check_short_rights('view_filter__login')
            ? (
                {
                    name         => 'login',
                    label        => gettext('Login'),
                    login_filter => $product . '_an_partner=1'
                }
              )
            : ()
        ),
        {name => 'page_id',     label => gettext('Page ID')},
        {name => 'id',          label => gettext('Block ID')},
        {name => 'page.domain', label => gettext('Resource')},
        {name => 'caption',     label => gettext('Block\'s caption')},
        {name => 'multistate',  label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted          => d_pgettext('Block multistate', 'Archived')],
            [working          => d_pgettext('Block multistate', 'Working')],
            [check_statistics => d_pgettext('Block multistate', 'Check statistics'), deprecated => TRUE],
            [
                deleted_with_page => d_pgettext('Block multistate', 'Archived with page'),
                private           => TRUE
            ],
            [need_update => d_pgettext('Block multistate', 'Need update')],
            [updating    => d_pgettext('Block multistate', 'Updating')],
            # По сути получается, что флаги approved и rejected будут условно фиктивными:
            #   rejected - будет использоваться только для отклонения ещё ни разу не зааппрувленного блока,
            #              потом при реджекте будет просто оставаться старое зааппрувленное значение поля
            #   approved - будет использоваться для обозначения,
            #              что блок когда-то успешно прошёл модерацию
            #   need_approve - флаг того, что есть непромодерированные элементы
            [need_approve => d_pgettext('Block multistate', 'On moderation')],
            [approved     => d_pgettext('Block multistate', 'Approved')],
            [rejected     => d_pgettext('Block multistate', 'Rejected')],
            [blocked      => d_pgettext('Block multistate', 'Blocked')],
            [testing      => d_pgettext('Block multistate', 'Testing')],
        ],
        right_name_prefix => $self->accessor . '_',
        actions           => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            duplicate              => d_pgettext('Block action', 'Duplicate'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
        },
        right_actions => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
            set_need_approve => d_pgettext('Block action', 'Return to moderation'),
            approve          => d_pgettext('Block action', 'Approve'),
            reject           => d_pgettext('Block action', 'Reject'),
            set_blocked      => d_pgettext('Block action', 'Set blocked'),
            reset_blocked    => d_pgettext('Block action', 'Reset blocked'),
            start_testing    => d_pgettext('Block action', 'Start testing'),
            stop_testing     => d_pgettext('Block action', 'Stop testing'),
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page and not blocked',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not (deleted or blocked)',
            },
            {
                action    => 'start',
                from      => 'approved and not ( working or deleted or blocked or rejected )',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
            {
                action      => 'set_need_approve',
                from        => 'not blocked',
                set_flags   => ['need_approve'],
                reset_flags => ['rejected'],
            },
            {
                action      => 'approve',
                from        => 'need_approve or rejected',
                set_flags   => ['approved'],
                reset_flags => ['need_approve', 'rejected'],
            },
            {
                action      => 'reject',
                from        => 'need_approve and not approved',
                set_flags   => ['rejected'],
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reject',
                from        => 'need_approve and approved',
                reset_flags => ['need_approve'],
            },
            {
                action    => 'set_blocked',
                from      => 'not blocked',
                set_flags => ['blocked'],
            },
            {
                action      => 'reset_blocked',
                from        => 'blocked',
                reset_flags => ['blocked'],
            },
            {
                action => 'duplicate',
                from   => 'not (deleted or deleted_with_page)',
            },
            {
                action    => 'start_testing',
                from      => 'not testing',
                set_flags => ['testing'],
            },
            {
                action      => 'stop_testing',
                from        => 'testing',
                reset_flags => ['testing'],
                set_flags   => ['need_approve'],
            },
        ],
    };
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $page_accessor . '_view_field__owner' => [qw(owner_client_id)],
            $page_accessor . '_view_field__%s'    => 'login',
            $accessor . '_view_field__%s'         => [qw(comment photo_id_list)],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{'page_id'} = TRUE;

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    return $self->_get_common_add_edit_fields();
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              caption
              resolution
              )
        ],
        right_fields => {edit => ['comment', 'photo_id_list']}
    );

    return $fields;
}

sub can_action_delete {TRUE}

sub can_action_approve {TRUE}

sub on_action_approve { }

sub can_action_reject {TRUE}

sub on_action_reject { }

sub can_action_restore {
    my ($self, $block) = @_;

    my $page_accessor = $self->get_campaign_model_name();
    my $fields = $self->_get_object_fields($block, [qw(page)]);

    return $self->$page_accessor->check_action($fields->{'page'}, 'restore_block');
}

sub on_action_restore {
    my ($self, $block, %opts) = @_;

    my $page_accessor = $self->get_campaign_model_name();
    my $tmp_rights    = $self->app->add_tmp_rights($self->$page_accessor->get_right('view_field__owner'));

    my $fields = $self->_get_object_fields($block, [qw(page owner_client_id)]);

    undef($tmp_rights);

    # Since old blocks were archived the agreement might have been changed
    #
    if ($self->$page_accessor->check_multistate_flag($fields->{'page'}{'multistate'}, 'working')) {
        my $can_start = $self->agreement_checker->has_agreement_for_any_product_for_today(
            client_id => $fields->{'owner_client_id'},
            products  => [$self->accessor],
        );

        unless ($can_start) {
            throw Exception::Denied $self->get_agreement_error_message();
        }
    }

    my $block_fields = $self->_get_object_fields($block, [qw(page_id id)]);

    $self->maybe_do_action($block, 'start');

    $self->update_in_bk($block) unless $opts{'do_not_update_in_bk'};

    return TRUE;
}

sub on_action_set_blocked { }

sub on_action_set_need_approve { }

sub on_action_stop {
    my ($self, $obj, %opts) = @_;

    $self->update_in_bk($obj);

    return TRUE;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    my $page_accessor = $self->get_campaign_model_name();
    $self->partner_db->transaction(
        sub {
            $opts->{'id'} = $self->$page_accessor->get_next_block_id($opts->{'page_id'});
        }
    );
    $opts->{'photo_id_list'} //= [];
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    $self->SUPER::hook_processing_after_insert($opts);

    my $page_model = $self->get_page_model();
    my $page_id    = $self->hook_stash->get('id')->{page_id};
    my $tmp_rights = $self->app->add_tmp_rights($page_model->get_right('view_field__multistate'));
    my $page       = $page_model->get_all(
        fields => [qw(multistate)],
        filter => {$page_model->get_page_id_field_name() => $page_id},
    )->[0];
    if ($page_model->check_multistate_flag($page->{multistate}, 'testing')) {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('start_testing'));
        $self->do_action($self->hook_stash->get('id'), 'start_testing');
        $self->update_in_bk($self->hook_stash->get('id'));
    } else {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('set_need_approve'));
        $self->do_action($self->hook_stash->get('id'), 'set_need_approve');
    }

    # mds_avatars.block_cnt += 1 to ids from opts->photo_id_list
    $self->_photo_list_counters(new_value => $self->hook_stash->get('settings')->{photo_id_list},);
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    $self->SUPER::hook_processing_after_update($opts);
    $self->_photo_list_counters(
        approved  => $self->hook_stash->get('current')->{photo_id_list_approved},
        new_value => $self->hook_stash->get('settings')->{photo_id_list},
        old_value => $self->hook_stash->get('current')->{photo_id_list},
    );
    $self->update_in_bk($self->hook_stash->get('id')) if $self->need_update_in_bk($self->hook_stash->get('fields'));
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} qw(
      address
      caption
      gps
      resolution
      );

    return \%fields_need_update;
}

sub get_multistate_filter_for_update_in_bk {'(approved or testing) and not deleted and not blocked'}

sub get_bk_block_data {
    my ($self, $block) = @_;

    my $public_id = $block->{'public_id'};

    my $bk_data = {
        'BlockModel'   => $self->accessor(),
        'DSPType'      => 2,                                 #2**1 video
        'Sizes'        => [{'Width' => 0, 'Height' => 0}],
        'BlockCaption' => $block->{'caption'},
        'MultiState'   => $block->{'multistate'},
        'OptimizeType' => 0,
        'AdTypeSet'    => {
            'text'              => 0,
            'media'             => 0,
            'video'             => 1,
            'media-performance' => 0,
            'video-performance' => 1,
        },
        'AdType' => {
            'video' => {
                'currency' => 'RUB',
                'value'    => 1000,
            },
        },
        'Video'           => {'CountPositions' => 1},
        'CustomBlockData' => $self->get_custom_block_data($block),
    };

    return $bk_data;
}

sub get_bk_common_data_wrapper {
    return ();
}

sub get_bk_block_data_key {'rtb_blocks'}

sub get_custom_block_data {
    my ($self, $block) = @_;

    return {
        dooh => {
            facility_type => $block->{'facility_type'} + 0,
            photo_list    => $self->_get_photo_list($block),
            resolution    => $block->{'resolution'},
        }
    };
}

sub get_dsp_rules {
    my ($self) = @_;

    return [
        @{$self->SUPER::get_dsp_rules()},
        Partner::DSP::Rule->new(
            default_dsps => Partner::DSP::Rule::Part->new(
                sub => sub {
                    extend => {id => [$DSP_DIRECT_ID]},;
                },
            ),
        ),
    ];
}

sub get_fields_moderated {
    my ($self) = @_;

    return {photo_id_list => 'ARRAY',};
}

# API

sub api_available_actions {
    my ($self) = @_;

    my @actions = qw(
      delete
      restore
      edit
      start
      stop
      duplicate
      );

    return @actions;
}

sub api_can_edit {TRUE}
sub api_can_add  {TRUE}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'zone_category'}) {
        $result{'zone_category'} = $self->get_zone_categories();
    }

    my @extra_page_fields = $self->get_fields_defaults_extra_page_fields();

    if ($need_fields->{'page_id'}) {
        $result{'page_id'} = $self->get_all_campaigns_for_adding(
            %{$opts->{attributes}},
            (@extra_page_fields   ? (fields   => \@extra_page_fields)  : ()),
            ($opts->{page_filter} ? (filter   => $opts->{page_filter}) : ()),
            ($opts->{page_limit}  ? (limit    => $opts->{page_limit})  : ()),
            ($opts->{page_sort}   ? (order_by => $opts->{page_sort})   : ()),
        );
    }

    if ($need_fields->{'dsps'}) {
        my $rules = $self->get_dsp_rule_set();
        $result{'dsps_available'} = $self->get_available_dsps($opts->{attributes}, $rules);
        $result{'dsps_default'} = $self->get_default_dsps($opts->{attributes}, $rules);
    }

    if ($need_fields->{'resolution'}) {
        my $facility_type = $self->page->facility_type();
        $result{'resolutions'} =
          [map {{id => $_, resolutions => $facility_type->{$_}->{resolutions} // []}} keys %$facility_type];
    }

    return \%result;
}

sub get_fields_defaults_extra_page_fields {
    return qw();
}

sub get_zone_categories {
    return [];
}

sub get_fields_depends {
    return {
        #если поменялось поле из ключ, то
        #нужно перезапросить поля из значения
        depends => {},
        #для поля из ключа обязятельно нужно передать поля из значения
        required => {},
    };
}

# SEE https://st.yandex-team.ru/PI-14453#5c6544fd5f45c300210c3612
#[
#    {
#      "formats" => [
#        {
#          "height" => 640,
#          "path" => "/get-partner/65861/99_cent.jpg/orig",
#          "width" => 1178
#        },
#      ]
#    },
#];
sub _get_photo_list {
    my ($self, $block) = @_;

    my $test_mode = $self->check_multistate_flag($block->{multistate}, 'testing');
    return [
        map {
            {formats => [sort {$a->{'path'} cmp $b->{'path'}} values %{$_->{links}}]}
          } @{$block->{$test_mode ? 'photo_link_list' : 'photo_link_list_approved'}}
    ];
}

# Параметры:
#   approved  - уже используемые утверждённые значения
#               для них надо оставить
#   new_value - новые данные, часть может быть не утверждена
#               для новых надо увеличить счётчик
#   old_value - старые данные, часть может уже не использоваться
#               для неиспользуемых надо уменьшить счётчик
sub _photo_list_counters {
    my ($self, %opts) = @_;

    my $dt = curdate(oformat => 'db_time');
    $opts{$_} //= [] foreach (qw(approved new_value old_value));

    my %hh_photo_id_list_new = map {$_ => 1} @{$opts{approved}}, @{$opts{new_value}};
    my %hh_photo_id_list_old = map {$_ => 1} @{$opts{approved}}, @{$opts{old_value}};
    my @id_list2update;

    # mds_avatars.block_cnt -= 1 to images that are no longer with the block
    @id_list2update = grep {!exists $hh_photo_id_list_new{$_}} keys %hh_photo_id_list_old;
    if (@id_list2update) {
        $self->mds_avatars->partner_db_table()->edit(
            \@id_list2update,
            {
                update_dt => $dt,
                block_cnt => ['-', ['block_cnt', \1]],
            }
        );
    }

    # mds_avatars.block_cnt += 1 to new ids from opts->photo_id_list
    @id_list2update = grep {!exists $hh_photo_id_list_old{$_}} keys %hh_photo_id_list_new;
    if (@id_list2update) {
        $self->mds_avatars->partner_db_table()->edit(
            \@id_list2update,
            {
                update_dt => $dt,
                block_cnt => ['+', ['block_cnt', \1]],
            }
        );
    }
}

sub get_dooh_block_photos_max {
    return $DOOH_BLOCK_PHOTOS_MAX;
}

sub get_ad_type_list {
    return [qw(video)];
}

TRUE;
