package Application::Model::Block::External::Site;

use qbit;

use base qw(Application::Model::Block::External);

use Exception::Denied;
use Exception::Validator::Fields;

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        context_on_site_campaign => 'Application::Model::Product::AN::ContextOnSite::Campaign',
        search_on_site_campaign  => 'Application::Model::Product::AN::SearchOnSite::Campaign',
    };
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        campaign_id => {
            (
                $self->get_page_id_field_name() eq 'campaign_id'
                ? (
                    pk      => TRUE,
                    default => TRUE,
                  )
                : (db_expr => 'page_id',)
            ),
            db          => TRUE,
            label       => d_gettext('Campaign ID'),
            hint        => d_gettext('The unique campaign identifier (the same in BK)'),
            type        => 'number',
            adjust_type => 'str',
            need_check  => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $page_id) = @_;
                    $qv->app->validate_page_reachable($page_id);
                  }
            },
        },
        campaign => {
            depends_on => [
                'page_id',        'pages.id',    'pages.multistate', 'pages.caption',
                'pages.owner_id', 'pages.login', 'pages.domain_id',  'pages.domain'
            ],
            get => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}};
            },
            type => 'context_on_site_campaign',
            api  => 1
        },
        is_campaign_deleted => {
            depends_on => ['is_page_deleted'],
            get        => sub {
                $_[1]->{'is_page_deleted'};
            },
            type => 'boolean',
            api  => 1
        },
        domain_id => {
            depends_on => ['page_id', 'pages.domain_id'],
            label      => d_gettext('Domain ID'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'domain_id'};
            },
            check_rights => ($self->get_campaign_model_name() . '_view_field__domain_id'),    # page's right!
            type         => 'number',
            adjust_type  => 'str',
        },
        domain => {
            depends_on => ['page_id', 'pages.domain'],
            label      => d_gettext('Domain'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'domain'};
            },
            type => 'string',
            api  => 1
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {
            %{$parent_filter->{'fields'}},
            campaign_id       => {type => 'number', label => d_gettext('Page ID')},
            campaign          => {type => 'alias',  path  => [qw(page)]},
            login             => {type => 'alias',  path  => [qw(page owner login)]},
            is_graysite_block => {type => 'alias',  path  => [qw(page domain is_graysite)],},
        }
    };
}

sub on_action_restore {
    my ($self, $block, %opts) = @_;

    my $fields = $self->_get_object_fields($block, [qw(page owner_client_id)]);

    # Since old blocks were archived the agreement might have been changed
    #
    if ($self->page->check_multistate_flag($fields->{'page'}{'multistate'}, 'working')) {
        my $can_start = $self->agreement_checker->has_agreement_for_any_product_for_today(
            client_id => $fields->{'owner_client_id'},
            products  => [$self->accessor],
        );

        unless ($can_start) {
            throw Exception::Denied $self->get_agreement_error_message();
        }
    }

    $self->maybe_do_action($block, 'start');

    $self->update_in_bk($block);

    return TRUE;
}

TRUE;
