package Application::Model::Block::InDoor;

use qbit;

use base qw( Application::Model::Block::Dooh );

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Brands
  Application::Model::Role::Has::DoohDurationMax
  Application::Model::Role::Has::DoohDurationMin
  );

use Exception;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

use PiConstants qw($INDOOR_PARTNER_ROLE_ID $MAX_CPM);

my $ZONE_CATEGORY = {
    # Прикассовая зона
    1 => {code => 'cash_zone', label => d_gettext('cash zone')},
    # Овощи-фрукты
    2 => {code => 'vegetables_and_fruits', label => d_gettext('vegetables and fruits')},
    # Мясо-рыба
    3 => {code => 'meat-fish', label => d_gettext('meat-fish')},
    # Молочная продукция
    4 => {code => 'dairy_products', label => d_gettext('dairy products')},
    # Алкогольная продукция
    5 => {code => 'alcoholic_beverages', label => d_gettext('alcoholic beverages')},
    # Кондитерские изделия
    6 => {code => 'confectionery', label => d_gettext('confectionery')},
    # Бакалея
    7 => {code => 'grocery', label => d_gettext('grocery')},
    # Детское питание
    8 => {code => 'baby_food', label => d_gettext('baby food')},
    # Бытовая химия
    9 => {code => 'chemistry', label => d_gettext('chemistry')},
    # Косметика
    10 => {code => 'cosmetic', label => d_gettext('cosmetic')},
    # Товары для дома (разное)
    11 => {code => 'for_home', label => d_gettext('for home (miscellaneous)')},
    # Товары для животных
    12 => {code => 'pet_products', label => d_gettext('pet products')},
    # Спорт
    13 => {code => 'sport', label => d_gettext('sport')},
    # Одежда
    14 => {code => 'clothes', label => d_gettext('clothes')},
    # Обувь
    15 => {code => 'shoes', label => d_gettext('shoes')},
    # Фойе
    16 => {code => 'lobby', label => d_gettext('lobby')},
    # Другое
    17 => {code => 'other', label => d_gettext('other')},
};

my $ZONE_CATEGORY_ID_LIST = [sort keys %$ZONE_CATEGORY];
my $ZONE_CATEGORY_ARRAY   = [
    map {id => $_, code => $ZONE_CATEGORY->{$_}->{'code'}, label => $ZONE_CATEGORY->{$_}->{'label'}->()},
    sort keys %{$ZONE_CATEGORY}
];

sub accessor      {'indoor_block'}
sub db_table_name {'indoor_block'}

sub get_campaign_model_name {'indoor'}

sub get_product_name {gettext('indoor_block')}

sub public_id_prefix {'I-D-'}

sub get_opts_schema_name {'indoor_block_opts'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {indoor => 'Application::Model::Page::InDoor',};
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();
    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          view_filter__login
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $page_accessor = $self->get_campaign_model_name();
    return {
        %{$self->SUPER::get_structure_model_fields()},
        address => {
            depends_on => ['page_id', 'pages.address'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'address'};
            },
            type => 'string',
            api  => 1
        },
        aspect_ratio => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_trim  => TRUE,
            need_check => {
                type   => 'scalar',
                regexp => qr/^[1-9]\d*:[1-9]\d*$/,
                msg    => d_gettext('Aspect ratio must be in format like "16:9"'),
            },
            api => 1
        },
        gps => {
            depends_on => ['page_id', 'pages.gps'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'gps'};
            },
            type => 'string',
            api  => 1
        },
        sound => {
            from_opts  => 'db_generated',
            type       => 'boolean',
            need_check => {type => 'boolean', optional => TRUE},
            api        => 1
        },
        touch_screen => {
            from_opts  => 'db_generated',
            type       => 'boolean',
            need_check => {type => 'boolean', optional => TRUE},
            api        => 1
        },
        zone_category => {
            from_opts  => 'db_generated',
            type       => 'number',
            need_check => {
                type => 'int_un',
                in   => $ZONE_CATEGORY_ID_LIST,
                msg  => d_gettext("unknown zone_category"),
            },
            api => 1
        },
    };
}

sub get_custom_block_data {
    my ($self, $block) = @_;

    my $design = $self->SUPER::get_custom_block_data($block);

    $design->{'dooh'}->{'business_oid'} //= $block->{'page'}{'business_oid'};
    $design->{'dooh'}->{'zone_category'} = $block->{'zone_category'} + 0;
    $design->{'dooh'}->{'touch_screen'}  = $block->{'touch_screen'} ? JSON::XS::true : JSON::XS::false;
    $design->{'dooh'}->{'sound'}         = $block->{'sound'} ? JSON::XS::true : JSON::XS::false;
    $design->{'dooh'}->{'aspect_ratio'} =
      $block->{'aspect_ratio'} ? [map {0 + $_;} split ':', $block->{'aspect_ratio'}] : [];

    return $design;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields      => $self->SUPER::_get_common_add_edit_fields(),
        no_right_fields => [
            qw(
              aspect_ratio
              sound
              touch_screen
              zone_category
              )
        ],
    );

    return $fields;
}

sub on_action_stop_testing {
    my ($self, $obj) = @_;
    $self->_try_moderate_in_place($obj->{moderation}, $obj);
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = (
        %{$self->SUPER::get_need_update_in_bk_fields()},
        map {$_ => TRUE} (
            qw(
              sound
              touch_screen
              zone_category
              )
        )
    );

    return \%fields_need_update;
}

sub get_fields_defaults_extra_page_fields {
    return qw(address);
}

sub get_zone_categories {$ZONE_CATEGORY_ARRAY}

sub add_blocks {
    my ($self, $login, $articles, $brands, $blocks) = @_;

    my $partner = $self->app->users->get_all(
        fields => [qw(id)],
        filter => [AND => [['login' => '=' => $login], ['role_id' => '=' => $INDOOR_PARTNER_ROLE_ID]]]
    );
    throw gettext('Not found: partner for: %s', $login) unless (1 == @$partner && $partner->[0]->{id});

    my ($articles2block, $articles_not_found) = _get_articles2block($self->app, $articles);
    my ($brands2block, $brands_not_found) = _get_brands2block($self->app, $brands);

    my $facility_types = $self->get_page_model->facility_type();

    my %pages;
    my @errors;
    for (my $i = 0; $i < @$blocks; $i++) {
        my $block = $blocks->[$i];
        $errors[$i] = [];
        next if ($block->{page_id} && $block->{block_id});

        unless (exists $facility_types->{$block->{facility_type}}) {
            push @{$errors[$i]}, gettext('Not found: facility_type: %s', $block->{facility_type});
            next;
        }

        my $gps_ok = QBit::Validator->new(
            data     => $block->{page_gps},
            template => {type => 'gps',},
        );
        if ($gps_ok->has_error()) {
            push @{$errors[$i]}, $gps_ok->get_error();
            next;
        }

        my $org_name = $block->{page_oname};
        unless (exists $pages{$block->{page_caption}}{$block->{page_gps}}{$org_name}) {

            my $business_oid;
            unless ($facility_types->{$block->{facility_type}}->{skip_business_oid}) {
                $business_oid = _get_business_oid($self->app, $block->{page_gps}, $org_name);
                unless ($business_oid) {
                    push @{$errors[$i]}, gettext('Not found: business_oid for: %s', $block->{page_oname});
                    next;
                }
            }

            my @ll = map {$_ + 0} split(',', $block->{page_gps});
            my $filter = $self->partner_db->filter(
                [
                    AND => [
                        [
                            {truncate => [{substring_index => ['gps', \',', \1]}, \7]} => '=' =>
                              {truncate => [\$ll[0], \7]}
                        ],
                        [
                            {truncate => [{substring_index => ['gps', \',', \-1]}, \7]} => '=' =>
                              {truncate => [\$ll[1], \7]}
                        ],
                        ['caption'       => '=' => \$block->{page_caption}],
                        ['facility_type' => '=' => \$block->{facility_type}],
                        $business_oid ? (['business_oid' => '=' => \$business_oid]) : (),
                    ]
                ]
            );

            my $page_existing = $self->get_page_model()->get_all(
                fields => [qw(owner_id page_id)],
                filter => $filter,
                limit  => 1,
            );

            if (scalar @$page_existing) {
                # error: page exists and owner is not login listed in the file
                if ($partner->[0]->{id} != $page_existing->[0]{owner_id}) {
                    push @{$errors[$i]},
                      gettext(
                        'Other partner owns this page: page_id=%s owner_id=%s',
                        $page_existing->[0]{page_id},
                        $page_existing->[0]{owner_id}
                      );
                    next;
                } else {
                    $pages{$block->{page_caption}}{$block->{page_gps}}{$org_name} = $page_existing->[0]{page_id};
                }
            } else {
                try {
                    my $indoor_id = $self->get_page_model()->add(
                        login         => $login,
                        address       => $block->{page_address},
                        caption       => $block->{page_caption},
                        facility_type => $block->{facility_type},
                        gps           => $block->{page_gps},
                        ($business_oid ? (business_oid => $business_oid) : ()),
                    );

                    my $page_added = $self->get_page_model()->get($indoor_id, fields => ['page_id']);
                    $pages{$block->{page_caption}}{$block->{page_gps}}{$org_name} = $page_added->{page_id};

                }
                catch {
                    my ($ex) = @_;
                    push @{$errors[$i]}, gettext('Cant create page: %s', $ex->message());
                };
            }
        }

        $block->{page_id} = $pages{$block->{page_caption}}{$block->{page_gps}}{$org_name};

        next unless $block->{page_id};

        my @mds_avatars_id_list;
        foreach my $img_link (@{$block->{block_photos}}) {
            try {
                my $id = $self->app->mds_avatars->add(page_id => $block->{page_id}, img_link => $img_link);
                push @mds_avatars_id_list, $id;
            }
            catch {
                push @{$errors[$i]}, gettext('WARN: cant add img: %s', $img_link);
            };
        }

        try {
            my $indoor_block_id = $self->add(
                page_id       => $block->{page_id},
                caption       => $block->{block_caption},
                comment       => $block->{block_comment},
                zone_category => $block->{zone_category},
                aspect_ratio  => $block->{block_aspect_ratio},
                resolution    => sprintf('%sx%s', $block->{block_reso_width}, $block->{block_reso_height}),
                min_duration  => $block->{block_min_duration},
                max_duration  => $block->{block_max_duration},
                sound         => $block->{block_sound},
                touch_screen  => $block->{block_touch},
                strategy      => 0,
                mincpm        => $block->{block_mincpm},
                brands        => $brands2block,
                articles      => $articles2block,
                photo_id_list => \@mds_avatars_id_list,
            );
            $block->{block_id} = $indoor_block_id;
        }
        catch {
            my ($ex) = @_;
            push @{$errors[$i]}, gettext('Cant create block: %s', $ex->message());
        };
    }

    return ($blocks, \@errors, $articles_not_found, $brands_not_found);
}

sub _get_business_oid {
    my ($app, $page_gps, $org_name) = @_;

    my @org_list = grep {_org_name_eq($org_name, $_->{name})} @{$app->api_http_maps->get_companies_by_ll($page_gps)};

    return scalar @org_list ? $org_list[0]->{business_oid} : undef;
}

sub _get_articles2block {
    my ($app, $articles) = @_;

    my $list = $app->tns_dict_article->get_all(
        fields => [qw(sid name)],
        filter => [AND => [['name' => 'IN' => $articles // []],]]
    );
    return (
        to_json([map {{id => $_->{sid}, cpm => $MAX_CPM}} @$list]),
        arrays_difference($articles, [map {$_->{name}} @$list]),
    );
}

sub _get_brands2block {
    my ($app, $brands) = @_;

    my $list = $app->tns_dict_brand->get_all(
        fields => [qw(bid name)],
        filter => [AND => [['name' => 'IN' => $brands // []],]]
    );
    return ([map {{bid => $_->{bid} + 0, blocked => 1}} @$list], arrays_difference($brands, [map {$_->{name}} @$list]),
    );
}

sub _org_name_eq {
    my ($s1, $s2) = @_;
    return trim(lc($s1 // '')) eq trim(lc($s2 // ''));
}

TRUE;
