package Application::Model::Block::MIXIN::Direct;

use qbit;
use Exception::Validator::Fields;

use base qw(QBit::Application::Model);

use PiConstants qw($FONT_SIZES);

sub get_direct_model_fields {
    my ($self) = @_;

    return {
        type => {
            db         => TRUE,
            label      => d_gettext('Block format'),
            need_check => {in => [map {$_->{'id'}} @{$self->direct_block_types()}]}
        },
        limit => {
            db         => TRUE,
            label      => d_gettext('Count ad'),
            need_check => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $limit) = @_;

                    my %direct_blocks =
                      map {$_->{'id'} => $_} @{$qv->app->get_types_direct_blocks()};

                    my $direct_block = $qv->data->{'type'};
                    throw Exception::Validator::Fields gettext('Invalid limit')
                      unless $direct_blocks{$direct_block}
                          && $direct_blocks{$direct_block}->{'min_banners'}
                          && $limit >= $direct_blocks{$direct_block}->{'min_banners'}
                          && $direct_blocks{$direct_block}->{'max_banners'}
                          && $limit <= $direct_blocks{$direct_block}->{'max_banners'};
                },
            },
            adjust_type => 'str',
        },
        border_radius => {
            db          => TRUE,
            need_check  => {type => 'boolean',},
            adjust_type => 'str',
        },
        border_type => {
            db         => TRUE,
            need_check => {in => [qw(none block ad collapse)],}
        },
        no_sitelinks => {
            db          => TRUE,
            need_check  => {type => 'boolean', optional => TRUE,},
            adjust_type => 'str',
        },

        site_bg_color   => {db => TRUE, type => 'string', need_check => {type => 'color'}},
        header_bg_color => {db => TRUE, type => 'string', need_check => {type => 'color', optional => TRUE,}},
        bg_color        => {db => TRUE, type => 'string', need_check => {type => 'color', optional => TRUE,}, api => 1},
        border_color => {db => TRUE, type => 'string', need_check => {type => 'color',}, api => 1},
        favicon      => {
            db          => TRUE,
            type        => 'boolean',
            need_check  => {type => 'boolean'},
            api         => 1,
            adjust_type => 'str',
        },
        font_family =>
          {db => TRUE, need_check => {in => ['', 'arial', 'courier new', 'tahoma', 'times new roman', 'verdana',]},},
        font_size       => {db => TRUE, type => 'string', need_check => {in   => $FONT_SIZES,}, api => 1},
        hover_color     => {db => TRUE, type => 'string', need_check => {type => 'color'},      api => 1},
        links_underline => {
            db          => TRUE,
            type        => 'boolean',
            need_check  => {type => 'boolean'},
            api         => 1,
            adjust_type => 'str',
        },
        sitelinks_color => {db => TRUE, type => 'string', need_check => {type => 'color'}, api => 1},
        text_color      => {db => TRUE, type => 'string', need_check => {type => 'color'}, api => 1},
        title_color     => {db => TRUE, type => 'string', need_check => {type => 'color'}, api => 1},
        title_font_size => {
            db          => TRUE,
            type        => 'string',
            need_check  => {in => [qw(1 2 3)],},
            api         => 1,
            adjust_type => 'str',
        },
        url_color => {db => TRUE, type => 'string', need_check => {type => 'color'}, api => 1},
    };
}

sub fix_template {
    my ($self, $qv) = @_;

    my $template = $qv->template;

    if ($qv->data->{'border_type'} && $qv->data->{'border_type'} eq 'none') {
        foreach (
            qw(
            border_color
            border_radius
            )
          )
        {
            $template->{'fields'}{$_} = {eq => undef};
        }
    }

    if ($qv->data->{'no_sitelinks'}) {
        $template->{'fields'}{'sitelinks_color'} = {eq => undef};
    }

    my $direct_block = $qv->data->{'type'};

    my %direct_blocks =
      map {$_->{'id'} => $_} @{$qv->app->direct_block_types()};

    if (   defined($direct_block)
        && $direct_blocks{$direct_block}
        && $direct_blocks{$direct_block}->{'skip_fields'})
    {
        foreach (@{$direct_blocks{$direct_block}->{'skip_fields'}}) {
            my $val = $_ eq 'adaptive_height' ? {in => ['']} : {eq => undef};
            $template->{'fields'}{$_} = $val;
        }
    }

    $qv->template($template);
}

sub set_block_default_values {
    my ($self, $new_settings, $old_settings) = @_;

    # TODO: remove parameters & stash inited check when PREMIUM models start using add/on_action_edit from Common
    my $old_block_settings = $self->hook_stash->inited ? $self->hook_stash->get('current') : $old_settings;
    my $changed_settings   = $self->hook_stash->inited ? $self->hook_stash->get('opts')    : $new_settings;

    if (   $changed_settings->{'type'}
        && $old_block_settings->{'type'}
        && $changed_settings->{'type'} ne $old_block_settings->{'type'})
    {
        # when changing from type 'horizontal' to 'vertical' these values are undef
        # so we set defaults from frontend here
        $changed_settings->{'font_family'}     //= '';
        $changed_settings->{'hover_color'}     //= '0066FF';
        $changed_settings->{'sitelinks_color'} //= '0000CC';
        $changed_settings->{'title_color'}     //= '0000CC';
        $changed_settings->{'title_font_size'} //= '3';
        $changed_settings->{'url_color'}       //= '006600';
        # TODO: fix this when PREMIUM is on hooks
        # FIXME: internal PREMIUM doesnt have this field
        if ('internal_search_on_site_premium' ne $self->accessor()) {
            $changed_settings->{'url_background_color'} //= '0000CC';
        }
    }

    my $direct_block = $changed_settings->{'type'} || $old_block_settings->{'type'};

    my %direct_blocks =
      map {$_->{'id'} => $_} @{$self->direct_block_types()};

    if (   defined($direct_block)
        && $direct_blocks{$direct_block}
        && $direct_blocks{$direct_block}->{'skip_fields'})
    {
        foreach (@{$direct_blocks{$direct_block}->{'skip_fields'}}) {
            my $val = $_ eq 'adaptive_height' ? '' : undef;
            $changed_settings->{$_} = $val;
        }
    }

    if (exists($changed_settings->{'border_type'}) && $changed_settings->{'border_type'}) {
        if ($changed_settings->{'border_type'} eq 'none') {
            foreach (qw(border_color border_radius)) {
                $changed_settings->{$_} = undef;
            }
        }
    } else {
        if ($old_block_settings->{'border_type'} && $old_block_settings->{'border_type'} eq 'none') {
            foreach (qw(border_color border_radius)) {
                $changed_settings->{$_} = undef;
            }
        }
    }

    if (exists($changed_settings->{'no_sitelinks'})) {
        if ($changed_settings->{'no_sitelinks'}) {
            $changed_settings->{'sitelinks_color'} = undef;
        }
    } else {
        if ($old_block_settings->{'no_sitelinks'}) {
            $changed_settings->{'sitelinks_color'} = undef;
        }
    }

    # TODO: remove parameters & stash inited check when PREMIUM models start using add/on_action_edit from Common
    return $self->hook_stash->inited
      ? $self->hook_stash->get('settings')
      : {%$old_block_settings, %$changed_settings};
}

TRUE;
