package Application::Model::Block::OutDoor;

use qbit;

use base qw( Application::Model::Block::Dooh );

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Brands
  Application::Model::Role::Has::DoohDurationMax
  Application::Model::Role::Has::DoohDurationMin
  Application::Model::Role::Has::DoohDurationSlot
  );

use Exception;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

sub accessor      {'outdoor_block'}
sub db_table_name {'outdoor_block'}

sub get_campaign_model_name {'outdoor'}

sub get_product_name {gettext('outdoor_block')}

sub public_id_prefix {'O-D-'}

sub get_save_dsps_right_name {'outdoor_block_edit_field__dsps'}

sub get_opts_schema_name {'outdoor_block_opts'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {outdoor => 'Application::Model::Page::OutDoor',};
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();
    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          view_filter__login
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $pattern_dot = qr(\[(?:\d+(?:\.\d+)?,){2}\d+(?:\.\d+)?\]);

    return {
        %{$self->SUPER::get_structure_model_fields()},
        address => {
            from_opts  => 'from_hash',
            need_check => {
                len_min => 10,
                len_max => 512,
            },
            need_trim => TRUE,
            type      => 'string',
            api       => 1
        },
        gps => {
            from_opts  => 'from_hash',
            need_check => {type => 'gps'},
            need_trim  => TRUE,
            type       => 'string',
            api        => 1
        },
        #overwrite. Отличается от базового только в depends_on
        page => {
            depends_on =>
              ['page_id', 'pages.id', 'pages.multistate', 'pages.owner_id', 'pages.domain', 'pages.facility_type'],
            get => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}};
            },
            type => $self->get_campaign_model_name(),
        },
        direction => {
            from_opts  => 'from_hash',
            type       => 'number',
            need_check => {
                type => 'scalar',
                min  => 0,
                max  => 359,
            },
            api => 1
        },
        height => {
            from_opts  => 'from_hash',
            type       => 'number',
            need_check => {
                type => 'scalar',
                min  => 1,
            },
            api => 1
        },
        # зона видимости, набор точек
        view_zone => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_trim  => TRUE,
            need_check => {
                type     => 'scalar',
                optional => TRUE,
                # https://st.yandex-team.ru/PI-14462#5c6692df6e2a9b001e3aa5a8
                # array of polygons; polygon = array of dots; dot = array of 3 float
                # assuming string '[[[0.3, 0.5,  0.0], [1.3, 1.5,  1.07]], [[2.3,  2.5, 2.04], [3.3, 3.5, 3.04]]]'
                regexp => qr/^\[(?:\[(?:$pattern_dot,)*$pattern_dot\],)*\[(?:$pattern_dot,)*$pattern_dot\]\]$/,
                msg    => d_gettext('View zone must be in format like "[[0.3,0.5,0.0],[1.3,1.5,1.07]]"'),
            },
            api => 1
        },
        width => {
            from_opts  => 'from_hash',
            type       => 'number',
            need_check => {
                type => 'scalar',
                min  => 1,
            },
            api => 1
        },
    };
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields      => $self->SUPER::_get_common_add_edit_fields(),
        no_right_fields => [
            qw(
              address
              direction
              gps
              height
              view_zone
              width
              )
        ],
    );

    return $fields;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = (
        %{$self->SUPER::get_need_update_in_bk_fields()},
        map {$_ => TRUE} (
            qw(
              aspect_ratio
              direction
              )
        )
    );

    return \%fields_need_update;
}

sub get_custom_block_data {
    my ($self, $block) = @_;

    my $design = $self->SUPER::get_custom_block_data($block);

    $design->{'dooh'}->{'view_zone'} = $block->{'view_zone'} ? from_json($block->{'view_zone'}) : [];
    $design->{'dooh'}->{$_} = $block->{$_} foreach (qw( direction height width ));

    return $design;
}

sub get_fields_defaults_extra_page_fields {
    return qw(facility_type);
}

sub get_fields_moderated {
    my ($self) = @_;

    return {%{$self->SUPER::get_fields_moderated()}, map {$_ => 1,} qw(address gps caption),};
}

TRUE;
