package Application::Model::Block::Video;

use qbit;

use base qw(
  Application::Model::Block::External
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

use Partner::Filters qw(get_filters_not_overlay_instream_blocks);
use Utils::JSON qw(fix_type_for_complex);

consume qw(
  Application::Model::Role::Block::Has::BlocksLimit
  Application::Model::Role::Block::Has::PartnerReadOnly
  Application::Model::Role::Block::Has::Tags
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::Categories
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Brands
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::CustomBkData
  Application::Model::Role::Has::DSPS::WithSettings
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::Geo
  Application::Model::Role::Has::Strategies
  Application::Model::Role::Has::Readonly
  );

use PiConstants qw(
  $BLIND_VALUES
  $DSP_DIRECT_ID
  $DSP_VIDEO_TYPE_ID
  $INPAGE
  $INTERSTITIAL
  $MAX_REVENUE_STRATEGY_ID
  $MIDROLL
  $MIN_CPM_STRATEGY_ID
  $PAUSE
  $POSTROLL
  $PREROLL
  $INROLL
  $S2SROLL
  $OVERLAY
  $INROLL_OVERLAY
  $OVERLAY_TYPES
  $SEPARATE_CPM_STRATEGY_ID
  $VIDEO_BLOCK_TYPES
  $VIDEO_PLATFORM_FLASH
  $VIDEO_PLATFORM_HTML5
  @VIDEO_DEFAULT_DSPS
  );

use Exception;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

my %MAX_BLOCKS_BY_TYPE = (
    $PREROLL        => 1,
    $POSTROLL       => 1,
    $MIDROLL        => 30,
    $PAUSE          => 1,
    $INPAGE         => 1,
    $INTERSTITIAL   => 1,
    $INROLL         => 30,
    $S2SROLL        => 30,
    $OVERLAY        => 30,
    $INROLL_OVERLAY => 30,
);

my @MAX_DURATION = (5, 15, 20, 30, 45, 60, 90, 180);

__PACKAGE__->abstract_methods(
    qw(
      get_product_name
      get_types_ids
      )
);

sub get_campaign_model_name {'video_an_site'}

sub get_page_id_field_name {'page_id'}

sub get_dsp_type {[$DSP_VIDEO_TYPE_ID]}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        video_an_site            => 'Application::Model::Page::Video',
        dsp                      => 'Application::Model::DSP',
        video_an_site_categories => 'Application::Model::Product::VideoAN::Site::Categories',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => $self->accessor(),
            description => d_gettext('Right to manage %s', $self->accessor()),
            rights      => {
                map {$self->get_description_right($_)}
                  qw(
                  view
                  view_all
                  add
                  view_action_log
                  view_field__comment
                  view_field__blind
                  view_field__get_code
                  view_field__video_motion
                  edit
                  edit_field__comment
                  edit_field__blind
                  edit_field__dsps
                  edit_field__video_motion
                  view_filter__login
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        category_id => {
            db          => TRUE,
            label       => d_gettext('Category ID'),
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        category_path => {db => TRUE, label => d_gettext('Category ID'), type => 'string', api => 1},
        type          => {
            db         => TRUE,
            label      => d_gettext('Type'),
            type       => 'number',
            api        => 1,
            need_check => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $type_id) = @_;

                    my $types = $qv->app->get_types_ids();
                    unless (in_array($type_id, $types)) {
                        throw Exception::Validator::Fields gettext('Got value "%s" not in array: %s', $type_id,
                            join(', ', @$types));
                    }
                  }
            },
            adjust_type => 'str',
        },
        max_duration => {
            db         => TRUE,
            label      => d_gettext('Max Duration'),
            type       => 'number',
            need_check => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $max_duration) = @_;
                    throw Exception::Validator::Fields gettext('For "max_duration" select one of the values: %s',
                        join(', ', @MAX_DURATION))
                      unless in_array($max_duration, \@MAX_DURATION);
                  }
            },
            api         => 1,
            adjust_type => 'str',
        },
        alternative_code => {
            db         => TRUE,
            label      => d_gettext('Alternative code'),
            type       => 'string',
            api        => 1,
            need_check => {
                type     => 'scalar',
                optional => TRUE,
                check    => sub {
                    my ($qv, $alternative_code) = @_;

                    throw Exception::Validator::Fields gettext("Can't create adfox block with alternative_code")
                      if $qv->{'data'}{'adfox_block'} && $alternative_code;
                  }
            }
        },
        blind => {
            db          => TRUE,
            label       => d_gettext('Blind'),
            type        => 'number',
            need_check  => {in => $BLIND_VALUES},
            api         => 1,
            adjust_type => 'str',
        },
        video_motion => {
            db         => TRUE,
            type       => 'boolean',
            need_check => {
                optional => TRUE,
                type     => 'boolean',
            },
            api         => 1,
            adjust_type => 'str',
        },
        type_label => {
            depends_on => [qw(type)],
            label      => d_gettext('Type label'),
            get        => sub {
                throw Exception::Validation::BadArguments gettext('Unknown type with id = "%s"', $_[1]->{'type'})
                  unless $_[0]->model->check_type($_[1]->{'type'});

                return $VIDEO_BLOCK_TYPES->{$_[1]->{'type'}}{'label'}();
            },
            type => 'string',
            api  => 1
        },
        type_bk => {
            depends_on => [qw(type)],
            label      => d_gettext('Type BK'),
            get        => sub {
                throw Exception::Validation::BadArguments gettext('Unknown type with id = "%s"', $_[1]->{'type'})
                  unless $_[0]->model->check_type($_[1]->{'type'});

                return $VIDEO_BLOCK_TYPES->{$_[1]->{'type'}}{'bk'};
              }
        },
        domain => {
            depends_on => ['page_id', 'pages.domain'],
            label      => d_gettext('Domain'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'domain'};
            },
            type => 'string',
            api  => 1
        },
        login => {
            depends_on   => ['page_id', 'pages.login'],
            label        => d_gettext('Login'),
            check_rights => $self->video_an_site->get_right('view_field__owner'),
            type         => 'string',
            get          => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'login'};
            },
            api => 1
        },
        owner_id => {
            depends_on   => ['page_id', 'pages.owner_id'],
            label        => d_gettext('Owner ID'),
            check_rights => $self->video_an_site->get_right('view_field__owner'),
            get          => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'owner_id'};
            },
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        owner_client_id => {
            depends_on   => ['page_id', 'pages.client_id'],
            label        => d_gettext('Client ID'),
            check_rights => $self->video_an_site->get_right('view_field__owner'),
            get          => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'client_id'};
            },
            adjust_type => 'str',
        },
        bk_state_name => {
            depends_on => ['page_id', 'pages.bk_state_name'],
            label      => d_gettext('BK status'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'bk_state_name'};
            },
        },
        vpaid_enabled => {
            depends_on => ['page_id', 'pages.vpaid_enabled'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'vpaid_enabled'};
            },
            adjust_type => 'str',
        },
        platform => {
            depends_on => ['page_id', 'pages.platform'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'platform'};
            },
            adjust_type => 'str',
        },
        vast_version => {
            depends_on => ['page_id', 'pages.vast_version'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'vast_version'};
            },
            adjust_type => 'str',
        },

        # fictive, obsolete (redefined in InStream)
        count_positions => {
            db          => TRUE,
            db_expr     => \1,
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
    };
}

sub storage_method_of_strategies {'db'}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {
            %{$parent_filter->{'fields'}},
            page => {
                type           => 'subfilter',
                model_accessor => 'video_an_site',
                field          => 'page_id',
                label          => d_gettext('Site'),
            },
            type => {
                type   => 'dictionary',
                label  => gettext('Platforms'),
                values => sub {
                    [
                        map {
                            {hash_transform($_, ['id', 'label'])}
                          } @{$_[0]->get_types()}
                    ];
                  }
            },
            category_id   => {type => 'number', label => d_gettext('Category ID')},
            login         => {type => 'alias',  path  => [qw(page owner login)]},
            vpaid_enabled => {type => 'alias',  path  => [qw(page vpaid_enabled)],},
        }
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login')
            ? (
                {
                    name         => 'login',
                    label        => gettext('Login'),
                    login_filter => 'video_an_partner=1'
                }
              )
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type')
            ? {
                name  => 'user_type',
                label => gettext('User type')
              }
            : ()
        ),
        {name => 'page_id',     label => gettext('Page ID')},
        {name => 'category_id', label => gettext('Categories ID')},
        {name => 'id',          label => gettext('Block ID')},
        {name => 'caption',     label => gettext('Block\'s caption')},
        {name => 'page.domain', label => gettext('Resource')},
        {name => 'type',        label => gettext('Type')},
        {name => 'multistate',  label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted          => d_pgettext('Block multistate', 'Archived')],
            [working          => d_pgettext('Block multistate', 'Working')],
            [check_statistics => d_pgettext('Block multistate', 'Check statistics')],
            [
                deleted_with_page => d_pgettext('Block multistate', 'Archived with page'),
                private           => TRUE
            ],
            [need_update => d_pgettext('Block multistate', 'Need update')],
            [updating    => d_pgettext('Block multistate', 'Updating')],
        ],
        right_name_prefix => $self->accessor . '_',
        actions           => {
            delete => d_pgettext('Block action', 'Archive'),
            duplicate => {label => d_pgettext('Block action', 'Duplicate')},
            edit                   => d_pgettext('Block action', 'Edit'),
            restore                => d_pgettext('Block action', 'Restore'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
        },
        right_actions => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
            {
                action => 'duplicate',
                from   => 'not (deleted or deleted_with_page)',
            },
        ],
    };
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => [qw(comment blind get_code is_tutby video_motion)],

            $page_accessor . '_view_field__owner' => [qw(owner_client_id)],
            $page_accessor . '_view_field__%s'    => 'login',
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %result = ();

    foreach (
        qw(
        page_id
        caption
        category_id
        category_path
        type
        max_duration
        alternative_code
        )
      )
    {
        $result{$_} = TRUE;
    }

    $result{'dsps'} = TRUE
      if $self->check_rights('video_an_site_block_dsps_save');

    foreach (
        qw(
        comment
        blind
        video_motion
        )
      )
    {
        $result{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    return \%result;
}

sub get_actions_depends {
    [
        qw(
          cur_user_is_read_assistant
          id
          is_protected
          multistate
          page
          page_id
          category_id
          type
          )
    ];
}

sub get_available_fields_depends { }

sub get_editable_fields_depends {
    [
        qw(
          cur_user_is_read_assistant
          id
          is_protected
          multistate
          page_id
          page
          type
          )
    ];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res;

    map {$res{$_} = TRUE} qw(
      caption
      max_duration
      );

    $res{'alternative_code'} = TRUE unless $data->{'adfox_block'};

    foreach (qw(comment blind video_motion)) {
        $res{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    $res{'dsps'} = TRUE if $self->check_rights('video_an_site_block_dsps_save');

    return \%res;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_processing_before_validation($opts);
    #TODO: удалить после добавления роли RTB
    $opts->{'blind'} //= 0;

    if ($self->hook_stash->mode('add')) {
        $self->can_add_block(%$opts);
    }
}

sub hook_fields_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_validation($opts);

    # Т.к. фронт при добавлении видеоблока вызывает get_default_dsps без параметров,
    # сбросить DSP Директа автоматически не получается
    if ($self->hook_stash->mode('add') && defined $opts->{'dsps'} && grep {$_->{'dsp_id'} == $DSP_DIRECT_ID}
        @{$opts->{'dsps'}})
    {
        my $page =
          $self->video_an_site->get_all(fields => [qw(vpaid_enabled)], filter => {page_id => $opts->{'page_id'}});

        throw Exception::Validation::BadArguments gettext("Can't set DSP Direct with disabled vpaid option")
          if @$page && !$page->[0]->{'vpaid_enabled'};
    }
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{'category_path'} = $self->get_path_by_category_id($opts->{'page_id'}, $opts->{'category_id'});

    $self->partner_db->transaction(
        sub {
            $opts->{'id'} = $self->video_an_site->get_next_block_id($opts->{'page_id'});
        }
    );
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;
    my $block_id = $self->hook_stash->get('id');

    $self->maybe_do_action($block_id, 'start');

    $self->update_in_bk($block_id);
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;
    $self->SUPER::hook_processing_after_update($opts);
    $self->update_in_bk($self->hook_stash->get('id')) if $self->need_update_in_bk($self->hook_stash->get('fields'));
}

sub can_action_duplicate {
    my ($self, $block) = @_;

    my $block_being_duped = $self->_get_object_fields($block, [qw(type)]);
    return (1 == $MAX_BLOCKS_BY_TYPE{$block_being_duped->{type}}) ? FALSE : TRUE;
}

sub on_action_duplicate {
    my ($self, $block, %opts) = @_;

    # this is not in 'can_action' bc its heavy AND is called by frontend for each block in block lists
    my $block_being_duped = $self->_get_object_fields($block, [qw(page_id category_id type id)]);
    # remove 'id' from args, or currently duplicated block will not count towards the limit
    my $block_id = delete $block_being_duped->{id};

    $self->can_add_block(%$block_being_duped);

    $self->SUPER::on_action_duplicate({%$block_being_duped, id => $block_id}, %opts);
}

sub can_action_restore {
    my ($self, $block) = @_;

    my $fields = $self->_get_object_fields($block, [qw(page)]);

    return $self->video_an_site->check_action($fields->{'page'}, 'restore_block');
}

sub on_action_restore {
    my ($self, $block, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights($self->video_an_site->get_right('view_field__owner'));

    my $fields = $self->_get_object_fields($block, [qw(page owner_client_id)]);

    undef($tmp_rights);

    # Since old blocks were archived the agreement might have been changed
    #
    if ($self->video_an_site->check_multistate_flag($fields->{'page'}{'multistate'}, 'working')) {
        my $can_start = $self->agreement_checker->has_agreement_for_any_product_for_today(
            client_id => $fields->{'owner_client_id'},
            products  => [$self->accessor],
        );

        unless ($can_start) {
            throw Exception::Denied $self->get_agreement_error_message();
        }
    }

    my $block_fields = $self->_get_object_fields($block, [qw(page_id category_id type id)]);

    $self->can_add_block(%$block_fields);

    $self->maybe_do_action($block, 'start');

    $self->update_in_bk($block) unless $opts{'do_not_update_in_bk'};

    return TRUE;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} qw(
      blind
      max_duration
      alternative_code
      dsps
      caption
      video_motion
      );

    return \%fields_need_update;
}

sub get_bk_block_data_key {'rtb_blocks'}

sub get_bk_block_data {
    my ($self, $block) = @_;

    my $public_id = $block->{'public_id'};

    my $bk_data = {};

    $bk_data->{'page_id'} = $block->{'page_id'};

    $bk_data->{'BlockModel'} = $self->accessor();

    $bk_data->{'DSPType'} = 2;    #2**1 video

    # For video direct (PI-8892)
    $bk_data->{'Sizes'} = [{Width => 0, Height => 0}];

    $bk_data->{'BlindLevel'} = $block->{'blind'};

    $bk_data->{'BlockCaption'} = $block->{'caption'};
    $bk_data->{'MultiState'}   = $block->{'multistate'};

    $bk_data->{'AlternativeCode'} = $block->{'alternative_code'};

    my $vpaid_enabled = $block->{'vpaid_enabled'};
    my $platform      = $block->{'platform'};
    my $vast_version  = $block->{'vast_version'};

    $bk_data->{'Video'} = {
        CategoryID     => $block->{'category_id'},
        MaxDuration    => $block->{'max_duration'},
        Type           => $block->{'type_bk'},
        CountPositions => $block->{'count_positions'},
        API            => $self->get_api_frameworks(
            vpaid_enabled => $vpaid_enabled,
            platform      => $platform
        ),
        Protocols => $self->get_protocols($vast_version),
        StartTime => $block->{'start_time'} // 0,
    };

    if ($self->accessor() eq 'video_an_site_inpage') {
        my $video = $bk_data->{'Video'};
        $video->{'Collapse'} = $block->{'hide_at_the_end'};
        $video->{'Repeat'}   = $block->{'auto_repeat'} unless $block->{'hide_at_the_end'};
        $video->{'Stick'}    = $block->{'stick_outstream'};
        if ($block->{'stick_outstream'}) {
            $video->{'StickTo'} = $block->{'stick_to'} // 'auto';
            $video->{'ColorScheme'} = $block->{'light_color_scheme'} ? 'light' : 'dark';
        }
    }

    my $strategy                  = $block->{'strategy'};
    my $video_active              = $block->{'video_active'};
    my $video_blocked             = $block->{'video_blocked'};
    my $video_cpm                 = $block->{'video_cpm'};
    my $video_performance_active  = $block->{'video_performance_active'};
    my $video_performance_blocked = $block->{'video_performance_blocked'};
    my $video_performance_cpm     = $block->{'video_performance_cpm'};

    throw 'Do not turn off video and video-performance ads'
      if $video_active
          && $video_blocked
          && $video_performance_active
          && $video_performance_blocked;

    my $video_motion = $vpaid_enabled && $block->{'video_motion'} ? 1 : 0;

    if ($strategy == $MAX_REVENUE_STRATEGY_ID) {
        $bk_data->{'OptimizeType'} = 1;
    } else {
        $bk_data->{'OptimizeType'} = 0;
    }

    $bk_data->{'AdTypeSet'} = {
        'text'              => 0,
        'media'             => 0,
        'video'             => 1,
        'media-performance' => 0,
        'video-performance' => $vpaid_enabled ? 1 : 0,
        'video-motion'      => $video_motion,
    };

    $bk_data->{'AdType'} = {};

    if ($video_active) {
        if ($video_blocked) {
            $bk_data->{'AdTypeSet'} = {
                'text'              => 0,
                'media'             => 0,
                'video'             => 0,
                'media-performance' => 0,
                'video-performance' => $vpaid_enabled ? 1 : 0,
                'video-motion'      => $video_motion,
            };
        } else {
            throw 'video_cpm must be defined' unless $video_cpm;

            $bk_data->{'AdType'} = {
                'video' => {
                    'currency' => 'RUB',
                    'value'    => $video_cpm * 1000,
                },
            };
        }
    }

    if ($video_performance_active) {
        if ($video_performance_blocked) {
            $bk_data->{'AdTypeSet'} = {
                'text'              => 0,
                'media'             => 0,
                'video'             => 1,
                'media-performance' => 0,
                'video-performance' => 0,
                'video-motion'      => 0,
            };
        } else {
            throw 'video_performance_cpm must be defined'
              unless $video_performance_cpm;

            $bk_data->{'AdType'}{$_} = {
                'currency' => 'RUB',
                'value'    => $video_performance_cpm * 1000,
            } foreach (qw(video-performance video-motion));
        }
    }

    if (grep {$block->{type} == $_} ($INROLL, $MIDROLL, $PREROLL, $POSTROLL)) {
        if (1 == $bk_data->{'AdTypeSet'}{'video'}) {
            $bk_data->{'AdTypeSet'}{'video-non-skippable'} = 1;
            $bk_data->{'AdType'}{'video-non-skippable'}    = {
                'currency' => 'RUB',
                'value'    => $video_performance_cpm * 1000,
              }
              if $video_performance_cpm;
        }
    }

    return $bk_data;
}

# API

sub api_available_actions {
    my ($self) = @_;

    my @actions = qw(
      delete
      restore
      edit
      start
      stop
      duplicate
      );

    return @actions;
}
sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub can_add_block {
    my ($self, %opts) = @_;

    my $all_sites_for_adding = $self->get_all_sites_for_adding(
        %opts,
        filter             => [id => '=' => $opts{'page_id'}],
        exclude_multistate => [],
    );

    throw Exception::Validation::BadArguments gettext('Can not find site')
      unless @$all_sites_for_adding;

    if (!$self->check_feature_on_page($opts{page_id}, 'instream_vmap')) {
        throw Exception::Validation::BadArguments gettext("Missed required field '%s'", 'category_id')
          unless defined $opts{'category_id'};

        my %categories =
          map {$_->{'id'} => $_} @{$all_sites_for_adding->[0]{'categories'}};

        if (!exists($categories{$opts{'category_id'}})) {
            throw Exception::Validation::BadArguments gettext('The site was not selected to participate in Ya.Video')
              if $self->video_an_site_categories->is_yandex_video_search($opts{'category_id'})
                  && !defined($all_sites_for_adding->[0]{'partner_type'});

            throw Exception::Validation::BadArguments gettext('The site was not selected to participate in Yahosting')
              if $self->video_an_site_categories->is_yahosting($opts{'category_id'})
                  && (!defined($all_sites_for_adding->[0]{'partner_type'})
                      || $all_sites_for_adding->[0]{'partner_type'} ne 'yahosting');

            throw Exception::Validation::BadArguments gettext('Can not add block on category with id %s',
                $opts{'category_id'});
        }

        throw Exception::Validation::BadArguments gettext("Missed required field '%s'", 'type')
          unless defined $opts{'type'};

        throw Exception::Validation::BadArguments gettext(
'For one category of the site, you can create 1 inpage, 1 pre, 1 post, 30 mid and 1 pause blocks. For this section of the site has already created the maximum number of blocks of this type.'
        ) unless in_array($opts{'type'}, [map {$_->{'id'}} @{$categories{$opts{'category_id'}}->{'block_types'}}]);
    }
}

sub check_type {
    my ($self, $type) = @_;

    throw Exception gettext('Type must be defined') unless defined($type);

    return exists($VIDEO_BLOCK_TYPES->{$type}) && in_array($type, $self->get_types_ids());
}

sub get_all_sites_for_adding {
    my ($self, %opts) = @_;

    my @filters = ();
    push(@filters, $opts{'filter'}) if defined($opts{'filter'});

    my $incompatible_platforms = $self->get_incompatible_platforms();
    push(@filters, ['platform', '<>', $incompatible_platforms])
      if @$incompatible_platforms;

    my $pages = $self->SUPER::get_all_campaigns_for_adding(
        %opts,
        fields => [qw(platform id domain categories partner_type)],
        (scalar @filters ? (filter => ['AND', \@filters]) : ()),
        order_by => ['domain'],
    );

    if ($self->check_feature_on_page($opts{page_id}, 'instream_vmap')) {
        return $pages;
    }

    my $types = $self->get_types();

    my $block_types_on_page = $self->video_an_site->get_block_types_on_pages([map {$_->{'id'}} @$pages], $opts{'id'});

    my $max_blocks_type = $self->get_max_blocks_by_type();

    my $without_yandex_video_search = $self->without_yandex_video_search();

    my @result = ();
    foreach my $row (@$pages) {
        my $categories = delete($row->{'categories'});

        $row->{'categories'} = [];
        foreach my $category (@$categories) {
            next
              if $category->{'archived'}
                  || (   $without_yandex_video_search
                      && $self->video_an_site_categories->is_yandex_video_search($category->{'id'}));

            $category->{'block_types'} = [
                grep {
                         !$block_types_on_page->{$row->{'id'}}{$category->{'id'}}
                      || !$block_types_on_page->{$row->{'id'}}{$category->{'id'}}{$_->{'id'}}
                      || $block_types_on_page->{$row->{'id'}}{$category->{'id'}}{$_->{'id'}} <
                      $max_blocks_type->{$_->{'id'}}
                  } @$types
            ];

            push(@{$row->{'categories'}}, $category)
              if @{$category->{'block_types'}};
        }

        push(@result, $row) if @{$row->{'categories'}};
    }

    return \@result;
}

sub get_api_frameworks {
    my ($self, %opts) = @_;

    my $vpaid_enabled = $opts{'vpaid_enabled'};
    my $platform      = $opts{'platform'};

    my @api = ();

    #https://st.yandex-team.ru/PI-9287
    if ($vpaid_enabled) {
        if ($platform == $VIDEO_PLATFORM_FLASH) {
            push(@api, 2);
        } elsif ($platform == $VIDEO_PLATFORM_HTML5) {
            push(@api, 2, 3);
        } else {
            push(@api, 3);
        }
    }

    return \@api;
}

sub get_bk_common_data {
    return ();
}

sub get_dsp_rules {
    my ($self) = @_;

    return [
        @{$self->SUPER::get_dsp_rules()},
        # vpaid_enabled
        Partner::DSP::Rule->new(
            available_dsps => Partner::DSP::Rule::Part->new(
                sub => sub {
                    my ($block) = @_;

                    if ($block->{type} && $OVERLAY_TYPES->{$block->{type}}) {
                        return limit => ['id', 'IN', [$DSP_DIRECT_ID]];
                    } else {
                        return extend => Partner::DSP::Rule::Filter->zero();
                    }
                },
            ),
            default_dsps => Partner::DSP::Rule::Part->new(
                depends_on => {page => [qw(vpaid_enabled)],},
                sub        => sub {
                    my ($block, $page) = @_;

                    if ($block->{type} && $OVERLAY_TYPES->{$block->{type}}) {
                        return limit => ['id', 'IN', [$DSP_DIRECT_ID]];
                    } elsif (not defined $page->{vpaid_enabled} or $page->{vpaid_enabled}) {
                        return extend => ['id', 'IN', \@VIDEO_DEFAULT_DSPS];
                    } else {
                        return extend => ['id', 'IN', arrays_difference(\@VIDEO_DEFAULT_DSPS, [$DSP_DIRECT_ID])];
                    }
                },
            ),
            turn_on_dsp => Partner::DSP::Rule::Part->new(
                depends_on => {dsp => [qw(id)]},
                sub        => sub {
                    my ($dsp) = @_;

                    if ($dsp->{id} == $DSP_DIRECT_ID) {
                        return limit => ['vpaid_enabled', '=', 1];
                    } else {
                        return limit => ['type', '<>', [sort {$a <=> $b} keys %$OVERLAY_TYPES]];
                    }
                },
            ),
        ),
    ];
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'page_id'}) {
        $result{'page_id'} = $self->get_all_sites_for_adding(
            %{$opts->{attributes}},
            ($opts->{page_filter} ? (filter   => $opts->{page_filter}) : ()),
            ($opts->{page_limit}  ? (limit    => $opts->{page_limit})  : ()),
            ($opts->{page_sort}   ? (order_by => $opts->{page_sort})   : ()),
        );
    }

    if ($need_fields->{'dsps'}) {
        my $block = {
            $opts->{attributes}{block_type} ? (type => $opts->{attributes}{block_type} // '')
            : (),
            $opts->{attributes}{page_id} ? (page_id => $opts->{attributes}{page_id})
            : ()
        };

        $result{'dsps_available'} = $self->get_available_dsps($block);
        $result{'dsps_default'}   = $self->get_default_dsps($block);
    }

    if ($need_fields->{'strategy'}) {
        $result{'strategy'} = $self->get_strategies_specification->get_strategies_types($self, $opts->{attributes});
    }

    if ($need_fields->{'type'}) {
        $result{'type'} = $self->get_types();
    }

    if ($need_fields->{'max_duration'}) {
        $result{'max_duration'} = \@MAX_DURATION;
    }

    return \%result;
}

sub get_fields_depends {
    return {
        #если поменялось поле из ключ, то
        #нужно перезапросить поля из значения
        depends => {},
        #для поля из ключа обязятельно нужно передать поля из значения
        required => {},
    };
}

sub get_incompatible_platforms {[]}

sub get_max_blocks_by_type {
    return clone(\%MAX_BLOCKS_BY_TYPE);
}

sub get_path_by_category_id {
    my ($self, $page_id, $category_id) = @_;

    return '' unless defined $category_id;

    my $categories = $self->video_an_site->get($page_id, fields => ['categories'])->{'categories'};

    my @sort_categories = sort {$b->{'id'} <=> $a->{'id'}} @$categories;

    my @path    = ();
    my $find_id = $category_id;
    foreach (@sort_categories) {
        if ($_->{'id'} == $find_id) {
            push(@path, $_->{'name'});
            $find_id = $_->{'parent_id'};
            last if $find_id == 0;
        }
    }

    return join('/', reverse(@path));
}

sub get_protocols {
    my ($self, $vast_version) = @_;

    #https://st.yandex-team.ru/PI-9314
    my %protocols = (
        2 => [2, 5],
        3 => [2, 3, 5, 6],
    );

    return $protocols{$vast_version};
}

sub get_strategies {
    my ($self, $obj) = @_;

    my $strategies = {
        $MIN_CPM_STRATEGY_ID      => {},
        $MAX_REVENUE_STRATEGY_ID  => {},
        $SEPARATE_CPM_STRATEGY_ID => {
            fields        => [qw(video video_performance)],
            applicable_to => get_filters_not_overlay_instream_blocks(),
        },
    };

    return $strategies;
}

sub get_types {
    my ($self) = @_;

    my $types = clone($VIDEO_BLOCK_TYPES);

    my %type_ids = map {$_ => TRUE} @{$self->get_types_ids()};

    unless ($self->check_rights('video_an_site_instream_can_add_overlay')) {
        delete $type_ids{$_} foreach (keys %$OVERLAY_TYPES);
    }

    return [
        map {{id => $_, label => $types->{$_}{'label'}(), bk => $types->{$_}{'bk'}}}
        sort {$a <=> $b} grep {$type_ids{$_}} keys(%$types)
    ];
}

sub related_models {
    my ($self) = @_;

    return {
        %{$self->SUPER::related_models()},
        block_dsps => {

            # TODO: support page_id in _dsps models and move this definition up the inheritance chain.
            accessor => 'block_dsps',
            filter   => sub {
                +{
                    page_id    => array_uniq(map {$_->{'page_id'}} @{$_[1]}),
                    is_deleted => 0
                 };
            },
            key_fields => [qw(page_id block_id)],
            value_type => 'array',
        },
    };
}

sub without_yandex_video_search {FALSE}

sub set_block_default_values {
    my ($self, $opts, $old_settings) = @_;

    my $new_settings = {%{$old_settings // {}}, %$opts};

    $new_settings;
}

sub get_ad_type_list {
    my ($self, %opts) = @_;

    return [qw(video video_performance)];
}

sub get_statistics_level {
    return $_[0]->accessor;
}

TRUE;
