package Application::Model::BlockPresets;

use qbit;

use base qw(
  Application::Model::Common
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

use Exception::Validator::Fields;

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::EditableFields
  );

sub accessor {'block_presets'}

sub db_table_name {'block_presets'}

sub get_product_name {gettext('block_presets')}

sub get_structure_model_accessors {
    my ($self) = @_;

    return {
        partner_db        => 'Application::Model::PartnerDB',
        picategories_dict => 'Application::Model::PICategoriesDict',
    };
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $FIELDS_DEPENDS;

    return {
        id        => {db => TRUE, pk => TRUE, default => TRUE, api => TRUE, type => 'string'},
        public_id => {
            default    => TRUE,
            depends_on => [qw(id)],
            get        => sub {
                $_[0]->model->public_id($_[1]);
            },
            api  => TRUE,
            type => 'string',
        },
        caption => {
            db         => TRUE,
            default    => TRUE,
            need_trim  => TRUE,
            api        => TRUE,
            type       => 'string',
            need_check => {len_min => 1, len_max => 255}
        },
        owner_id => {
            db   => TRUE,
            api  => TRUE,
            type => 'number',
        },
        settings => {
            db  => TRUE,
            get => sub {
                $_[1]->{'settings'} //= '{}';
                utf8::decode($_[1]->{'settings'});
                return from_json($_[1]->{'settings'});
            },
            api        => TRUE,
            type       => 'complex',
            need_check => {type => 'block_preset'},
        },
        multistate      => {db => TRUE, type => 'number', need_check => {type => 'int_un'}, api => 1},
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1,
        },
        fields_depends => {
            get => sub {
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
            api  => 1,
        }
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            id         => {type => 'number',},
            caption    => {type => 'text'},
            owner_id   => {type => 'number'},
            multistate => {type => 'multistate'},
        }
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => 'id',         label => gettext('Preset ID')},
        {name => 'caption',    label => gettext('Preset caption')},
        {name => 'owner_id',   label => gettext('Owner ID')},
        {name => 'multistate', label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name => d_pgettext('Preset multistate', 'New'),
        multistates => [[deleted => d_pgettext('Preset multistate', 'Archived')],],
        actions     => {
            edit    => d_pgettext('Preset action', 'Edit'),
            delete  => d_pgettext('Preset action', 'Archive'),
            restore => d_pgettext('Preset action', 'Restore'),
        },
        right_name_prefix  => $self->accessor . '_',
        right_actions      => {add => d_pgettext('Preset action', 'Add'),},
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
        ],
    };
}

sub get_available_fields {
    my ($self, $data) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_add_fields {
    my ($self, $data) = @_;

    return $self->_get_common_add_edit_fields($data);
}

sub get_editable_fields {
    my ($self, $data) = @_;

    return $self->collect_editable_fields($data);
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    return $self->_get_common_add_edit_fields($data);
}

sub _get_common_add_edit_fields {
    my ($self, $data) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              caption
              settings
              )
        ]
    );

    return $fields;
}

sub apply {
    my ($self, %opts) = @_;

    my $model      = $opts{'model'};
    my $public_ids = $opts{'public_ids'};
    my $preset     = $opts{'preset'};

    my $db_filter = $model->get_db_filter({id => $public_ids});

    my $preset_filter = $preset->get_filter($model);

    if (ref($preset_filter) eq '' && $preset_filter eq FALSE) {
        return {real_time => JSON::XS::true, result => 'CANNOT_BE_APPLIED'};
    }

    my $db_preset_filter = $model->get_db_filter($preset_filter);

    $db_filter->and($db_preset_filter);

    my $depends_for_actions = $model->get_depends_for_field('actions');

    my %blocks =
      map {$_->{'public_id'} => $_}
      @{$model->get_all(fields => [qw(public_id editable_fields), @$depends_for_actions], filter => $db_filter)};

    my $patch_orig = $preset->get_patch($model);

    my $result = {real_time => JSON::XS::true, result => 'OK'};

    foreach my $public_id (@$public_ids) {
        if (exists($blocks{$public_id})) {
            my $patch = clone($patch_orig);
            my $block = $blocks{$public_id};

            my %update = map {$_ => $patch->{$_}} grep {$block->{'editable_fields'}{$_}} keys(%$patch);

            try {
                $model->do_action($block, 'edit', %update);
            }
            catch {
                my ($exception) = @_;

                $result->{'result'} = 'ERROR';

                push(
                    @{$result->{'errors'}},
                    {
                        public_id => $block->{'public_id'},
                        info      => $exception->isa('Exception::Validator::Errors')
                        ? from_json($exception->message)
                        : [{name => [], messages => [$exception->message()]}]
                    }
                );
            };
        } else {
            $result->{'result'} = 'ERROR';

            push(
                @{$result->{'errors'}},
                {
                    public_id => $public_id,
                    info => [{name => [], messages => [gettext('Current preset cannot be applied to this block')]}],
                }
            );
        }
    }

    return $result;
}

sub check_blocks {
    my ($self, %opts) = @_;

    my $model  = $opts{'model'};
    my $filter = $opts{'filter'};
    my $preset = $opts{'preset'};

    my $db_filter = $model->get_db_filter($filter);

    my $preset_filter = $preset->get_filter($model);

    unless (ref($preset_filter) eq '' && $preset_filter eq FALSE) {
        my $db_preset_filter = $model->get_db_filter($preset_filter);

        # фильтр который выбирает не доступные блоки для текущего пресета
        $db_filter->and_not($db_preset_filter);
    }

    my $blocks = $model->get_all(fields => [qw(public_id)], filter => $db_filter);

    foreach (@$blocks) {
        $_->{'code'}    = 0;
        $_->{'message'} = gettext('Current preset cannot be applied to this block');
    }

    return $blocks;
}

# API

sub api_available_actions {
    return qw(edit delete restore);
}

sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub get_actions_depends {
    [
        qw(
          id
          multistate
          )
    ];
}

sub get_available_fields_depends {
    [qw(id)];
}

sub get_editable_fields_depends {
    [qw(id)];
}

sub make_fields_defaults {{}}

sub get_fields_depends {{}}

sub hook_preparing_fields_to_save {
    my ($self, $opts) = @_;

    $opts = $self->hook_stash->get('opts');

    if (exists($opts->{'settings'})) {

        if ($opts->{settings}{picategories_group} and $opts->{settings}{picategories_group}{picategories}) {
            $opts->{settings}{articles_group}{articles} =
              $self->picategories_dict->map_articles_from_picategories(
                $opts->{settings}{picategories_group}{picategories});
        } elsif ($opts->{settings}{articles_group} and $opts->{settings}{articles_group}{articles}) {
            $opts->{settings}{picategories_group}{picategories} =
              $self->picategories_dict->map_picategories_from_articles($opts->{settings}{articles_group}{articles});
        }

        $opts->{'settings'} = to_json($opts->{'settings'});
    }

    $self->SUPER::hook_preparing_fields_to_save($opts);
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($self->hook_stash->get('opts'));

    $opts->{'owner_id'} = $self->get_option('cur_user')->{'id'};
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_owner($filter);

    return $filter;
}

TRUE;
