package Application::Model::BlockTags;

=encoding UTF-8

=cut

use qbit;

use base qw(Application::Model::Product Application::Model::ValidatableMixin RestApi::MultistateModel);

use Exception::DB::DuplicateEntry;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

my $FIELDS_DEPENDS;

sub accessor               {'block_tags'}
sub db_table_name          {'block_tags'}
sub get_product_name       {gettext('block_tags')}
sub get_page_id_field_name {'page_id'}

sub public_id {
    return sprintf '%s_%s', $_[1]->{'tag_id'}, $_[1]->{'page_id'};
}

sub get_structure_model_accessors {
    return {
        partner_db => 'Application::Model::PartnerDB',
        all_pages  => 'Application::Model::AllPages',
    };
}

sub get_structure_model_fields {
    return {
        page_id => {
            default    => TRUE,
            db         => TRUE,
            pk         => TRUE,
            type       => 'number',
            need_check => {type => 'int_un'},
            api        => 1,
            type       => 'number',
        },
        domain => {
            label      => d_gettext('Domain'),
            depends_on => ['page_id', 'page.domain'],
            get        => sub {
                my ($self, $obj) = @_;

                return $self->{'page'}{$obj->{'page_id'}}[0] ? $self->{'page'}{$obj->{'page_id'}}[0]{'domain'} : '';
            },
            api  => 1,
            type => 'string',
        },
        tag_id => {
            default    => TRUE,
            db         => TRUE,
            pk         => TRUE,
            type       => 'number',
            need_check => {
                type => 'int_un',
                # as in docs: https://yandex.ru/support/partner2/web/products-rtb/partner-code.html
                max => 1_000_000_000,
            },
            api         => 1,
            type        => 'number',
            adjust_type => 'str',
        },
        caption => {
            default    => TRUE,
            db         => TRUE,
            need_check => {
                type     => 'scalar',
                len_min  => 1,
                len_max  => 255,
                optional => TRUE,
            },
            api  => 1,
            type => 'string',
        },
        public_id => {
            db         => TRUE,
            need_check => {
                check => sub {
                    my ($qv, $public_id) = @_;

                    throw Exception::Validator::Fields gettext('No Page ID') unless $qv->data->{page_id};
                    throw Exception::Validator::Fields gettext('No Tag ID')  unless defined($qv->data->{tag_id});

                    my $expected = $qv->data->{tag_id} . '_' . $qv->data->{page_id};
                    throw Exception::Validator::Fields gettext('Incorrect public_id') unless $public_id eq $expected;
                },
            },
            api  => 1,
            type => 'string',
        },
        multistate => {
            db   => TRUE,
            api  => 1,
            type => 'number',
        },
        multistate_name => {
            depends_on => ['multistate'],
            label      => d_gettext('Multistate name'),
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            api  => 1,
            type => 'string',
        },
        editable_fields => {
            get => sub {
                $_[0]->model->get_editable_fields();
            },
            api  => 1,
            type => 'complex',
        },
        available_fields => {
            get => sub {
                return $_[0]->model->get_available_fields();
            },
            api  => 1,
            type => 'complex',
        },
        fields_depends => {
            get => sub {
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
        },
        actions => {
            label => d_gettext('Actions'),
            #depends_on => [qw(is_standart_report multistate)],
            get => sub {
                $_[0]->model->get_actions($_[1]);
            },
            api  => 1,
            type => 'complex',
        },
    };
}

sub get_structure_model_filter {
    return {
        db_accessor => 'partner_db',
        fields      => {
            page_id   => {type => 'number', label => d_gettext('Page ID')},
            tag_id    => {type => 'number', label => d_gettext('Tag ID')},
            public_id => {
                type   => 'publicid',
                label  => d_gettext('Public ID'),
                regexp => {'^([0-9]+)_([0-9]+)\z' => ['tag_id', 'page_id'],}
            },
            multistate => {type => 'multistate', label => d_gettext('Status')},
        },
    };
}

sub get_structure_multistates_graph {
    return {
        empty_name => d_pgettext('Block tags', 'New'),
        multistates =>
          [['new' => d_pgettext('Block tags', 'New')], ['deleted' => d_pgettext('Block tags', 'Deleted')],],
        actions => {
            add    => d_pgettext('Block tags', 'Add'),
            edit   => d_pgettext('Block tags', 'Edit'),
            delete => d_pgettext('Block tags', 'Delete'),
        },
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['new'],
            },
            {
                action      => 'edit',
                from        => 'new or deleted',
                reset_flags => ['deleted'],
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
        ],
    };
}

sub get_available_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} keys(%{$self->get_model_fields});

    foreach (qw(multistate)) {
        delete($fields{$_});
    }

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = {
        page_id => TRUE,
        tag_id  => TRUE,
        caption => TRUE,
    };

    return $fields;
}

sub get_editable_fields {
    my ($self) = @_;

    my $fields = {caption => TRUE};

    return $fields;
}

sub add {
    my ($self, %opts) = @_;

    $opts{public_id} = ($opts{tag_id} // '') . '_' . ($opts{page_id} // '');

    $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%{$self->get_add_fields}), 'public_id']),
    );

    if (
        $self->get_all(
            filter => ['AND', [{'public_id' => $opts{public_id}}, {'multistate' => 'deleted'}]],
            fields => [qw(public_id)]
        )->[0]
       )
    {
        $self->do_action(\%opts, 'edit', %opts);
    } else {
        my $id;
        try {
            $self->partner_db->transaction(
                sub {
                    $id = $self->partner_db->block_tags->add(\%opts);

                    # This should be checked here while we (temporary) have this page_id in block_tags table.
                    # Then it will be rolled back if proven wrong.
                    throw Exception::Validation::BadArguments gettext('Non-existing Page ID')
                      unless $self->_check_page_for_existence($opts{page_id});

                    $self->do_action($id, 'add', %opts);
                }
            );
        }
        catch Exception::DB::DuplicateEntry with {
            throw Exception::Validation::BadArguments gettext('Caption for Page ID %s Tag ID %s already exists.',
                $opts{page_id}, $opts{tag_id});
        };
    }
    return $opts{public_id};
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;

    $obj = $self->_get_object_fields($obj, [qw(page_id tag_id caption editable_fields)]);
    my $editable_fields = $obj->{'editable_fields'};

    my $old_obj_settings = $self->get($obj, fields => [sort keys(%$editable_fields)]);

    my %new_obj_settings = (%$old_obj_settings, %opts);

    $self->app->validator->check(
        data  => \%new_obj_settings,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%$editable_fields)]),
    );

    if (%opts) {
        $self->partner_db->block_tags->edit($obj, \%opts);
    }

    return TRUE;
}

sub delete {
    my ($self, $block_tag_id) = @_;

    $self->do_action($self->_split_id($block_tag_id), 'delete');

    return TRUE;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_current_user_pages($filter);

    return $filter;
}

sub related_models {
    return {
        page => {
            accessor => 'all_pages',
            filter   => sub {
                return {page_id => array_uniq(map {$_->{'page_id'} // ()} @{$_[1]})};
            },
            key_fields => ['page_id'],
            value_type => 'array',
        },
    };
}

sub _check_page_for_existence {
    my ($self, $page_id) = @_;

    my $filter = $self->partner_db->filter([page_id => '=' => \$page_id]);

    $filter = $self->limit_filter_by_current_user_pages($filter);

    my $res = $self->partner_db->query->select(
        table  => $self->partner_db->block_tags,
        fields => {result => \1},
        filter => $filter,
    )->limit(1)->get_all();

    return @$res > 0;
}

sub _split_id {
    my ($self, $id) = @_;

    return defined($id) && !ref($id) && $id =~ /^(\d+)_(\d+)\z/
      ? {tag_id => $1, page_id => $2}
      : $id;
}

#
# API
#

sub api_available_actions {
    return qw(
      edit
      delete
      );
}

sub api_can_add         {TRUE}
sub api_can_edit        {TRUE}
sub api_can_delete      {TRUE}
sub api_check_public_id {$_[1] =~ /^\d+_\d+$/}

1;
