package Application::Model::BusinessBlocks;

=pod

Модель Блоков Бизнес правил (#PI-10891)

=cut

use qbit;

use base qw( Application::Model::Common );

use Utils::PublicID qw( split_block_public_id );

sub accessor         {'business_blocks'}
sub db_table_name    {'business_blocks'}
sub get_product_name {'business_blocks'}

sub get_structure_model_accessors {
    my ($class) = @_;
    return {partner_db => 'Application::Model::PartnerDB',};
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $model_fields = {
        # TODO: везде вызывать метод родителя
        #%{$self->SUPER::get_structure_model_fields()},
        rule_id => {
            default => TRUE,
            db      => TRUE,
            pk      => TRUE,
            label   => d_gettext('Rule ID'),
        },
        page_id => {
            db    => TRUE,
            pk    => TRUE,
            label => d_gettext('Page ID'),
        },
        block_id => {
            db    => TRUE,
            pk    => TRUE,
            label => d_gettext('Login'),
        },
        block_accessor => {
            db    => TRUE,
            label => d_gettext('Model'),
        },
        multistate => {
            db         => TRUE,
            label      => d_gettext('Status'),
            need_check => {type => 'int_un'},
        },
        multistate_name => {
            depends_on => ['multistate'],
            label      => d_gettext('Multistate name'),
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
        },
        is_deleted => {
            db    => TRUE,
            label => d_gettext('Deleted'),
        },
        send_time => {
            db    => TRUE,
            label => d_gettext('Time applying updates'),
            type  => 'string',
            api   => 1,
        },
        update_time => {
            db    => TRUE,
            label => d_gettext('Time applying updates'),
            type  => 'string',
            api   => 1,
        },
        #---
        block_public_id => {
            depends_on => [qw( page_id  block_id  block_accessor )],
            label      => d_gettext('Block ID'),
            get        => sub {
                my ($model, $row) = @_;
                my ($page_id, $block_id, $block_accessor) = @$row{qw( page_id block_id  block_accessor )};

                # TODO: Utils::PublicID
                return sprintf '%s%d-%d', $_[0]->model->app->$block_accessor->public_id_prefix(), $page_id, $block_id;
            },
        },

    };

    return $model_fields;
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        # TODO: везде вызывать метод родителя
        #%{$self->SUPER::get_structure_model_filter()},
        db_accessor => 'partner_db',
        fields      => {
            rule_id         => {type => 'number',     label => d_gettext('Rule ID')},
            page_id         => {type => 'number',     label => d_gettext('Page ID')},
            block_id        => {type => 'number',     label => d_gettext('Block ID')},
            is_deleted      => {type => 'boolean',    label => d_gettext('Block ID')},
            multistate      => {type => 'multistate', label => d_gettext('ID')},
            update_time     => {type => 'date',       label => d_gettext('Time queuing for updating')},
            block_public_id => {
                type   => 'publicid',
                label  => d_gettext('Block ID'),
                regexp => {
                    # TODO: Utils::PublicID
                    '^(?:[A-Z-]+)-([0-9]+)-([0-9]+)\z' => ['page_id', 'block_id'],
                    '^([0-9]+)-([0-9]+)\z'             => ['page_id', 'block_id'],
                }
            },
        }
    };
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    my $multistates_graph = {
        # TODO: везде вызывать метод родителя
        # %{$self->SUPER::get_structure_multistates_graph()},

        empty_name  => d_pgettext('Business block status', 'New'),
        multistates => [
            [need_update => d_pgettext('Business block status', 'Need update')],
            [updating    => d_pgettext('Business block status', 'Updating')],
        ],
        actions => {
            set_need_update =>
              {label => d_pgettext('Business block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update =>
              {label => d_pgettext('Business block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update =>
              {label => d_pgettext('Business block action', 'Stop update'), dont_write_to_action_log => TRUE},
        },
        right_name_prefix  => $self->accessor . '_',
        right_actions      => {},
        multistate_actions => [
            # set_need_update, start_update, stop_update
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };

    return $multistates_graph;
}

sub get_cnt {
    my ($self, $rules_id_arr) = @_;

    my $data = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => {
            rule_id => '',
            cnt     => {count => ['block_id']}
        },
        filter => [AND => [[rule_id => 'IN' => \$rules_id_arr], [is_deleted => '=' => \0],]],
    )->group_by('rule_id')->get_all();

    return {map {$_->{'rule_id'} => $_->{'cnt'}} @$data};

}

##### TODO: использовать базовый
sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
    $self->multistates_graph($self->get_structure_multistates_graph());
}

sub update_blocks {
    my ($self, $rule_id, $blocks_to_add, $is_start_stopped) = @_;
    #    $blocks_to_add = {
    #       'R-A-207848-1' => 'context_on_site_rtb',
    #       'D-A-45551-1' => 'context_on_site_direct'
    #     },

    my $cur_blocks = $self->get_all(
        fields => [qw(block_public_id)],
        filter => {'rule_id' => $rule_id},
    );

    my $need_update_bit = $self->get_multistate_by_name('need_update');

    my @affected_blocks_filter = ();
    my $affected_blocks_keys   = {};

    # Проставляем is_deleted
    if ($blocks_to_add) {
        my @filter = ();
        foreach (@$cur_blocks) {
            my $public_id = $_->{'block_public_id'};

            next if exists($blocks_to_add->{$public_id});

            my (undef, $page_id, $block_id) = split_block_public_id($public_id);
            push @filter,
              ['AND', [[rule_id => '=' => \$rule_id], [page_id => '=' => \$page_id], [block_id => '=' => \$block_id]]];

            # sprintf('%s-%s', $page_id, $block_id)
            unless ($affected_blocks_keys->{$page_id, $block_id}++) {
                push @affected_blocks_filter, ['AND', [[page_id => '=' => \$page_id], [block_id => '=' => \$block_id]]];
            }
        }

        $self->partner_db_table()->edit($self->partner_db->filter(['OR', \@filter]), {is_deleted => 1}) if @filter;
    }

    $blocks_to_add //= {};

    # Добавляем новые блоки
    my @buffer = ();
    foreach my $public_id (keys %$blocks_to_add) {
        my (undef, $page_id, $block_id) = split_block_public_id($public_id);

        push @buffer,
          {
            rule_id        => $rule_id,
            page_id        => $page_id,
            block_id       => $block_id,
            block_accessor => $blocks_to_add->{$public_id},
            is_deleted     => 0,
          };

        if (@buffer >= 10_000) {
            $self->partner_db_table()->add_multi(\@buffer, duplicate_update => TRUE);
            @buffer = ();
        }

        unless ($affected_blocks_keys->{$page_id, $block_id}++) {
            push @affected_blocks_filter, ['AND', [[page_id => '=' => \$page_id], [block_id => '=' => \$block_id]]];
        }

    }

    if ($is_start_stopped) {
        foreach (@$cur_blocks) {
            my $public_id = $_->{'block_public_id'};
            my (undef, $page_id, $block_id) = split_block_public_id($public_id);

            unless ($affected_blocks_keys->{$page_id, $block_id}++) {
                push @affected_blocks_filter, ['AND', [[page_id => '=' => \$page_id], [block_id => '=' => \$block_id]]];
            }
        }
    }

    $self->partner_db_table()->add_multi(\@buffer, duplicate_update => TRUE) if @buffer;

    # Проставляем need_update всем блокам, в том числе в других правилах
    $self->partner_db_table()->edit(
        $self->partner_db->filter(['OR', \@affected_blocks_filter]),
        {
            multistate  => $need_update_bit,
            update_time => curdate(oformat => 'db_time')
        }
    ) if @affected_blocks_filter;

    return 1;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_processing_before_validation($opts);

    $opts->{'update_time'} //= curdate(oformat => 'db_time');
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{'send_time'} //= curdate(oformat => 'db_time');
}

TRUE;
