package Application::Model::BusinessRules;

use qbit;

use base qw(
  Application::Model::Common
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields);

use Utils::PublicID qw( validate_public_ids  check_public_ids  split_block_public_id);

use Exception::Validation::BadArguments;

use Exception::Validator::Fields;
use Exception::Validation::BadArguments;

use PiConstants qw($MAX_CPM $MYSQL_DEFAULT_DATETIME);

sub accessor         {'business_rules'}
sub db_table_name    {'business_rules'}
sub get_product_name {gettext('business_rules')}

our $CONDITION_TYPES = {
    browsers => {
        label       => d_gettext('Browser'),
        type        => 'technologies',
        defaults    => \&get_defaults_for__browsers,
        keyword     => 'browser-name',
        values_type => 'string',
        values      => {
            chrome        => {val => 2,  label => 'Chrome'},
            explorer      => {val => 3,  label => 'Internet Explorer'},
            opera         => {val => 4,  label => 'Opera'},
            safari        => {val => 5,  label => 'Safari'},
            firefox       => {val => 6,  label => 'Firefox'},
            yandex_bro    => {val => 7,  label => 'Yandex.Browser'},
            uc_bro        => {val => 28, label => 'UCBrowser'},
            edge          => {val => 55, label => 'Edge'},
            amigo         => {val => 56, label => 'Amigo'},
            chrome_mobile => {val => 86, label => 'Chrome Mobile'},
            safari_mobile => {val => 88, label => 'Safari Mobile'},
        }
    },
    devices => {
        label       => d_gettext('Device'),
        type        => 'technologies',
        defaults    => \&get_defaults_for__devices,
        keyword     => undef,
        values_type => 'string',
        values      => {
            desktop => {keyword => 'device-is-desktop', val => 1, label => d_gettext('Desktop')},
            tablet  => {keyword => 'device-is-tablet',  val => 1, label => d_gettext('Tablet')},
            mobile  => {keyword => 'device-is-mobile',  val => 1, label => d_gettext('Mobile')},
            smarttv => {keyword => 'device-is-tv',      val => 1, label => d_gettext('SmartTV')},
        },
    },
    query_args => {
        label            => d_gettext('Query arguments'),
        type             => 'dict-value',
        defaults         => \&get_defaults_for__query_args,
        keyword          => 'query',
        values_type      => 'string',
        available_values => {adb_enabled => ['1']},
        bk_expression    => sub {
            my ($param, $val) = @_;
            utf8::encode($val) if utf8::is_utf8($val);
            return [$param, ($param =~ /^puid\d+\z/ ? 'match' : 'equal'), $val];
        },
    },
    systems => {
        label       => d_gettext('Operation systems'),
        type        => 'technologies',
        defaults    => \&get_defaults_for__systems,
        keyword     => 'detailed-device-type',
        values_type => 'string',
        values      => {
            'android'       => {val => 2,  label => 'Android'},
            'ios'           => {val => 3,  label => 'iOS'},
            'windows_phone' => {val => 4,  label => 'Windows Phone'},
            'linux'         => {val => 15, label => 'Linux'},
            'macos'         => {val => 16, label => 'MacOS'},
            'windows'       => {val => 33, label => 'Windows'},
        }
    },

    # Для этих справочников нет - произвольный выбор
    headers => {
        label       => d_gettext('HTTP Headers'),
        type        => 'key-value',
        keyword     => '??noboby-knows-yet??',
        values_type => 'string',
        # пока никому не даем, должно быть 'context_on_site_adblock_view' (#PI-11720)
        check_right => 'nobody-has-this-cool-feature-yet',
    },
    regions => {
        label       => d_gettext('Regions'),
        keyword     => 'reg-id',
        values_type => 'string',
        type        => 'geo',
    },
    urls => {
        label         => d_gettext('URL'),
        keyword       => 'referer',
        values_type   => 'string',
        type          => 'url',
        bk_expression => sub {
            my ($param, $val) = @_;

            return [$param, 'like', '%' . lc($val) . '%'];
        },
    },

    # А здесь все типы вместе
    conditions => {
        label    => d_gettext('Conditions'),
        type     => 'others',
        defaults => \&get_defaults_for__conditions,
    }
};

sub public_id {
    my ($self, $obj) = @_;

    return $obj->{'rule_id'};
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        partner_db      => 'Application::Model::PartnerDB',
        product_manager => 'Application::Model::ProductManager',
        business_blocks => 'Application::Model::BusinessBlocks',
        users           => 'Application::Model::Users',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = [
        # TODO: везде вызывать метод родителя
        # @{$self->SUPER::get_structure_rights_to_register()},
        {
            name        => 'business_rules',
            description => 'Right to manage Business rules',
            rights      => {
                (map {$self->get_description_right($_)} qw( view_field__login )),

                context_on_site_adblock_view                                 => d_gettext('Right to view Adblock'),
                $self->accessor() . '_' . 'nobody-has-this-cool-feature-yet' => '',
            },
        }
    ];

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $model_fields = {
        # TODO: везде вызывать метод родителя
        #%{$self->SUPER::get_structure_model_fields()},
        actions => {
            label      => d_gettext('Actions'),
            depends_on => [qw(rule_id multistate)],
            get        => sub {
                $_[0]->model->get_actions($_[1]);
            },
            type => 'complex',
            api  => 1,
        },
        rule_id => {
            default => TRUE,
            db      => TRUE,
            pk      => TRUE,
            label   => d_gettext('Rule ID'),
            type    => 'number',
            api     => 1,
        },
        # TODO: нужно придумать более лаконичную констуркцию для простых алиасов
        public_id => {    # нужно для API
            depends_on => ['rule_id'],
            get        => sub {
                return $_[1]->{'rule_id'};
            },
            type => 'string',
            api  => 1,
        },
        fields_depends => {    # нужно для API хз зачем
            get => sub {
                return $_[0]->model->get_fields_depends();
            },
            type => 'complex',
            api  => 1,
        },
        status => {
            depends_on => ['rule_id'],
            get        => sub {
                return 'sync';
            },
            type => 'string',
            api  => 1,
        },
        owner_id => {
            db         => TRUE,
            label      => d_gettext('Owner ID'),
            type       => 'number',
            need_check => {type => 'int_un'},
            api        => 1,
        },
        login => {
            depends_on => ['owner_id', 'users.login'],
            label      => d_gettext('Login'),
            get        => sub {
                my ($model, $row) = @_;
                return $model->{'users'}->{$row->{'owner_id'}}->{'login'} // '';
            },
            type       => 'string',
            need_check => {
                type    => 'scalar',
                len_max => 40,
                check   => sub {
                    my ($qv, $value) = @_;

                    # менеджеру нужно передеавать login (но не при редактировании),
                    # а партнер может передавать только свой логин или не передавать вовсе (это рулится флагом "optinal" в need_check)
                    unless ($qv->app->check_short_rights('add_other')) {
                        my $cur_login = $qv->app->app->get_option('cur_user', {})->{'login'} // '';
                        throw Exception::Validator::Fields gettext('Login is not yours "%s"', $value)
                          unless $value eq $cur_login;
                    }
                },
            },
            api => 1,
        },
        blocks => {
            depends_on => ['rule_id'],
            get        => sub {
                $_[0]->{'__BLOCKS__'}{$_[1]->{'rule_id'}} // [];
            },

            type       => 'array',
            sub_type   => 'string',
            need_check => {
                optional => TRUE,
                type     => 'array',
                all      => {
                    type    => 'scalar',
                    len_max => 32,
                    # NOTE! это взывает ошибку Unhandled type: REGEXP at /usr/share/perl5/Devel/Cycle.pm line 107
                    # regexp  => qr/^(?:[A-Z-]+\-)?[0-9]+-[0-9]+\z/,
                },
                # TODO вынести в ADD и EDIT
                check => sub {
                    my ($qv, $value) = @_;

                    # fix Unhandled type: REGEXP at /usr/share/perl5/Devel/Cycle.pm line 107
                    my $wrong_public_ids = check_public_ids($value, 'prefix-?page-block');
                    throw Exception::Validator::Fields gettext('Wrong public_id "%s"', join(', ', @$wrong_public_ids))
                      if @$wrong_public_ids;

                    my ($found, $not_found) = ({}, []);
                    try {
                        ($found, $not_found) = validate_public_ids($qv->app->app, $value);
                    }
                    catch Exception::Validation::BadArguments with {
                        my ($e) = @_;
                        throw Exception::Validator::Fields $e->message;
                    };

                    throw Exception::Validator::Fields gettext('Not found %s', join(', ', @$not_found)) if @$not_found;

                    # Кладем полученные данные в stash валидатора (чтобы еще раз не ходить в базу в add)
                    $qv->set_stash('blocks', $found);
                  }
            },
            api => 1,
        },
        blocks_count => {
            depends_on => ['rule_id'],
            get        => sub {
                $_[0]->{'__BLOCKS_COUNT__'}{$_[1]->{'rule_id'}} // 0;
            },
            label => d_gettext('Blocks amount'),
            type  => 'number',
            api   => 1,
        },

        caption => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Business rule caption'),
            type       => 'string',
            need_check => {
                type    => 'scalar',
                len_max => 255,
            },
            api => 1,
        },
        cpm => {
            default    => TRUE,
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                type => 'int_un',
                min  => 1,
                max  => $MAX_CPM,
                msg  => d_gettext("cpm value must be from '1' to '%d'", $MAX_CPM),
            },
        },
        conditions => {
            default => TRUE,
            db      => TRUE,
            get     => sub {
                my ($model, $row) = @_;
                my $data = $row->{conditions} ? from_json($row->{conditions}) : {};
                return $data;
            },
            type     => 'complex',
            fix_type => sub {
                my ($model, $value) = @_;

                $value //= {
                    #   "browsers": [2,88],
                    #   "devices": ["desktop", "smarttv"],
                    #   "query_args" : {"puid1": ["foo", "bar"], "puid63": ["baz"]},
                    #   "systems": [15,33],
                    #   "headers": {"content-type": ["text/plain"], "accept": ["application/json"]}
                    #   "regions": [1, 255],
                    #   "urls": ["https://foo.com", "bar.net"],
                };

                foreach my $type (keys %$value) {
                    if (grep {$type eq $_} qw(browsers systems regions)) {
                        # Приводим строки числу для сериализации в JSON
                        map {$_ += 0} @{$value->{$type}};
                    }
                }
                return $value;
            },
            need_check => {
                optional => TRUE,
                type     => 'busines_rule_condition'
            },
            api => 1,
        },
        multistate => {
            db         => TRUE,
            label      => d_gettext('Status'),
            type       => 'number',
            need_check => {type => 'int_un'},
            api        => 1,
        },
        multistate_name => {
            depends_on => ['multistate'],
            label      => d_gettext('Multistate name'),
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1,
        },
        position => {
            default             => TRUE,
            default_for_actions => [qw(move)],
            db                  => TRUE,
            label               => d_gettext('Priority'),
            type                => 'number',
            need_check          => {
                type => 'int_un',
                max  => 10_000,
            },
            api => 1,
        },
        create_date => {
            db    => TRUE,
            label => d_gettext('Create date'),
            type  => 'string',
            api   => 1,
        },
        update_time => {
            db    => TRUE,
            label => d_gettext('Time applying updates'),
            type  => 'string',
            api   => 1,
        },
        is_deleted => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'deleted');
            },
            type => 'boolean',
            api  => 1,
        },
        is_stopped => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'stopped');
            },
            type => 'boolean',
            api  => 1,
        },
    };

    return $model_fields;
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        # TODO: везде вызывать метод родителя
        #%{$self->SUPER::get_structure_model_filter()},
        db_accessor => 'partner_db',
        fields      => {
            rule_id    => {type => 'number',     label => d_gettext('Rule ID')},
            caption    => {type => 'text',       label => d_gettext('Caption')},
            multistate => {type => 'multistate', label => d_gettext('ID')},

            users => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'owner_id',
                fk_field       => 'id',
            },

            blocks => {
                type            => 'subfilter',
                model_accessor  => 'business_blocks',
                field           => 'rule_id',
                fk_field        => 'rule_id',
                submodel_filter => {is_deleted => 0},
            },
            block_public_id => {
                # TODO: publicid
                # ( сейчас это вызовет добавление фильтра "[block_public_id', '=','100500']" при /v1/business_rules/100500 )
                type      => 'text',
                db_filter => sub {
                    my ($model, $filter) = @_;
                    $filter = ['AND', [$filter, {is_deleted => 0}]];
                    return [
                        'rule_id',
                        '= ANY',
                        $model->business_blocks->query(
                            fields => $model->business_blocks->_get_fields_obj(['rule_id']),
                            filter => $model->business_blocks->get_db_filter($filter),
                        )
                    ];
                  }
            },
        }
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => 'rule_id',         label => gettext('Rule ID')},
        {name => 'multistate',      label => gettext('Multistate')},
        {name => 'caption',         label => gettext('Business rule caption')},
        {name => 'blocks.page_id',  label => gettext('Page ID')},
        {name => 'block_public_id', label => gettext('Block ID')},
        (
            $self->check_short_rights('view_field__login')
            ? ({name => 'users.login', label => gettext('Login')})
            : ()
        )
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    my $multistates_graph = {
        # TODO: везде вызывать метод родителя
        # %{$self->SUPER::get_structure_multistates_graph()},

        empty_name  => d_pgettext('Business rule status', 'New'),
        multistates => [
            [working => d_pgettext('Business rule status', 'Working')],
            [stopped => d_pgettext('Business rule status', 'Stopped')],
            [deleted => d_pgettext('Business rule status', 'Archived')],
        ],
        actions => {
            delete  => d_pgettext('Business rule action', 'Archive'),
            restore => d_pgettext('Business rule action', 'Restore'),

            edit => d_pgettext('Business rule action', 'Edit'),
            move => d_pgettext('Business rule action', 'Move'),

            start => d_pgettext('Business rule action', 'Start'),
            stop  => d_pgettext('Business rule action', 'Stop'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add => d_pgettext('Business rule action', 'Add'),

            set_need_update =>
              {label => d_pgettext('Business rule action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update =>
              {label => d_pgettext('Business rule action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update =>
              {label => d_pgettext('Business rule action', 'Stop update'), dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            ######## actions with do_action* handlers
            # add, edit, move
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['stopped'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action => 'move',
                from   => 'not stopped and not deleted',
            },
            ########## actions with just setting bits
            # start, stop
            {
                action      => 'start',
                from        => 'stopped and not deleted',
                reset_flags => ['stopped'],
                set_flags   => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
                set_flags   => ['stopped'],
            },
            # delete, restore
            {
                action      => 'delete',
                from        => 'not deleted',
                reset_flags => ['working'],
                set_flags   => ['deleted', 'stopped'],
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
        ],
    };

    return $multistates_graph;
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;
    $self->SUPER::pre_process_fields($fields, $result, %opts);

    my $rule_ids = [];
    if ($fields->need('blocks_count') || $fields->need('blocks')) {
        $rule_ids = [map {$_->{'rule_id'}} @$result];
    }

    if ($fields->need('blocks')) {

        my $blocks = $self->business_blocks->get_all(
            fields => [qw( rule_id  block_public_id )],
            filter => [AND => [{'rule_id' => $rule_ids}, {'is_deleted' => 0},]],
        );

        my $rule_blocks = $fields->{'__BLOCKS__'} = {};
        map {push @{$rule_blocks->{$_->{rule_id}}}, $_->{block_public_id}} @$blocks;
    }

    if ($fields->need('blocks_count')) {
        $fields->{'__BLOCKS_COUNT__'} = $self->business_blocks->get_cnt($rule_ids) // {};
    }
}

sub get_available_fields_depends {
    [qw(rule_id)];
}

sub get_editable_fields_depends {
    [qw(rule_id  multistate)];
}

sub get_available_fields {
    my ($self, $fields, $obj) = @_;

    my $model_fields = $self->get_model_fields;

    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();
    $self->app->delete_field_by_rights(
        \%fields,
        {
            # <Право или маска под право>            <Поля которые оно закрывает>
            $accessor . '_view_field__login' => [qw( login )],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self, $object) = @_;

    my $fields = $self->_get_common_add_edit_fields($object);

    # менеджеру нужно передеавать login (но не при редактировании),
    # а партнер может передавать только свой логин или не передавать вовсе (это рулится флагом "optinal" в need_check)
    $fields->{'login'} = 1;

    return $fields;
}

sub get_editable_fields {
    my ($self, $object) = @_;

    return {} unless $self->check_action($object, 'edit');

    my $fields = $self->_get_common_add_edit_fields($object);

    return $fields;
}

sub _get_common_add_edit_fields {
    my ($self, $object) = @_;

    my $model_fields = $self->get_model_fields;

    # Поля заполняются автоматически и их не нужно указывать при добавлении
    my $skip_auto_fields = {map {$_ => 1} qw(rule_id  multistate  owner_id  position  create_date  update_time )};

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            # Поля не из базы
            qw( blocks ),

            # Поля из базы
            grep {$model_fields->{$_}->{db} && !$skip_auto_fields->{$_}}
              keys %$model_fields
        ]
    );

    return $fields;
}

sub add {
    my ($self, %opts) = @_;

    # <COPYPASTE: общий для любой модели копипаст НАЧАЛО
    # TODO: вынести в какой-нить в общий модуль
    throw Exception::Denied gettext('Access denied')
      unless $self->check_rights($self->get_rights_by_actions('add'));

    $self->_trim_opts(\%opts);

    my $allowed_fields = $self->get_add_fields(\%opts);
    $self->check_bad_fields_in_add(\%opts, $allowed_fields);

    $self->set_default_values(\%opts);

    my $user_id = $self->_get_request_user_id($opts{login}, 'add');
    $opts{owner_id} = $user_id;

    my $qv = $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => {%$allowed_fields, owner_id => 1})
    );

    # TODO: убрать из валидатора и заменить на вызов ф-ции
    $opts{_found_blocks} = $qv->get_stash('blocks');

    # TODO: бесит что везде передается не по ссылке, но если править то везде
    my $rule_id = $self->_add(%opts);

    return $rule_id;
    # </COPYPASTE
}

sub on_action_add {
    my ($self, $obj) = @_;
    $self->_set_update_time($obj);
}

sub on_action_edit {
    my ($self, $obj, %data) = @_;

    # <COPYPASTE: общий для любой модели копипаст НАЧАЛО
    # TODO: вынести в какой-нить в общий модуль
    $self->_trim_opts(\%data);

    $self->check_bad_fields($obj, \%data);

    my $model_fields = $self->get_model_fields();

    my @need_check_fields = sort grep {$model_fields->{$_}->{'need_check'}} keys(%$model_fields);

    my $tmp_all_rights = $self->app->add_all_tmp_rights();

    my $old_settings = $self->get($obj, fields => \@need_check_fields);

    undef($tmp_all_rights);

    my $new_settings = $self->set_default_values(\%data, $old_settings);

    my $qv = $self->app->validator->check(
        data  => $new_settings,
        app   => $self,
        throw => TRUE,
        $self->get_template()
    );
    # </COPYPASTE

    # Предобработка
    $data{conditions} = to_json($data{conditions}, canonical => TRUE) if $data{conditions};

    # Достаем полученные в валидаторе данные
    my $found_blocks = $qv->get_stash('blocks');
    delete $data{blocks};

    $self->partner_db_table()->edit($obj, \%data) if %data;

    $self->_edit_position_and_blocks($obj->{rule_id}, $found_blocks);

    $self->_set_update_time($obj);

    return TRUE;
}

sub _set_update_time {
    my ($self, $obj) = @_;
    $self->partner_db_table()->edit($obj, {'update_time' => curdate(oformat => 'db_time'),});
}

sub on_action_delete {
    _on_action_stop_delete(@_);
}

sub on_action_move {
    my ($self, $rule_obj, %opts) = @_;

    my $qv = $self->app->validator->check(
        data     => \%opts,
        app      => $self,
        throw    => TRUE,
        template => {
            type => 'hash',
            # несмотря на то что 'insert_before' может быть undef (optional), он должен быть прислан явно
            must_exists => ['insert_before'],
            fields      => {
                insert_before => {
                    type     => 'int_un',
                    optional => TRUE
                }
            }
        },
    );

    my $insert_before = delete $opts{insert_before};

    $self->move_object($rule_obj, $insert_before);

    $self->_edit_position_and_blocks($rule_obj->{rule_id}, undef, undef, 1);

    return TRUE;
}

sub on_action_start {
    my ($self, $obj) = @_;
    # Двигаем в конец ( $insert_before=undef)
    $self->_edit_position_and_blocks($obj->{rule_id}, undef, 'MOVE', 1);
}

sub on_action_stop {
    _on_action_stop_delete(@_);
}

sub api_available_actions {
    my ($self) = @_;

    my $graph = $self->get_multistates_graph_definition();

    my @actons = (map {sort keys %{$graph->{$_}}} qw( actions  right_actions ));

    return @actons;
}
sub api_can_edit   {TRUE}
sub api_can_action {TRUE}
sub api_can_add    {TRUE}

sub api_get {
    my ($self, $pk, %opts) = @_;

    # Ugly hack to fix 'Unexpected fields: insert_before"'
    $opts{fields} = [grep {$_ ne 'insert_before'} @{$opts{fields}}];

    $self->SUPER::api_get($pk, %opts);
}

sub check_action {
    my ($self, $object, $action) = @_;

    return FALSE
      unless $self->check_rights($self->get_right('edit'));

    $self->SUPER::check_action($object, $action);
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_owner($filter);

    return $filter;
}

sub fix_template {
    my ($self, $qv) = @_;

    my $template = $qv->template;

    # менеджеру нужно передеавать login (но не при редактировании),
    # а партнер может передавать только свой логин или не передавать вовсе
    $template->{'fields'}->{'login'}->{'optional'} = TRUE unless $self->check_short_rights('add_other');

    $qv->template($template);
}

sub get_cnt {
    my ($self, $owner_id_arr) = @_;

    my $working_bit = $self->get_multistate_by_name('working');

    my $data = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => {
            owner_id => '',
            all      => {count => ['rule_id']},
            active   => {sum => [{'if' => [['multistate', '&', \$working_bit], \1, \0]}]},
        },
        filter => [AND => [[owner_id => 'IN' => \$owner_id_arr], [owner_id => '<>' => \0],]],
    )->group_by('owner_id')->get_all();

    return {map {$_->{'owner_id'} => $_} @$data};
}

sub get_defaults_for__browsers {
    my $values = $CONDITION_TYPES->{browsers}->{values};
    return {map {$_->{val} => $_->{label}} values %$values};
}

sub get_defaults_for__conditions {
    my ($self, $owner_id) = @_;

    # NOTE! это как бы fake login менеджером под пользователя
    my $checked_rights = [sort grep {$_} map {$CONDITION_TYPES->{$_}->{'check_right'}} keys %$CONDITION_TYPES];
    my $rights = $self->app->rbac->get_rights_by_user_id($owner_id, $checked_rights);

    return [
        map +{
            id    => $_,
            label => $CONDITION_TYPES->{$_}->{label}->(),
            type  => $CONDITION_TYPES->{$_}->{type},
            $CONDITION_TYPES->{$_}->{available_values}
            ? (available_values => $CONDITION_TYPES->{$_}->{available_values})
            : ()
        },
        sort
          grep {!$CONDITION_TYPES->{$_}->{'check_right'} || $rights->{$CONDITION_TYPES->{$_}->{'check_right'}}}
          grep {$_ ne 'conditions'}
          keys %$CONDITION_TYPES
    ];
}

sub get_defaults_for__devices {
    my $values = $CONDITION_TYPES->{devices}->{values};
    return {map {$_ => $values->{$_}->{label}->()} keys %$values};
}

sub get_defaults_for__query_args {
    my ($self, $owner_id) = @_;

    # NOTE! это как бы fake login менеджером под пользователя
    my $checked_rights = ['context_on_site_adblock_view'];
    my $rights = $self->app->rbac->get_rights_by_user_id($owner_id, $checked_rights);

    my $data = {
        (map {my $puid = sprintf('puid%d', $_); ($puid => $puid)} (1 .. 64)),
        $rights->{'context_on_site_adblock_view'} ? ('adb_enabled' => 'adb_enabled') : ()
    };

    return $data;
}

sub get_defaults_for__systems {
    my $values = $CONDITION_TYPES->{systems}->{values};
    return {map {$_->{val} => $_->{label}} values %$values};
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my $unknown_dict_names = join '", "', sort grep {!exists $CONDITION_TYPES->{$_}} sort keys %$need_fields;
    throw Exception::Validation::BadArguments sprintf('Unknown fields "%s"', $unknown_dict_names)
      if $unknown_dict_names;

    my $user_id = $self->_get_request_user_id($opts->{attributes}{login}, 'edit');

    my $dictionaries = {};
    foreach my $dict_name (sort keys %$need_fields) {
        my $sub = $CONDITION_TYPES->{$dict_name}->{'defaults'};
        if ($sub) {
            my $dict = $sub->($self, $user_id);
            if (ref($dict) eq 'HASH') {
                $dictionaries->{$dict_name} =
                  [+map {id => $_, label => $dict->{$_}}, sort {$dict->{$a} cmp $dict->{$b}} keys %$dict];
            } else {
                $dictionaries->{$dict_name} = $dict;
            }
        }
    }

    return $dictionaries;
}

sub get_fields_depends {
    my ($self) = @_;

    my $can_add_other = $self->check_short_rights('add_other') ? 1 : 0;

    return $can_add_other
      ? {
        # при изменении поля, какие связанные с ним поля нужно перезапросить
        depends => {login => [qw( conditions  query_args )],},
        # что нужно передать на бэкенд чтобы запросить поле
        required => {
            conditions => [qw( login )],
            query_args => [qw( login )],
        },
      }
      : {
        depends  => {},
        required => {},
      };
}

sub init {
    my ($self) = @_;
    ##### TODO: использовать базовый

    $self->SUPER::init();

    $self->register_rights($self->get_structure_rights_to_register());
    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
    $self->multistates_graph($self->get_structure_multistates_graph());
}

sub move_object {
    my ($self, $rule_obj, $insert_before, $dont_move_obj) = @_;

    my $rule_id = $rule_obj->{rule_id};

    my $data = $self->get($rule_obj, fields => ['owner_id', 'position', 'multistate']);
    my ($owner_id, $cur_pos, $multistate) = @$data{qw( owner_id   position multistate )};

    my $max_pos = 0;
    if (!$cur_pos || !$insert_before) {

        my $data = $self->partner_db_table()->get_all(
            fields => {max_pos => {MAX => ['position']},},
            filter => [
                AND => [
                    ['owner_id'   => '='      => \$owner_id],
                    ['multistate' => 'NOT IN' => \$self->get_multistates_by_filter('deleted or stopped')]
                ]
            ],
        );

        $max_pos = $data->[0]->{'max_pos'};
    }

    my $dest_pos = undef;
    if ($insert_before) {
        return TRUE if $insert_before == $rule_id;

        my $data = $self->get_all(
            fields => ['rule_id', 'position'],
            filter => {
                rule_id    => $insert_before,
                multistate => 'working'
            }
        )->[0];

        throw Exception::Validation::BadArguments gettext('Unknown rule_id=%s', $insert_before) unless $data;

        $dest_pos = $data->{position};
        $dest_pos = $dest_pos > $cur_pos ? $dest_pos - 1 : $dest_pos;
    }

    my ($new_pos, $sign, $from, $to) = $self->_get_move_params($cur_pos, $dest_pos, $max_pos);

    $self->partner_db_table()->edit($rule_obj, {position => $new_pos})
      if !$dont_move_obj && (!$dest_pos || $cur_pos != $dest_pos) && defined($new_pos);

    if ($sign) {

        my $query = sprintf q[
            UPDATE %s
            SET    position = position + ?
            WHERE  owner_id      = ?
                   AND rule_id  != ?
                   AND position BETWEEN ? AND ?
        ], $self->db_table_name();

        $self->partner_db->_do($query, $sign, $owner_id, $rule_id, $from, $to);
    }

    return $sign;
}

sub related_models {
    my ($self) = @_;

    return {
        # TODO: везде вызывать метод родителя
        # %{$self->SUPER::related_models()},
        users => {
            accessor => 'users',
            filter   => sub {
                my ($self, $results) = @_;
                return {id => array_uniq(map {$_->{'owner_id'} // ()} @$results)};
            },
            key_fields => ['id'],
        },
        business_blocks => {
            accessor => 'business_blocks',
            filter   => sub {
                my ($self, $results) = @_;
                return {rule_id => array_uniq(map {$_->{'rule_id'} // ()} @$results)};
            },
            key_fields => ['rule_id'],
            value_type => 'array_block_public_id'
        },
    };
}

sub set_default_values {
    my ($self, $changed_settings, $old_block_settings) = @_;

    my $new_block_settings = $changed_settings;

    if (defined($old_block_settings)) {
        #edit
        $new_block_settings = {%$old_block_settings, %$changed_settings};

    }

    return $new_block_settings;
}

# NOTE! обычно _add вызывается в add и duplicate, но в этой модели duplicate не предусмотрен (но вдруг появится)
sub _add {
    my ($self, %opts) = @_;

    my $found_blocks = delete $opts{_found_blocks};

    # NOTE! Сразу удаляем из opts все что не должно доехать до базы именно этой моедли
    my ($blocks, $login) = map {delete $opts{$_}} qw( blocks  login );

    my $user_id = $self->_get_request_user_id($login, 'add');

    # Предобработка
    $opts{conditions} = to_json($opts{conditions}, canonical => TRUE) if $opts{conditions};

    my $rule_id;
    $self->partner_db->transaction(
        sub {
            $rule_id = $self->partner_db_table()->add(
                {
                    owner_id    => $user_id,
                    create_date => curdate(oformat => 'db_time'),
                    position    => 0,
                    %opts
                }
            );

            # Чтобы залогировать создание
            $self->do_action($rule_id, 'add', %opts);

            $self->_edit_position_and_blocks($rule_id, $found_blocks);
        }
    );

    return $rule_id;

}

sub _edit_position_and_blocks {
    my ($self, $rule_id, $blocks_to_add, $is_move, $is_start_stopped) = @_;

    $self->do_action($rule_id, 'move', 'insert_before' => undef) if $is_move;

    $self->business_blocks->update_blocks($rule_id, $blocks_to_add, $is_start_stopped);

    return 1;
}

sub _get_move_params {
    my ($self, $cur_pos, $dest_pos, $max_pos) = @_;

    #      назад               #       вперед
    #   1         1            #   1          1
    #       ----> 2            #   2 ---      x
    #   2   |     2 -> 3(+1)   #   3    |     3 -> 2(-1)
    #   3   |     3 -> 4(+1)   #   4    |     4 -> 3(-1)
    #   4 --      x            #        ----> 4
    #   5         5            #   5          5

    my $sign = 0;
    my $from = 0;
    my $to   = 0;

    my $new_pos = $dest_pos;
    unless ($new_pos) {
        if ($cur_pos) {
            $new_pos = (defined($max_pos) && $max_pos >= $cur_pos ? $max_pos : undef);
        } else {
            $new_pos = (defined($max_pos) ? $max_pos + 1 : undef);
        }
    }

    if (defined($new_pos) && ($cur_pos || $dest_pos) && (!$max_pos || $new_pos <= $max_pos + 1)) {
        if (!$cur_pos) {
            $sign = 1;
            $from = $new_pos;
            $to   = $max_pos;
        } elsif ($cur_pos != $new_pos) {
            if ($cur_pos < $new_pos) {
                $sign = -1;
                $from = $cur_pos + 1;
                $to   = $max_pos && $new_pos > $max_pos ? $max_pos : $new_pos;
            } else {
                $sign = 1;
                $from = $new_pos;
                $to   = $cur_pos - 1;
            }
        }
    }

    return ($new_pos, $sign, $from, $to);
}

sub _get_request_user_id {
    my ($self, $login, $context) = @_;

    my $right_name =
      $context eq 'add'
      ? 'add_other'
      : 'edit_other';

    # NOTE! Если создает менеджер, то логин прилетает в запросе
    # TODO: общий код - нужно куда-то выносить
    my $user;
    if ($self->check_short_rights($right_name) && $login) {
        $user = $self->users->get_by_login($login, fields => ['id', 'client_id'])
          // throw Exception::Validation::BadArguments gettext('Unknown user');

    } else {
        $user = $self->get_option('cur_user', {});
    }

    return $user->{'id'};
}

sub _on_action_stop_delete {
    my ($self, $obj) = @_;

    $self->move_object($obj, undef, 1);

    # Сбрасываем в 0
    $self->partner_db_table()->edit($obj, {position => 0});

    $self->_edit_position_and_blocks($obj->{rule_id}, undef, undef, 1);
}

# TODO: вынсти копипаст в общую модель
sub _trim_opts {
    my ($self, $opts) = @_;

    foreach my $field_name (qw(caption)) {
        $opts->{$field_name} =~ s/^\s+|\s+$//g if $opts->{$field_name};
    }
}

sub get_bk_expression {
    my ($self, $condition) = @_;

    my $expr = [];
    foreach my $type (sort keys %$condition) {
        my $vals = $condition->{$type};

        if (ref($vals) eq 'ARRAY') {
            # 'devices'  => ['desktop',        'smarttv'        ],
            # 'browsers' => [2,                88               ],
            # 'urls'     => ['http://foo.com', 'https://bar.net'],
            push(@$expr, $self->_get_bk_expression($type, $vals));
        } elsif (ref($vals) eq 'HASH') {
            # 'headers'  => { 'accept' => ['application/json','text/xml'], }
            my @type_expr = ();
            foreach my $param (sort keys %$vals) {
                my $param_vals = $vals->{$param};

                push(@type_expr, @{$self->_get_bk_expression($type, $param_vals, $param)});
            }

            push(@$expr, \@type_expr);
        } else {
            throw 'Unknown type';
        }
    }

    return [grep {@$_} @$expr];
}

sub _get_bk_expression {
    my ($self, $type, $vals, $keyword) = @_;

    $keyword //= $CONDITION_TYPES->{$type}{'keyword'};

    my ($values, $values_type, $bk_expression) = @{$CONDITION_TYPES->{$type}}{qw(values values_type bk_expression)};
    $values_type //= 'string';

    my @expression = ();
    foreach my $val (@$vals) {
        my $val_keyword = $values->{$val}{'keyword'} // $keyword;
        my $value       = $values->{$val}{'val'}     // $val;

        if ($values_type eq 'string') {
            $value .= '';
        } elsif ($values_type eq 'number') {
            $value += 0;
        }

        push(@expression, $bk_expression ? $bk_expression->($val_keyword, $value) : [$val_keyword, 'equal', $value]);
    }

    return \@expression;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{'send_time'} = $MYSQL_DEFAULT_DATETIME;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_processing_before_validation($opts);

    $opts->{'update_time'} = $MYSQL_DEFAULT_DATETIME;
}

TRUE;
