package Application::Model::CheckAlive;

use qbit;
use Utils::Logger qw(INFO ERROR);
use Utils::MonitoringUtils qw(get_pjapi);

use base qw(
  QBit::Application::Model
  Application::Model::CheckAlive::DB
  Application::Model::CheckAlive::Skip
  Application::Model::CheckAlive::Test
  Application::Model::CheckAlive::Net
  Application::Model::CheckAlive::TVM
  Application::Model::CheckAlive::FirstRun
  );

sub accessor {'check_alive'}

__PACKAGE__->model_accessors(
    partner_db => 'Application::Model::PartnerDB',
    kv_store   => 'QBit::Application::Model::KvStore',
);

use Application::Model::CheckAlive::DB;

my %CHECKS = (
    db => {
        handler       => 'check_db_alive',
        order         => 1,
        stop_on_error => TRUE,
    },
    test => {
        handler => 'check_force_fail',
        order   => 2,
    },
    net => {
        handler => 'check_net_alive',
        order   => 4,
    },
    tvm => {
        handler => 'check_tvm_alive',
        order   => 5,
    },
);

sub _is_alive_once {
    my ($self, %opts) = @_;

    INFO("CheckAlive BEGIN ");

    my $skip = $self->get_skip(%opts);
    my $checks = $opts{checks} // \%CHECKS;

    my $has_error = FALSE;
    my @log;
    for my $check_id (sort {$checks->{$a}{order} <=> $checks->{$b}{order}} keys %$checks) {
        my $check = $checks->{$check_id};
        if (exists $skip->{$check_id} && $skip->{$check_id}{':all'}) {
            INFO("CheckAlive SKIP $check_id");
            next;
        }
        my $sub = $check->{handler};
        if (my $msg = $self->$sub(%opts, skip => $skip->{$check_id})) {
            push @log, [$check_id, $msg];
            ERROR("CheckAlive $check_id: " . (ref $msg ? to_json($msg, canonical => TRUE) : $msg));
            $has_error = TRUE;
            last if $check->{stop_on_error};
        }
    }
    INFO("CheckAlive END $has_error");

    return ($has_error, \@log);
}

sub is_alive {
    my ($self, %opts) = @_;

    my ($has_error, $log);
    if ($self->app->get_option("stage", '') eq 'autotest') {
        # на автотестах никаких проверок не делаем
        ($has_error, $log) = (0, []);
    } else {
        ($has_error, $log) = $self->_is_alive(%opts);
        $self->send_to_juggler($has_error, $log);
    }

    my $check_result = $has_error ? 0 : 1;
    if ($opts{raw}) {
        return ($check_result, to_json($log, canonical => TRUE));
    }

    my $result_data = {
        alive => $check_result,
        (@$log ? (log => $log) : ())
    };

    my $result = to_json($result_data, canonical => TRUE);
    my $finished = 0;
    return sub {
        return [] if $finished;
        $finished = 1;
        return [$result];
    };
}

sub _is_alive {
    my ($self, %opts) = @_;

    my $res = $self->get_first_run_result(%opts);
    if ($res && !$res->[0]) {
        return @$res;
    }
    my @res = $self->_is_alive_once(%opts);
    $self->set_first_run_result(\@res, %opts);

    return @res;
}

sub send_to_juggler {
    my ($self, $error, $description) = @_;

    if (my $service = $self->get_service_name) {
        $description //= [];
        return get_pjapi($self->app)->send(
            events => [
                {
                    service     => $service,
                    host        => $self->get_host_name,
                    status      => $error ? 'CRIT' : 'OK',
                    description => to_json($description, canonical => TRUE),
                }
            ]
        );
    }
}

1;
