package Application::Model::Common;

use qbit;
use Application::Model::DAC;

use base qw(
  Application::Model::DBManager::Base
  Application::Model::Multistate::DBManager
  Application::Model::DAC
  );

use Exception::Validation::BadArguments;

sub accessor {$_[0]->{'accessor'}}

sub public_id {
    my ($self, $obj) = @_;

    return $obj->{id};
}

=encoding UTF-8

=cut

=head2 get_add_fields

    my $fields = $model->get_add_fields();

Возвращает hashref со список полей, которые доступны текущему пользователю при
использовании метода add().

В некоторых случаях, в качестве значения может быть какая-то структура с
данными, в которых содержаться некоторые подробности.

Пример возвращаемого значения:

    {
        caption          => 1,
        comment          => 1,
        direct_block     => 1,
        dsp_blocks       => [
            "400x240",
            "240x400",
            "728x90",
            "320x50",
        ],
    }

=cut

sub get_add_fields {
    my ($self) = @_;

    throw "get_add_fields() is abstract method";
}

=head2 get_fields_by_right

Вспомогательная ф-ция

=cut

sub get_fields_by_right {
    my ($self, %opts) = @_;

    my $res_fields = $opts{res_fields} // {};

    my $no_right_fields = $opts{no_right_fields};
    if ($no_right_fields) {
        $res_fields->{$_} = TRUE foreach (@$no_right_fields);
    }

    my $right_fields = $opts{right_fields};
    if ($right_fields) {
        foreach my $right_prefix (keys %$right_fields) {
            foreach my $field_name (@{$right_fields->{$right_prefix}}) {
                $res_fields->{$field_name} = TRUE if $self->check_short_rights("${right_prefix}_field__${field_name}");
            }
        }
    }

    return $res_fields;
}

=head2 check_bad_fields

=cut

sub check_bad_fields {
    my ($self, $id, $data, $editable_fields) = @_;

    unless (defined($editable_fields)) {
        my $model_fields = $self->get_model_fields();

        my $editable_fields_depends = $model_fields->{'editable_fields'}{'forced_depends_on'}
          // $model_fields->{'editable_fields'}{'depends_on'};

        my $tmp_rights = $self->app->add_all_tmp_rights();
        my $db_data = $self->get($id, fields => $editable_fields_depends);
        undef($tmp_rights);

        $editable_fields = $self->get_editable_fields({%$db_data, %$data});
    }

    my @bad_fields = grep {!$editable_fields->{$_}} sort keys(%$data);
    throw Exception::Validation::BadArguments gettext('You can not edit the following fields: %s',
        join(', ', @bad_fields))
      if @bad_fields;

    return $editable_fields;
}

=head2 check_bad_fields_in_add

=cut

sub check_bad_fields_in_add {
    my ($self, $data, $fields) = @_;

    $fields //= $self->get_add_fields();

    my @bad_fields = sort grep {!$fields->{$_}} keys(%$data);

    if (@bad_fields) {
        throw Exception::Validation::BadArguments gettext("You can't use following fields in add: %s",
            join(', ', @bad_fields));
    }

    return TRUE;
}

sub check_mobile_app_bundle_id {
    my ($self, $name, $type) = @_;

    return defined(get_bundle_id($name, $type));
}

sub d_get_right {
    my ($self, $right) = @_;

    return sub {$_[0]->get_right($right)};
}

sub get_description_right {
    my ($self, $right, $d_gettext) = @_;

    unless (defined($d_gettext)) {
        if ($right eq 'view') {
            $d_gettext = d_gettext('Right to view "%s" in menu', $self->get_product_name());
        } elsif ($right eq 'view_all') {
            $d_gettext = d_gettext('Right to view all for "%s"', $self->get_product_name());
        } elsif ($right =~ /^add$/) {
            $d_gettext = d_gettext('Right to add new item for %s', $self->get_product_name());
        } elsif ($right =~ /add_other$/) {
            $d_gettext = d_gettext('Right to add new item for other users for "%s"', $self->get_product_name());
        } elsif ($right =~ /^approve$/) {
            $d_gettext = d_gettext('Right to approve item for %s', $self->get_product_name());
        } elsif ($right =~ /^reject$/) {
            $d_gettext = d_gettext('Right to reject item for %s', $self->get_product_name());
        } elsif ($right =~ /^view_full_multistate_name$/) {
            $d_gettext = d_gettext('Right to view full multistate name for %s', $self->get_product_name());
        } elsif ($right =~ /^add_field__(.+)$/) {
            $d_gettext = d_gettext('Right to add field "%s" for %s', $1, $self->get_product_name());
        } elsif ($right =~ /^view_action_log$/) {
            $d_gettext = d_gettext('Right to view action logs for %s', $self->get_product_name());
        } elsif ($right =~ /^view_field__(.+)$/) {
            $d_gettext = d_gettext('Right to view field "%s" for %s', $1, $self->get_product_name());
        } elsif ($right =~ /^view_filter__(.+)$/) {
            $d_gettext = d_gettext('Right to view filter "%s" for %s', $1, $self->get_product_name());
        } elsif ($right =~ /^edit$/) {
            $d_gettext = d_gettext('Right to edit for %s', $self->get_product_name());
        } elsif ($right =~ /^edit_all$/) {
            $d_gettext = d_gettext('Right to edit all for %s', $self->get_product_name());
        } elsif ($right =~ /^edit_field__(.+)$/) {
            $d_gettext = d_gettext('Right to edit field "%s" for %s', $1, $self->get_product_name());
        } else {
            throw gettext('Unknown right %s', $right);
        }
    }

    return ($self->accessor() . '_' . $right => $d_gettext);
}

sub get_fields_cnt {
    my ($self, %opts) = @_;

    my $pk = shift(@{$opts{'fields'}});

    my $query = $self->query(
        fields => $self->_get_fields_obj(['id']),
        filter => $self->get_db_filter($opts{'filter'})
    );

    $query->{'__TABLES__'}[0]{'fields'} = {
        cnt_id => {count => [{distinct => [{CONCAT => [$pk->[0], \'_', $pk->[1]]}]}]},
        map {("cnt_$_" => {count => [{distinct => [$_]}]})} @{$opts{'fields'}}
    };

    my $data = $query->get_all();

    return $data->[0];
}

sub fields_from_db {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;

    foreach my $field (keys(%$model_fields)) {
        next unless exists $obj->{$field};
        my $descr = $model_fields->{$field};
        if ($descr && $descr->{t} && $descr->{t} eq 'json') {
            $obj->{$field} = from_json($obj->{$field} || '{}');
        } elsif ($descr && $descr->{t} && $descr->{t} eq 'set') {
            $obj->{$field} = [split(/,/, $obj->{$field} || '')];
        }
    }
    return $obj;
}

sub hook_stash {
    my ($self) = @_;
    $self->{_hook_stash} //= Utils::HookStash->new();
}

sub has_opts_storage {
    my ($self) = @_;
    return ref($self->get_model_fields()->{opts}) eq 'HASH';
}

sub hook_check_available_fields {
    my ($self, $obj) = @_;

    $self->hook_stash->set('allowed_fields', $self->get_add_fields($obj));

    $self->check_bad_fields_in_add($obj, $self->hook_stash->get('allowed_fields'));
}

sub hook_fields_processing_before_validation {
    my ($self, $obj) = @_;

    foreach my $field ($self->get_fields_to_trim()) {
        $obj->{$field} = trim($obj->{$field}) if defined($obj->{$field}) && ref($obj->{$field}) eq '';
    }
}

sub get_fields_to_trim {
    my ($self) = @_;
    my $model_fields = $self->get_model_fields();

    return grep {$model_fields->{$_}{'need_trim'}} keys(%$model_fields);
}

sub hook_owner_processing {
    my ($self, $obj) = @_;

    if (exists($obj->{'owner_id'}) && $self->check_short_rights('add_other')) {
        my $tmp_rights = $self->app->add_tmp_rights('users_view_field__client_id');
        $self->hook_stash->set('user',
            $self->users->get($obj->{'owner_id'}, fields => [qw(id client_id business_unit is_assessor)]));
        throw Exception::Validation::BadArguments gettext('User with id "%s" not found', $obj->{'owner_id'})
          unless $self->hook_stash->get('user');
    } elsif (exists($obj->{'login'}) && $self->check_short_rights('add_other')) {
        my $login      = delete($obj->{'login'});
        my $tmp_rights = $self->app->add_tmp_rights('users_view_field__client_id');
        $self->hook_stash->set('user',
            $self->users->get_by_login($login, fields => [qw(id client_id business_unit is_assessor)]));
        throw Exception::Validation::BadArguments gettext('User with login "%s" not found', $login)
          unless $self->hook_stash->get('user');
        $obj->{'owner_id'} = $self->hook_stash->get('user')->{'id'};
    } else {
        $self->hook_stash->set('user', $self->get_option('cur_user', {}));
        $obj->{'owner_id'} = $self->hook_stash->get('user')->{'id'};
    }

    delete $self->hook_stash->get('allowed_fields')->{'login'};
}

sub hook_post_validation {
    my ($self, $obj) = @_;
}

sub hook_fields_validation {
    my ($self, $obj) = @_;

    $self->app->validator->check(
        data  => $obj,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => $self->hook_stash->get('allowed_fields')),
    );
}

sub hook_set_initialize_settings {
    my ($self, $obj, %opts) = @_;

    #TODO: разнести по ролям
    $obj->{'multistate'} = 0;
    $obj->{'create_date'} = curdate(oformat => 'db_time');
}

sub hook_preparing_fields_to_save {
    my ($self, $obj) = @_;

    my %table_fields = map {$_->name => $_} @{$self->partner_db_table->fields()};

    my $model_fields = $self->get_model_fields();
    my %model_db_fields =
      map {$_ => TRUE} grep {$model_fields->{$_}{'db'} || $model_fields->{$_}{'from_opts'}} keys(%$model_fields);

    if ($table_fields{'model'}) {
        $model_db_fields{'model'} = TRUE;
    }

    if ($table_fields{'unique_id'}) {
        $model_db_fields{'unique_id'} = TRUE;
    }

    if ($self->has_opts_storage()) {
        my @model_opts_fields = grep {$model_fields->{$_}{'from_opts'}} keys(%$model_fields);

        my $settings = $self->hook_stash->get('settings');

        my $db_opts =
          {($settings->{'opts'} ? %{$settings->{'opts'}} : ()), hash_transform($settings, \@model_opts_fields)};

        $self->force_to_number($db_opts, $model_fields);

        $obj->{'opts'} = to_json($db_opts);

        my $qv = $self->app->validator->check(
            data     => $obj->{'opts'},
            app      => $self,
            template => {
                type   => 'json',
                schema => $self->get_opts_schema_name(),
            }
        );

        throw Exception $qv->get_all_errors if $qv->has_errors;
    }

    my %fields_from_native_table;
    my %fields_from_related_models;
    foreach my $field (keys(%$obj)) {
        if (exists($table_fields{$field}) && $model_db_fields{$field} && !exists($table_fields{$field}->{'generated'}))
        {
            $fields_from_native_table{$field} = $obj->{$field};
        } elsif (!$model_db_fields{$field}) {
            $fields_from_related_models{$field} = $obj->{$field};
        }
    }

    $self->hook_stash->set('fields_from_native_table',   \%fields_from_native_table)   if %fields_from_native_table;
    $self->hook_stash->set('fields_from_related_models', \%fields_from_related_models) if %fields_from_related_models;
}

sub hook_writing_to_database {
    my ($self, $obj) = @_;

    $self->partner_db->transaction(
        sub {
            my $is_change_multistate_needed = FALSE;
            unless ($self->hook_stash->check('id')) {
                my $id = $self->partner_db_table()->add($self->hook_stash->get('fields_from_native_table'));

                my %id;
                my $pk = $self->partner_db_table()->primary_key;
                @id{@$pk} = ref $id ? @$id : ($id);
                $self->hook_stash->set('id', \%id);
                $is_change_multistate_needed = TRUE;
            }

            if ($self->hook_stash->check('fields_from_related_models')) {
                $self->hook_save_fields_from_related_models($obj);
            }

            $self->do_action($self->hook_stash->get('id'), 'add', %$obj) if $is_change_multistate_needed;
        }
    );
}

sub hook_save_fields_from_related_models { }

sub hook_processing_after_insert { }

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied') unless $self->check_rights($self->get_rights_by_actions('add'));

    my $_stash = $self->hook_stash->init(
        mode => 'add',
        $self->hook_stash_add_fields,
        data => {opts => \%opts, current => \%opts}
    );

    $self->hook_check_available_fields(\%opts);
    if ($self->isa('Application::Model::Block') && 'page_id' ne $self->get_page_id_field_name()) {
        $opts{page_id} //= $opts{$self->get_page_id_field_name()};
        $opts{$self->get_page_id_field_name()} //= $opts{page_id};
    }
    $self->hook_fields_processing_before_validation(\%opts);
    $self->hook_owner_processing(\%opts);
    $self->hook_fields_validation(\%opts);
    $self->hook_set_initialize_settings(\%opts);
    $self->hook_preparing_fields_to_save(\%opts);

    $self->partner_db->transaction(
        sub {
            $self->hook_writing_to_database(\%opts);
            $self->hook_processing_after_insert(\%opts);

            $self->hook_post_validation(\%opts);
        }
    );
    return $self->public_id($self->hook_stash->get('id'));
}

sub on_action_edit {
    my ($self, $object, %opts) = @_;

    my $model_fields = $self->get_model_fields();

    my @need_check_fields = grep {$model_fields->{$_}->{'need_check'}} keys(%$model_fields);

    my $tmp_all_rights = $self->app->add_all_tmp_rights();
    my $old_settings   = $self->get(
        $object,
        fields            => \@need_check_fields,
        all_locales       => 1,
        use_java_json_api => FALSE,
    );
    undef($tmp_all_rights);

    my %id;
    my $pk = $self->get_pk_fields;
    @id{@$pk} = @{$object}{@$pk};

    my $_stash = $self->hook_stash->init(
        mode => 'edit',
        data => {
            id      => \%id,
            opts    => \%opts,
            current => $old_settings,
            fields  => [sort keys %opts],
        },
        $self->hook_stash_edit_fields,
    );

    $self->check_bad_fields($object, \%opts, $object->{'editable_fields'});

    $self->hook_fields_processing_before_validation(\%opts);

    my %fix;

    if ($self->isa('Application::Model::Block')) {
        my $page_id_name = $self->get_page_id_field_name();
        %fix = ($page_id_name => {type => 'int_un', optional => TRUE}, page_id => {type => 'int_un', optional => TRUE});
    }

    if ($self->can('get_template')) {
        $self->app->validator->check(
            stash => {current => $old_settings},
            data  => $self->hook_stash->get('settings'),
            app   => $self,
            throw => TRUE,
            $self->get_template('fix' => \%fix),
        );
    }

    $self->hook_post_validation(\%opts);

    $self->hook_preparing_fields_to_save(\%opts);

    $self->hook_save_fields_from_related_models(\%opts)
      if $self->hook_stash->check('fields_from_related_models');

    $self->partner_db_table()->edit($object, $self->hook_stash->get('fields_from_native_table'))
      if $self->hook_stash->check('fields_from_native_table');

    $self->hook_processing_after_update(\%opts);

    return TRUE;
}

sub on_action_duplicate {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();

    $obj = $self->get($obj, fields => ['*']);

    map {$obj->{$_} = $opts{modify_fields}->{$_}} keys %{$opts{modify_fields}};
    delete $obj->{$_} foreach @{$opts{unset_fields}};

    my $stash = $self->hook_stash->init(
        mode => [qw(add duplicate)],
        $self->hook_stash_add_fields,
        data => {opts => $obj, current => $obj},
    );
    $self->hook_stash->set(allowed_fields => undef);

    $self->hook_fields_validation($obj);
    $self->hook_set_initialize_settings($obj, %opts);
    $self->hook_preparing_fields_to_save($obj);
    $self->hook_writing_to_database($obj);
    $self->hook_processing_after_insert($obj);

    return {public_id => $self->public_id($self->hook_stash->get('id')),};
}

sub hook_processing_after_update { }

sub hook_stash_add_fields {
    my ($self) = @_;

    return (
        once => [
            qw(
              user
              allowed_fields
              fields_from_native_table
              id
              fields_from_related_models
              )
        ],
        free  => [],
        merge => {settings => ['current', 'opts']},
    );
}

sub hook_stash_edit_fields {
    my ($self) = @_;

    return (
        once => [
            qw(
              fields_from_native_table
              fields_from_related_models
              )
        ],
        free  => [],
        merge => {settings => ['current', 'opts']},
    );
}

sub force_to_number {
    my ($self, $settings, $model_fields) = @_;

    if (ref($settings) eq 'HASH') {
        foreach my $field (keys(%$settings)) {
            if ($model_fields->{$field}{'type'}
                && defined($settings->{$field}))
            {

                my $type = $model_fields->{$field}{'type'};

                $settings->{$field} += 0
                  if $type eq 'number'
                      || $type eq 'boolean';

                if (   $type eq 'array'
                    && $model_fields->{$field}{'sub_type'} eq 'number')
                {
                    foreach (@{$settings->{$field}}) {
                        $_ += 0;
                    }
                }
            }
        }
    } else {
        throw sprintf('Unknown type: %s', ref($settings));
    }
}

sub get_fields_defaults {
    my ($self, $opts) = @_;

    my $need_fields = $self->make_need_fields_for_defaults($opts);
    return $self->make_fields_defaults($opts, $need_fields);
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;
    return {};
}

sub make_need_fields_for_defaults {
    my ($self, $opts) = @_;

    my $fields_depends = $self->get_fields_depends();

    my %need_fields = map {$_ => TRUE} @{$opts->{fields}};
    foreach (map {@{$fields_depends->{'depends'}{$_} // []}} @{$opts->{changed_fields}}) {
        $need_fields{$_} = TRUE;
    }
    return \%need_fields;
}

TRUE;
