package Application::Model::CommonOffer;

use qbit;

use base qw(QBit::Application::Model RestApi::SimpleModel);

use PiConstants qw(
  @ADFOX_PRESET_DEFAULT
  @ADFOX_PRESET_PAIDPRODUCTS
  $CONTRACT_OFFER_ERROR_MAIL
  $PARTNER2_COMMON_OFFER_ERRORS
  );

sub accessor {'common_offer'}

__PACKAGE__->model_accessors(
    api_adfox_graphql => 'Application::Model::API::Yandex::AdFoxGraphQL',
    api_balance       => 'Application::Model::API::Yandex::Balance',
    mail_notification => 'Application::Model::MailNotification',
    partner_db        => 'Application::Model::PartnerDB',
    users             => 'Application::Model::Users',
);

sub after_offer_accept {
    my ($self, $pi_client_id, $contracts, $result, $adfox_offer) = @_;

    my $user_id = $self->app->get_option('cur_user', {})->{id};

    # this must be called before users->edit so we are able to get
    # contracts->{adfox_contract_paid_products} in users.hook_processing_after_update
    $self->partner_db->common_offer_allowed_users->edit(
        $user_id,
        {
            accept_date => curdate(oformat => 'db_time'),
            opts => to_json({contracts => $contracts, result => $result, pi_client_id => $pi_client_id})
        }
    );

    my $tmp_rights =
      $self->app->add_tmp_rights(qw(users_edit_field__adfox_offer users_edit_field__has_common_offer users_edit_all));
    $self->users->edit(
        $user_id,
        has_common_offer     => 1,
        has_mobile_mediation => 1,
        has_rsya             => 1,
        adfox_offer          => ($adfox_offer ? 1 : 0),
    );

    $tmp_rights = $self->app->add_all_tmp_rights();
    my $user = $self->users->get($user_id, fields => [qw(cooperation_form id inn)]);
    undef($tmp_rights);

    my $filter = [
        AND => [
            [
                page_id => 'IN' => $self->partner_db->query->select(
                    table  => $self->partner_db->all_pages,
                    fields => ['page_id'],
                    filter => [owner_id => '=' => \$user_id],
                ),
            ],
            [dissabled => '=' => \1],
        ]
    ];

    $self->partner_db->assistants->edit($self->partner_db->assistants->filter($filter), {dissabled => 0});
}

sub offer_text {
    my ($self) = @_;

    my $user_id = $self->app->get_option('cur_user', {})->{id};
    my $lang;
    my $url        = $self->get_option('url');
    my $tmp_rights = $self->app->add_tmp_rights('users_view_field__client_id');
    my $user       = $self->app->users->get($user_id, fields => [qw(client_id country_id)]);
    undef $tmp_rights;
    my $client_id = $user->{client_id} // $self->app->api_balance->get_client_id_by_uid($user_id);
    my $raw_contracts = $self->app->documents->get_contracts(client_id => $client_id);
    my $contract      = $raw_contracts->[-1];
    my $person_type   = $contract->{Person}{type} // '';
    my $tld;

    if (!$person_type || $person_type eq 'ph' || $person_type eq 'ur') {
        $url .= '/yan_service_offer';
        $lang = 'ru';
        $tld  = 'ru';
    } else {
        $url .= '/oferta_yan_ch';
        $lang = 'en';
        $tld  = 'com';
    }

    my $data = $self->app->api_cabinet->request(
        uri     => $url,
        method  => 'POST',
        headers => {'Content_type' => 'application/json', 'Accept' => 'application/json'},
        content => to_json(
            {
                doccenterParams => {
                    components     => $self->get_option('components'),
                    lang           => $lang,
                    forceFullLinks => \1,
                },
                tld => $tld,
            }
        ),
    );

    my $content = from_json(from_json($data)->{doccenterResponse})->{content};
    utf8::decode($content);
    return $content;
}

sub offer_get {
    my ($self) = @_;

    my $user_id = $self->get_option('cur_user', {})->{'id'};

    # требуется доработка метода для этого случая
    #my $contracts = $self->users->check_pi_adfox_contracts(user_id => $user_id, from_offer => TRUE);

    return TRUE;
}

sub offer_accept {
    my ($self, %opts) = @_;

    my $result;
    my %extra = (adfox_login => $opts{adfox_login}, adfox_offer => $opts{adfox_offer},);

    try {
        my $user_id = $extra{user_id} = $self->get_option('cur_user', {})->{'id'};

        my $contracts = $extra{check_pi_adfox_contracts} = $self->users->check_pi_adfox_contracts(
            user_id             => $user_id,
            adfox_login         => $opts{adfox_login},
            adfox_password      => $opts{adfox_password},
            allow_paid_services => $opts{adfox_offer},
            from_offer          => TRUE,
        );

        my $tmp_rights   = $self->app->add_tmp_rights('users_view_field__client_id');
        my $user         = $self->users->get($user_id, fields => [qw(client_id login)]) // {};
        my $pi_client_id = $extra{pi_client_id} = $user->{client_id}
          // $self->api_balance->get_client_id_by_uid($user_id);
        undef $tmp_rights;

        $result = $extra{accept_common_offer} = $self->users->accept_common_offer(
            adfox_login => $opts{adfox_login},
            client_id   => $pi_client_id,
            contracts   => $contracts,
            from_offer  => TRUE,
            user_id     => $user_id,
        );

        my $adfox_user;
        unless ($opts{adfox_login}) {
            $adfox_user = $extra{create_user} =
              $self->api_adfox_graphql->create_user(user_id => $user_id, presets => \@ADFOX_PRESET_DEFAULT);
        } else {
            my %args4bind_user = (
                user_id     => $user_id,
                adfox_login => $opts{adfox_login},
                presets     => [@ADFOX_PRESET_DEFAULT, ($opts{adfox_offer} ? @ADFOX_PRESET_PAIDPRODUCTS : ())],
            );

            # переход на ЕО + PAIDPRODUCTS
            # partner has AdFox paid products
            if ($opts{adfox_offer}) {
                my $adfox_contract_created = $self->users->create_contract_adfox_paid_products(
                    user_id     => $user_id,
                    adfox_login => $opts{adfox_login},
                    mail_info   => {
                        adfox_contract => $contracts->{adfox_contract},
                        adfox_info     => {
                            id    => '',
                            login => $opts{adfox_login},
                        },
                        client_id => $pi_client_id,
                        contract  => {
                            contract2_id => $result->{ID},
                            external_id  => $result->{EXTERNAL_ID},
                        },
                        login   => $user->{login},
                        user_id => $user_id,
                    },
                );
                if ($adfox_contract_created) {
                    $args4bind_user{billing_data} = {
                        client_id   => $adfox_contract_created->{client_id},
                        contract_id => $adfox_contract_created->{id},
                    };
                    # to save it later in after_offer_accept
                    $contracts->{adfox_contract_paid_products} = $args4bind_user{billing_data};
                } else {
                    $contracts->{adfox_contract_paid_products} = 'via mail';
                }
            }
            $adfox_user = $extra{bind_user} = $self->api_adfox_graphql->bind_user(%args4bind_user);
        }

        if ($adfox_user) {
            # Так как у нас уже есть логин пользователя АФ, то передаётся флаг mobile_mediation
            # который отвечает за то, чтобы не ходить в ручку АФ за инфой о пользователе
            $self->users->do_action(
                $user_id,
                'link_adfox_user',
                adfox_id         => $adfox_user->{id},
                adfox_login      => $adfox_user->{login},
                mobile_mediation => TRUE,
            );
        }

        $self->partner_db->users->edit($user_id,
            {opts => {json_set => ['opts', \'$.paid_offer', \1, \'$.has_common_offer', \1,]}})
          if $opts{adfox_offer};

        $self->after_offer_accept($pi_client_id, $contracts, $result, $opts{adfox_offer});
    }
    catch Exception with {
        my ($exception) = @_;

        unless ($exception->{'__mailed'}) {
            local $Data::Dumper::Sortkeys = 1;
            my $login = ($self->get_option('cur_user', {})->{'login'} || $extra{user_id} // '');
            my $message = join("\n",
                'Пользователь не смог перейти на оферту',
                '-' x 40,
                'Error: ' . $exception->message(),
                sprintf('PI: login=%s, id=%s', $login, ($extra{user_id} // ''),),
                'AdFox: login=' . ($extra{adfox_login} // ''),
                '',
                'Технические детали:',
                '-' x 40,
                $exception,
                Dumper \%extra);
            $self->mail_notification->add_common_offer_error(
                subject => sprintf(
                    'Common offer failed - Оферта, login=%s, adfox_login=%s',
                    $login, ($extra{adfox_login} // '')
                ),
                to      => [$PARTNER2_COMMON_OFFER_ERRORS, $CONTRACT_OFFER_ERROR_MAIL],
                message => $message,
            );
        }

        throw $exception;
    };

    return $result;
}

sub get_product_name {gettext('common_offer')}

sub get_model_fields {+{}}

sub api_available_actions {()}

sub api_get_all {[]}

TRUE;
