package Application::Model::Contracts;

use qbit;

use base qw(Application::Model::Common Application::Model::ValidatableMixin);

use Clone qw(clone);

sub accessor             {'contracts'}
sub db_table_name        {'contracts'}
sub get_opts_schema_name {'contracts_opts'}

sub get_structure_model_accessors {
    my ($self) = @_;

    return {
        api_balance => 'Application::Model::API::Yandex::Balance',
        documents   => 'Application::Model::Documents',
        partner_db  => 'Application::Model::PartnerDB',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => 'contracts',
            description => d_gettext('Partner contracts'),
            rights      => {
                contracts_view => d_gettext('Right to view contracts'),
                contracts_add  => d_gettext('Right to add contracts'),
                contracts_edit => d_gettext('Right to edit contracts'),
            },
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        id              => {default   => TRUE,        db   => TRUE, pk => TRUE, type => 'number',},
        multistate      => {db        => TRUE,        type => 'number',},
        type            => {db        => TRUE,        type => 'string',},
        external_id     => {db        => TRUE,        type => 'string',},
        client_id       => {db        => TRUE,        type => 'number',},
        dt              => {db        => TRUE,        type => 'string',},
        end_dt          => {db        => TRUE,        type => 'string',},
        update_dt       => {db        => TRUE,        type => 'string',},
        contract        => {db        => TRUE,        type => 'string',},
        person_id       => {db        => TRUE,        type => 'number',},
        raw_contract    => {from_opts => 'from_hash', type => 'string',},
        offset          => {from_opts => 'from_hash', type => 'number',},
        table_name      => {from_opts => 'from_hash', type => 'string',},
        row_number      => {from_opts => 'from_hash', type => 'number',},
        version         => {from_opts => 'from_hash', type => 'number',},
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
        },
        actions => {
            depends_on => [qw(id multistate)],
            get        => sub {
                $_[0]->model->get_actions($_[1]);
              }
        },
        available_fields => {
            label => d_gettext('Available fields'),
            get   => sub {
                return $_[0]->model->get_available_fields();
              }
        },
        editable_fields => {
            get => sub {
                $_[0]->model->get_editable_fields();
              }
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            client_id  => {type => 'number',     label => d_gettext('Client ID')},
            person_id  => {type => 'number',     label => d_gettext('Person ID')},
            id         => {type => 'number',     label => d_gettext('Contract ID')},
            id_name    => {type => 'text',       label => d_gettext('Contract Number')},
            end_dt     => {type => 'text',       label => d_gettext('End date')},
            multistate => {type => 'multistate', label => d_gettext('Status')},
        }
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [{name => 'id', label => gettext('Contract ID')}, {name => 'multistate', label => gettext('Status')},];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => 'New',
        multistates => [],
        actions     => {
            add  => d_gettext('Add'),
            edit => d_gettext('Edit'),
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action => 'edit',
                from   => '__EMPTY__',
            },
        ],
    };
}

sub get_available_fields {
    my ($self) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} grep {$model_fields->{$_}{db} || $model_fields->{$_}{from_opts}} keys(%$model_fields);

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    return $self->get_available_fields();
}

sub get_editable_fields {
    my ($self) = @_;

    return $self->get_available_fields();
}

sub update {
    my ($self, $data) = @_;

    # split raw contract on base contract and collaterals
    my $raw_contract = clone $data->{raw_contract};
    my $collaterals  = delete $raw_contract->{collaterals};
    $self->api_balance->_fix_dates(values %$collaterals);
    my $base_contract = delete $collaterals->{0};
    @$base_contract{qw(client_id external_id person_id type)} =
      @$raw_contract{qw(client_id external_id person_id type)};
    my $contract = {
        Collaterals =>
          [map {$collaterals->{$_}{class} = 'COLLATERAL'; $collaterals->{$_}} sort {$a <=> $b} keys %$collaterals],
        Contract => $base_contract,
    };

    # apply collaterals on base contract
    my $merged_contract = $self->documents->get_contracts(
        contract_list                  => [$contract],
        skip_collateral_start_dt_check => TRUE,
    );
    if (@$merged_contract) {
        $data->{contract} = to_json $merged_contract->[0]{Contract};

        # add additional fields
        @$data{qw(client_id external_id id type update_dt)} =
          @$raw_contract{qw(client_id external_id id type update_dt)};
        $data->{dt}         = $merged_contract->[0]{Contract}{dt};
        $data->{end_dt}     = $merged_contract->[0]{Contract}{end_dt};
        $data->{version}    = $raw_contract->{version_id};
        $data->{multistate} = 0;

        # save data
        if ($self->get($data->{id})) {
            $self->do_action($data->{id}, 'edit', %$data);
        } else {
            $self->add(%$data);
        }
    }
}

TRUE;
