package Application::Model::CookieMatch;

use qbit;

use base qw(Application::Model::Common RestApi::MultistateModel Application::Model::ValidatableMixin);

use Exception::DB::DuplicateEntry;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;
use Utils::JSON qw(
  fix_type_for_complex
  );

sub accessor      {'cookie_match'}
sub db_table_name {'cookie_match'}

sub get_product_name {
    gettext('cookie_match');
}

sub get_structure_model_accessors {
    return {
        partner_db               => 'Application::Model::PartnerDB::CookieMatch',
        users                    => 'Application::Model::Users',
        context_on_site_campaign => 'Application::Model::Product::AN::ContextOnSite::Campaign',
        search_on_site_campaign  => 'Application::Model::Product::AN::SearchOnSite::Campaign',
        video_an_site            => 'Application::Model::Page::Video',
        api_http_bk              => 'QBit::Application::Model::API::Yandex::HTTPBK',
        cookie_match_link_pi     => 'Application::Model::CookieMatch::LinkPi',
        cookie_match_owners      => 'Application::Model::CookieMatch::Owners',
    };
}

sub get_structure_rights_to_register {
    return [
        {
            name        => 'cookie_match',
            description => d_gettext('Right to manage Cookie Match'),
            rights      => {
                cookie_match_view                => d_gettext('Right to view Cookie match in menu'),
                cookie_match_view_all            => d_gettext('Right to view all Cookie match'),
                cookie_match_view_action_log     => d_gettext('Right to view Cookie match action logs'),
                cookie_match_view_field__comment => d_gettext('Right to view field "comment" of Cookie match'),
                cookie_match_add_field__comment  => d_gettext('Right to add field "comment" of Cookie match'),
                cookie_match_edit_field__comment => d_gettext('Right to edit field "comment" of Cookie match'),
            }
        }
    ];
}

sub get_structure_model_fields {
    return {
        id => {
            default     => TRUE,
            db          => TRUE,
            pk          => TRUE,
            label       => d_gettext('ID'),
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        comment => {
            db           => TRUE,
            label        => d_gettext('Comment'),
            check_rights => 'cookie_match_view_field__comment',
            type         => 'string',
            api          => 1,
            need_check   => {
                len_max  => 255,
                optional => TRUE,
            },
        },
        tag => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Tag'),
            type       => 'string',
            api        => 1,
            need_check => {
                check => sub {
                    my ($qv, $tag) = @_;

                    throw Exception::Validator::Fields gettext(
'Tag must consist of latin lowercase letters and numbers and must be no more 32 characters in length'
                    ) unless $tag =~ /\A[a-z0-9]{1,32}\z/;

                    #hack for uniq tags
                    my %tags =
                      map {$_->{'tag'} => TRUE} @{
                        $qv->app->partner_db->dsp->get_all(
                            fields => ['tag'],
                            filter => ['AND' => [{'NOT' => [{isnull => ['tag']}]}]]
                        )
                      };

                    foreach (map {$_->{'tag'}} @{$qv->app->get_all(fields => ['tag'])}) {
                        $tags{$_} = TRUE;
                    }

                    throw Exception::Validator::Fields gettext('Tag "%s" already exists', $tag) if $tags{$tag};
                  }
            }
        },
        data_key => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Data key'),
            type       => 'string',
            api        => 1,
            need_check => {
                check => sub {
                    my ($qv, $data_key) = @_;

                    throw Exception::Validator::Fields gettext(
'data_key must consist of latin lowercase letters and numbers and must be 32 characters in length'
                    ) unless $data_key =~ /\A[a-z0-9]{32}\z/;
                  }
            }
        },
        skip_data_key => {
            db         => TRUE,
            label      => d_gettext('Skip data key'),
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },
        js_redir => {
            db         => TRUE,
            label      => d_gettext('JS redirect'),
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },
        multistate => {
            db          => TRUE,
            label       => d_gettext('Status'),
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        multistate_name => {
            depends_on => ['multistate'],
            label      => d_gettext('Multistate name'),
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1,
        },
        create_date => {
            depends_on => ['id'],
            label      => d_gettext('Create date'),
            get        => sub {
                $_[0]->{'__ACTION_LOGS__'}{$_[1]->{'id'}};
            },
            type => 'string',
            api  => 1,
        },
        owners => {
            depends_on => [qw(id)],
            get        => sub {
                $_[0]->{'__OWNERS__'}{$_[1]->{'id'}} // [];
            },
            type       => 'array',
            sub_type   => 'users',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'array',
                check    => sub {
                    my ($qv, $owners) = @_;

                    my @owners_ids;

                    foreach (@$owners) {
                        my $id = ref($_) ? $_->{'id'} : $_;
                        throw Exception::Validator::Fields gettext('id`s must be numbers') unless $id =~ /^[0-9]+$/;
                        push @owners_ids, $id;
                    }

                    my %found_user_ids =
                      map {$_->{'id'} => TRUE}
                      @{$qv->app->users->get_all(fields => ['id'], filter => {id => \@owners_ids})};

                    my @not_found_user_ids = grep {not $found_user_ids{$_}} @owners_ids;

                    return TRUE if @not_found_user_ids == 0;

                    throw Exception::Validator::Fields gettext('These id`s not found: %s',
                        join(', ', @not_found_user_ids));
                  }
            },
        },
        editable_fields => {
            depends_on => [qw(id multistate)],
            get        => sub {
                $_[0]->model->get_editable_fields($_[1]);
            },
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
            api      => 1,
        },
        available_fields => {
            depends_on => [qw(id multistate)],
            label      => d_gettext('Available fields'),
            get        => sub {
                return $_[0]->model->get_available_fields($_[1]);
            },
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
            api      => 1,
        },
        actions => {
            label      => d_gettext('Actions'),
            depends_on => [qw(id multistate owners)],
            get        => sub {
                $_[0]->model->get_actions($_[1]);
            },
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
            api      => 1,
        },
        is_deleted => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'deleted');
            },
            type => 'boolean',
            api  => 1,
        },
        page_ids => {
            depends_on => [qw(id owners)],
            label      => d_gettext('Page ids'),
            get        => sub {
                return @{$_[1]->{'owners'}}
                  ? $_[0]->{'__PAGE_IDS__'}{$_[1]->{'id'}} // []
                  : $_[0]->{'__LINKS_PAGES__'}{$_[1]->{'id'}} // [];
            },
            type     => 'array',
            sub_type => 'number',
            api      => 1,
        },
        public_id => {
            db          => TRUE,
            db_expr     => 'id',
            type        => 'string',
            api         => 1,
            adjust_type => 'str',
        },
        fields_depends => {
            depends_on => ['id'],
            get        => sub {
                {};
            },
            type => 'complex',
            api  => 1,
        },
        status => {
            depends_on => ['id'],
            get        => sub {
                'sinc'
            },
            type => 'string',
            api  => 1,
        },
        trace_back_reference => {
            db         => TRUE,
            label      => d_gettext('Enable trace-back-reference option'),
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },
    };
}

sub get_structure_model_filter {
    return {
        db_accessor => 'partner_db',
        fields      => {
            id         => {type => 'number',     label => d_gettext('ID')},
            tag        => {type => 'text',       label => d_gettext('Tag')},
            data_key   => {type => 'text',       label => d_gettext('Data key')},
            multistate => {type => 'multistate', label => d_gettext('Status')},
            owners     => {
                type           => 'subfilter',
                model_accessor => 'cookie_match_owners',
                field          => 'id',
                fk_field       => 'tag_id',
                label          => d_gettext('Owner'),
            },
            comment => {type => 'text', label => d_gettext('Comment')},
            page_id => {
                type      => 'number',
                label     => d_gettext('Page ID'),
                db_filter => sub {
                    my ($model, $filter) = @_;

                    my $query;
                    foreach my $accessor ($model->pages_with_user_synchronization()) {
                        if (defined($query)) {
                            my $page_query = $model->$accessor->query(
                                fields => $model->$accessor->_get_fields_obj(['tag_id']),
                                filter => $model->$accessor->get_db_filter($filter),
                            );

                            $query->union_all($page_query);
                        } else {
                            $query = $model->$accessor->query(
                                fields => $model->$accessor->_get_fields_obj(['tag_id']),
                                filter => $model->$accessor->get_db_filter($filter),
                            );
                        }
                    }

                    return ['id' => '= ANY' => $query];
                  }
            },
            pi_page => {
                type           => 'subfilter',
                model_accessor => 'cookie_match_link_pi',
                field          => 'id',
                fk_field       => 'tag_id',
                label          => d_gettext('Page from PI'),
            },
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [{name => 'tag', label => gettext('Source ID')},],
        [
            {name => 'page_id',         label => gettext('Page ID')},
            {name => 'pi_page.page_id', label => gettext('Page ID from PI')},
        ],
    ];
}

sub get_structure_multistates_graph {
    return {
        empty_name => d_pgettext('Cookie match', 'New'),
        multistates =>
          [['new' => d_pgettext('Cookie match', 'New')], [deleted => d_pgettext('Cookie match', 'Deleted')],],
        right_actions => {
            add                    => d_pgettext('Cookie match', 'Add'),
            edit                   => d_pgettext('Cookie match', 'Edit'),
            delete                 => d_pgettext('Cookie match', 'Delete'),
            restore                => d_pgettext('Cookie match', 'Restore'),
            link_with_page_from_pi => d_pgettext('Cookie match', 'Link with page from PI'),
            unlink                 => d_pgettext('Cookie match', 'Unlink tag'),
        },
        right_group        => [cookie_match => d_gettext('Right to manage Cookie match')],
        right_name_prefix  => 'cookie_match_',
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['new'],
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action => 'link_with_page_from_pi',
                from   => 'not deleted',
            },
            {
                action => 'unlink',
                from   => 'not deleted',
            },
        ],
    };
}

sub pages_with_user_synchronization {qw(context_on_site_campaign search_on_site_campaign video_an_site)}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my $ids;
    $ids = array_uniq(map {$_->{'id'}} @$result) if $fields->need('page_ids') || $fields->need('create_date');

    if ($fields->need('owners')) {
        my $owners = $self->cookie_match_owners->get_all(
            fields => [qw(tag_id user_id)],
            filter => {tag_id => array_uniq(map {$_->{'id'}} @$result)}
        );

        my %users = map {$_->{'id'} => $_} @{
            $self->users->get_all(
                fields => [qw(id login client_id)],
                filter => {id => array_uniq(map {$_->{'user_id'}} @$owners)},
            )
          };

        foreach (@$owners) {
            push(@{$fields->{'__OWNERS__'}{$_->{'tag_id'}}}, $users{$_->{'user_id'}});
        }
    }

    if ($fields->need('page_ids')) {
        $fields->{'__PAGE_IDS__'}    = {};
        $fields->{'__LINKS_PAGES__'} = {};

        foreach my $page_accessor ($self->pages_with_user_synchronization()) {
            foreach (
                @{
                    $self->$page_accessor->get_all(
                        fields => [qw(page_id tag_id)],
                        filter => {tag_id => $ids},
                    )
                }
              )
            {
                $fields->{'__PAGE_IDS__'}{$_->{'tag_id'}} //= [];

                push(@{$fields->{'__PAGE_IDS__'}{$_->{'tag_id'}}}, $_->{'page_id'});
            }
        }

        foreach (
            @{
                $self->cookie_match_link_pi->get_all(
                    fields => [qw(tag_id page_id)],
                    filter => {tag_id => $ids},
                )
            }
          )
        {
            $fields->{'__LINKS_PAGES__'}{$_->{'tag_id'}} //= [];

            push(@{$fields->{'__LINKS_PAGES__'}{$_->{'tag_id'}}}, $_->{'page_id'});
        }
    }

    if ($fields->need('create_date')) {
        $fields->{'__ACTION_LOGS__'} = {};

        foreach (
            @{
                $self->partner_db_action_log_table()->get_all(
                    fields => [qw(elem_id dt)],
                    filter => {elem_id => $ids, action => 'add'},
                )
            }
          )
        {
            $fields->{'__ACTION_LOGS__'}{$_->{'elem_id'}} = $_->{'dt'};
        }
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my %fields = map {$_ => TRUE} keys(%{$self->get_model_fields});

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(\%fields, {$accessor . '_view_field__%s' => 'comment',});

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $res = {};
    foreach (qw(tag data_key owners skip_data_key js_redir trace_back_reference)) {
        $res->{$_} = TRUE;
    }

    $res->{'comment'} = TRUE if $self->check_short_rights('add_field__comment');

    return $res;
}

sub get_editable_fields {
    my ($self, $obj) = @_;

    return {} unless $self->check_action($obj, 'edit');

    my $res = {};
    foreach (qw(skip_data_key js_redir owners trace_back_reference)) {
        $res->{$_} = TRUE;
    }

    $res->{'comment'} = TRUE if $self->check_short_rights('edit_field__comment');

    return $res;
}

sub add {
    my ($self, %opts) = @_;

    $self->_trim_opts(\%opts);

    my $allowed_fields = $self->get_add_fields();

    $self->check_bad_fields_in_add(\%opts, $allowed_fields);

    $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%{$allowed_fields})]),
    );

    my $id;
    try {
        $self->partner_db->transaction(
            sub {
                my $owners = delete($opts{'owners'});

                $id = $self->partner_db->cookie_match->add(\%opts);

                if (defined($owners) && @$owners) {
                    $self->cookie_match_owners->replace($id, $owners);

                    $opts{'owners'} = $owners;
                }

                $self->do_action($id, 'add', %opts);
            }
        );
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments gettext('Tag "%s" already exists', $opts{'tag'});
    };

    return $id;
}

sub on_action_add {
    my ($self, $obj, %opts) = @_;

    $self->create_or_update_in_bk(%opts);
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;

    $self->_trim_opts(\%opts);

    my $editable_fields = $self->check_bad_fields($obj->{id}, \%opts);

    my $object = $self->get($obj, fields => [qw(tag data_key)]);

    my $old_obj_settings = $self->get($obj, fields => [sort keys(%$editable_fields)]);
    my %new_obj_settings = (%$old_obj_settings, %opts);

    $self->app->validator->check(
        data  => \%new_obj_settings,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%$editable_fields)]),
    );

    my @fields = keys(%opts);

    if (defined($opts{'owners'})) {
        $self->cookie_match_owners->replace($obj->{'id'}, delete($opts{'owners'}));
    }

    if (%opts) {
        $self->partner_db->cookie_match->edit($obj, \%opts);

        $self->create_or_update_in_bk(%new_obj_settings, tag => $object->{'tag'}, data_key => $object->{'data_key'})
          if $self->need_update_in_bk(\@fields);
    }

    return TRUE;
}

sub can_action_link_with_page_from_pi {
    my ($self, $obj) = @_;

    return !@{$self->_get_object_fields($obj, ['owners'])->{'owners'}};
}

sub on_action_link_with_page_from_pi {
    my ($self, $obj, %opts) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "page_id"') unless defined($opts{'page_id'});

    my $tag = $self->_get_object_fields($obj, [qw(tag)])->{'tag'};

    $self->cookie_match_link_pi->add(tag_id => $obj->{'id'}, tag => $tag, page_id => $opts{'page_id'});
}

sub can_action_unlink {
    my ($self, $obj) = @_;

    return !@{$self->_get_object_fields($obj, ['owners'])->{'owners'}};
}

sub on_action_unlink {
    my ($self, $obj, %opts) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "page_id"') unless defined($opts{'page_id'});

    $self->cookie_match_link_pi->delete({tag_id => $obj->{'id'}, page_id => $opts{'page_id'}});
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_special($filter, 'id', 'cookie_match_owners', 'tag_id');

    return $filter;
}

sub create_or_update_in_bk {
    my ($self, %opts) = @_;

    $self->api_http_bk->import_cookie_match_settings(%opts);
}

sub get_by_user_id {
    my ($self, $user_id) = @_;

    return $self->get_all(
        fields => [qw(id tag data_key skip_data_key js_redir)],
        filter => ['owners', 'MATCH', ['user_id', '=', $user_id]]
    );
}

sub need_update_in_bk {
    my ($self, $fields) = @_;

    my %fields_need_update = map {$_ => TRUE} qw(
      skip_data_key
      js_redir
      trace_back_reference
      );

    foreach (@$fields) {
        return TRUE if $fields_need_update{$_};
    }

    return FALSE;
}

sub _generate_data_key {
    my ($self) = @_;

    my @symbols = ('a' .. 'z', 0 .. 9);

    my $result = '';
    for (0 .. 31) {
        $result .= $symbols[int(rand(@symbols))];
    }

    return $result;
}

sub _trim_opts {
    my ($self, $opts) = @_;

    foreach (qw(tag comment)) {
        $opts->{$_} =~ s/^\s+|\s+$//g if defined($opts->{$_});
    }
}

sub api_available_actions {
    return qw(
      add
      edit
      delete
      restore
      link_with_page_from_pi
      unlink
      );
}

sub api_can_add {TRUE}

sub api_can_edit {TRUE}

TRUE;
