package Application::Model::CurUser;

use qbit;

use base qw(QBit::Application::Model);

use Exception::Denied;

use PiConstants qw($TABLE_PAGE_MODELS);

sub accessor {'cur_user'}

__PACKAGE__->model_accessors(
    rbac            => 'Application::Model::RBAC',
    product_manager => 'Application::Model::ProductManager',
);

=encoding UTF-8

=head2 get_all_page_ids_available_for_cur_user

Возвращате ARRAY со списком Page ID к которым пользователь имеет доступ.

Наличие Page ID в этом списке не означает что пользователь может его
редактировать, это ознчает только то что он может его видеть.

Так же наличие Page ID в этом списке ничего не говорит о статусе этого
Page ID в БК. Page ID из этого списка могут быть как обычными пейджами
в БК, так и пейджами со статусом Protected или пейджами со статусом ReadOnly.

=cut

sub get_all_page_ids_available_for_cur_user {
    my ($self) = @_;

    my $can_view_all_page_ids = $self->rbac->cur_user_is_internal();
    my $user_id               = $self->get_cur_user_id();

    my @page_ids;

    if ($can_view_all_page_ids) {
        @page_ids = map {$_->{'page_id'}} @{$self->app->all_pages->get_all(fields => [qw(page_id)])};
    } else {
        my $page_accessors          = {map {$_ => 1} @{$self->product_manager->get_page_model_accessors()}};
        my $internal_page_accessors = {map {$_ => 1} @{$self->product_manager->get_internal_product_accessors}};
        my $special_page_accessors  = {map {$_ => 1} @{$self->product_manager->get_special_page_model_accessors}};
        my $available_page_resources = $self->app->resources->get_available_resources([keys %$page_accessors]);

        my $available_page_accessors = [
            grep   {!$internal_page_accessors->{$_}}                 # фильтруем internal
              grep {!$special_page_accessors->{$_}}                  # фильтруем distribution
              grep {$page_accessors->{$_}}                           # фильтруем "*_deleted" ресурсы
              map  {$_->{'resource'}} @{$available_page_resources}
        ];

        # сначала делаем запрос в таблицу "pages"
        if (grep {$TABLE_PAGE_MODELS->{$_}} @$available_page_accessors) {
            push @page_ids, map {$_->{page_id}} @{
                $self->app->partner_db->pages->get_all(
                    fields => [qw(page_id)],
                    filter => {owner_id => $user_id},
                )
              };
        }

        # потом в отдельные модели
        my @separate_page_accessors = (grep {!$TABLE_PAGE_MODELS->{$_}} @$available_page_accessors);
        if (@separate_page_accessors) {
            foreach my $page_accessor (@separate_page_accessors) {
                push @page_ids, map {$_->{page_id}} @{
                    $self->app->$page_accessor->get_all(
                        fields => [qw(page_id)],
                        filter => {owner_id => $user_id},
                    )
                  };
            }
        }

        if ($self->check_rights('is_assistant')) {
            push @page_ids,
              map {$_->{'page_id'}} @{$self->app->assistants->get_assistants(filter => {user_id => $user_id})};
        }
    }

    my @sorted_page_ids = sort {$a <=> $b} @page_ids;

    return @sorted_page_ids;
}

sub get_all_dsp_ids_available_for_cur_user {
    my ($self) = @_;

    my $filter;
    unless ($self->rbac->cur_user_is_internal()) {
        $filter = {owner_id => $self->get_cur_user_id()};
    }
    my @dsp_ids = map {$_->{id}} @{
        $self->app->dsp->get_all(
            fields => [qw(id)],
            ($filter ? (filter => $filter) : ())
        )
      };

    return sort {$a <=> $b} @dsp_ids;
}

=head2 get_all_page_ids_owned_by_cur_user_for_global_concurents

Возвращает ссылку на массив хэшей со списоком всех пейджей у которых текущий пользователь
является владельцем и на которые влияют глобальные конкуренты.

    [
        {
            page_id => 1532353673,
            id      => 452,
            model   => 'context_on_site_campaign'
        },
        ...
    ]

Метод не возвращет пейдж если его нельзя отправить в БК.

=cut

sub get_all_page_ids_owned_by_user_for_global_concurents {
    my ($self, $user_id) = @_;

    my @pages = ();

    foreach my $page_accessor (grep {$self->app->$_->is_external_page()}
        @{$self->product_manager->get_page_model_accessors()})
    {
        my $model = $self->app->$page_accessor;
        my $pages = $model->get_all(
            fields => [qw(id page_id)],
            filter => [
                'AND' => [
                    {multistate => $model->get_multistate_by_action('set_need_update')},
                    {owner_id   => $user_id},
                    (
                        $model->get_multistate_by_name('balance_registered')
                        ? {multistate => $model->get_multistates_by_filter('balance_registered')}
                        : ()
                    ),
                ]
            ],
        );

        foreach (@$pages) {
            $_->{'model'} = $page_accessor;

            push(@pages, $_);
        }
    }

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(ssp_mobile_app_settings_view_all ssp_context_on_site_campaign_view_all ssp_video_an_site_view_all));

    foreach my $page_accessor (qw(ssp_mobile_app_settings ssp_context_on_site_campaign ssp_video_an_site)) {
        my $pages = $self->app->$page_accessor->get_all(
            fields => [qw(id page_id)],
            filter => {multistate => 'working or testing', owner_id => $user_id},
        );

        foreach (@$pages) {
            $_->{'model'} = $page_accessor;

            push(@pages, $_);
        }
    }

    return \@pages;
}

=head2 get_cur_user_id

Возвращает user_id текущего пользователя или падает с ошибкой если нет
текущего пользователя.

=cut

sub get_cur_user_id {
    my ($self) = @_;

    my $user_id = $self->get_option('cur_user', {})->{'id'};
    throw Exception::Denied gettext('Unknown user') unless defined $user_id;

    return "$user_id";
}

TRUE;
