package Application::Model::CustomBKOptions;

use qbit;

use Exception::Validator::Fields;

use base qw(
  Application::Model::Common
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  );

sub accessor      {'custom_bk_options'}
sub db_table_name {'custom_bk_options'}

sub get_product_name {
    gettext('custom_bk_options');
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {partner_db => 'Application::Model::PartnerDB',};
}

sub get_structure_rights_to_register {
    my ($self) = @_;
    return [
        {
            name        => 'custom_bk_options',
            description => d_gettext('Rights to use custom bk options'),
            rights      => {
                custom_bk_options_view_action_log => d_gettext('Right to view %s', 'custom_bk_options log'),
                custom_bk_options_view            => d_gettext('Right to view %s', 'custom_bk_options'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        id        => {default => TRUE, db => TRUE, pk => TRUE, type => 'number', api => 1},
        public_id => {
            db      => TRUE,
            db_expr => 'id',
            type    => 'string',
            api     => 1,
        },
        multistate      => {default => TRUE, db => TRUE, type => 'number', api => 1},
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1
        },
        fields_depends => {
            get => sub {
                return $_[0]->model->get_fields_depends();
            },
            type => 'complex',
            api  => 1,
        },
        name => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {len_max => 32, len_min => 1,},
        },
        description => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {len_max => 255, len_min => 1,},
        },
        bk_name => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                check => sub {
                    my ($qv, $value) = @_;

                    throw Exception::Validator::Fields gettext('Options should have sublevel') if $value !~ /\./;

                    foreach (split /\./, $value) {
                        throw Exception::Validator::Fields gettext('Sublevels of option should not be empty')
                          unless length($_);
                    }
                },
                len_max => 255,
                len_min => 1,
            },
        },
        bk_value => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {len_max => 512, len_min => 1,},
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;
    return {
        db_accessor => 'partner_db',
        fields      => {
            id         => {type => 'number',     label => d_gettext('ID')},
            multistate => {type => 'multistate', label => d_gettext('Status')},
            name       => {type => 'text',       label => d_gettext('Name')},
            bk_name    => {type => 'text',       label => d_gettext('BK name')},
        },
    };
}

sub get_available_fields_depends {[]}

sub get_editable_fields_depends {[qw(id)]}

sub get_actions_depends {
    [qw(id multistate)];
}

# число (0 - active; 1 - deleted)
sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name    => d_gettext('Active'),
        multistates   => [[deleted => d_gettext('Deleted')],],
        right_actions => {
            delete  => d_gettext('Delete'),
            add     => d_gettext('Add'),
            edit    => d_gettext('Edit'),
            restore => d_gettext('Restore'),
        },
        right_group        => [custom_bk_options_action => d_gettext('Right to manage custom_bk_options')],
        right_name_prefix  => 'custom_bk_options_action_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action => 'edit',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
        ],
    };
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %res = map {$_ => TRUE} qw(
      name
      description
      bk_name
      bk_value
      );

    return \%res;
}

sub get_editable_fields {
    my ($self, $object) = @_;

    my %res = map {$_ => TRUE} qw(
      name
      description
      bk_name
      bk_value
      );

    return \%res;
}

sub api_available_actions {
    return qw(delete edit restore);
}

sub api_can_add {TRUE}

sub api_can_edit {TRUE}

sub get_fields_depends {return {}}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result;

    return \%result;
}

sub hook_fields_processing_before_validation {
    my ($self, $obj) = @_;

    $self->SUPER::hook_fields_processing_before_validation($obj);

    $obj->{bk_name} = 'CustomOptions.' . $obj->{bk_name} if $obj->{bk_name} !~ /\./;
}

1;
