package Application::Model::DBDumper;

use qbit;

use base qw(QBit::Application::Model);

use File::Temp qw(tempfile);
use Data::Rmap qw(rmap_all);
use Digest::SHA1 qw(sha1);

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB',);

my $ONLY_DIRECT_RELATIONSHIPS_TABLES = [
    qw(
      currencies
      users
      site
      bk_language
      dsp
      ya_categories
      tns_dict_brand
      tns_dict_article
      tnd_dict_advertiser
      user_adfox
      )
];

my $NON_EXISTS_KEYS;

sub accessor {'db_dumper'}

sub dump {
    my ($self, %opts) = @_;

    $NON_EXISTS_KEYS //= {
        context_on_site_rtb => [
            [['campaign_id'] => 'context_on_site_campaign'          => ['page_id']],
            [['campaign_id'] => 'internal_context_on_site_campaign' => ['id']],
            [['campaign_id'] => 'mobile_app_settings'               => ['context_page_id']],
            [['campaign_id'] => 'internal_mobile_app'               => ['application_id']],
        ],
        assistants => [
            [['page_id'] => 'mobile_app_settings' => ['context_page_id']],
            [['page_id'] => 'video_an_site'       => ['id']],
            map {[['page_id'] => $_ => ['page_id']]} qw(context_on_site_campaign search_on_site_campaign)
        ],
        block_dsps => [
            map {[['page_id', 'block_id'] => $_ => [$self->app->$_->get_page_id_field_name, 'id']]}
              qw(
              context_on_site_rtb
              context_on_site_adblock
              mobile_app_rtb
              internal_context_on_site_content
              internal_mobile_app_rtb
              video_an_site_instream
              video_an_site_inpage
              video_an_site_fullscreen
              indoor_block
              )
        ],
        managers =>
          [map {[['page_id'] => $_ => ['id']]} qw(internal_context_on_site_campaign internal_search_on_site_campaign)],
        brands => [
            [['page_id', 'block_id'] => context_on_site_adblock => ['page_id',     'id']],
            [['page_id', 'block_id'] => mobile_app_rtb          => ['campaign_id', 'id']],
            [['page_id', 'block_id'] => internal_mobile_app_rtb => ['campaign_id', 'id']],
            (
                map {[['page_id', 'block_id'] => $_ => ['page_id', 'id']]}
                  qw(
                  video_an_site_inpage
                  video_an_site_instream
                  video_an_site_fullscreen
                  )
            ),
            map {[['page_id', 'block_id'] => $_ => ['campaign_id', 'id']]}
              qw(
              context_on_site_adfox
              context_on_site_rtb
              )
        ],
        media_sizes => [
            [['page_id', 'block_id'] => context_on_site_rtb     => ['campaign_id', 'id']],
            [['page_id', 'block_id'] => internal_mobile_app_rtb => ['campaign_id', 'id']],
            [['page_id', 'block_id'] => mobile_app_rtb          => ['campaign_id', 'id']],
            [['page_id', 'block_id'] => ssp_context_on_site_rtb => ['campaign_id', 'id']],
            [['page_id', 'block_id'] => ssp_mobile_app_rtb      => ['campaign_id', 'id']],
            [['page_id', 'block_id'] => ssp_video_an_site_rtb   => ['campaign_id', 'id']],
        ],
    };

    my $page_ids = $opts{'page_ids'};
    my $logins   = $opts{'logins'};
    my $date     = $opts{'date'};

    my $filename_part = $opts{'filename_part'} // '';

    my @filter = ();

    push(@filter, [page_id => 'IN' => \$page_ids]) if defined($page_ids) && @$page_ids;
    push(
        @filter,
        [
            owner_id => '= ANY' => $self->partner_db->query->select(
                table  => $self->partner_db->users,
                fields => [qw(id)],
                filter => [login => 'IN' => \$logins]
            )
        ]
    ) if defined($logins) && @$logins;

    throw Exception::Validation::BadArguments gettext("Expected 'page_ids' or 'logins'") unless @filter;

    my $conf = $self->get_option('partner_db');

    my ($conf_fh, $conf_name) = tempfile('partner_db_config_XXXXXX', TMPDIR => 1, UNLINK => 1);

    print $conf_fh sprintf(
        '[client]
user            = %s
host            = %s
port            = %s
password        = %s
',
        map {$conf->{$_}} qw(user host port password)
    );

    my $mysqldump = sprintf('mysqldump --defaults-extra-file=%s --insert-ignore --compact --no-create-info -B %s -t -c',
        $conf_name, $conf->{'database'});

    my $pages = $self->partner_db->query->select(
        table  => $self->partner_db->all_pages,
        fields => [qw(model page_id)],
        filter => ['OR', \@filter],
    )->get_all();

    my $meta   = $self->partner_db->get_all_meta();
    my @tables = keys(%{$meta->{'tables'}});

    my $depends = {};

    foreach my $table (@tables) {
        my $foreign_keys = $self->partner_db->$table->foreign_keys();

        foreach my $fk (@$foreign_keys) {
            $self->_get_depends($depends, $table, $fk);
        }

        if (exists($NON_EXISTS_KEYS->{$table})) {
            foreach my $fk (@{$NON_EXISTS_KEYS->{$table}}) {
                $self->_get_depends($depends, $table, $fk, 'inverse_relationship');
            }
        }
    }

    my @sort_tables = $self->partner_db->_sorted_tables(@tables);

    my $dir = $self->get_option('pages_dump_dir', '');
    $dir = $self->get_option('ApplicationPath') . $dir unless $dir =~ m/^\//;

    my ($dump_fh, $dump_name) = tempfile("db_dump_$filename_part.XXXXXX", DIR => $dir, UNLINK => 0);

    my $filters = {};

    foreach my $page (@$pages) {
        my $model      = $page->{'model'};
        my $page_field = $self->app->$model->get_page_id_field_name();

        $self->_get_filters($filters, $depends, $model, [$page_field => '=' => \$page->{'page_id'}], 'START', $date);

        foreach my $table (@sort_tables) {
            if (exists($filters->{$table})) {
                my @filter = ();
                foreach (values(%{$filters->{$table}})) {
                    push(@filter, $_);
                }

                my $sql = $self->_get_sql($table, ['OR', \@filter]);

                my $cmd = sprintf($mysqldump . ' --where="%s" --tables %s >> %s', $sql, $table, $dump_name);

                system($cmd);
            }
        }
    }

    return $dump_name;
}

sub _get_depends {
    my ($self, $depends, $table, $fk, $relationship) = @_;

    $relationship //= 'direct_relationship';

    $depends->{$table}{$relationship}{$fk->[1]} = $fk;

    my @rfk = reverse(@$fk);
    $rfk[1] = $table;

    $relationship = $relationship eq 'direct_relationship' ? 'inverse_relationship' : 'direct_relationship';

    $depends->{$fk->[1]}{$relationship}{$table} = \@rfk;
}

sub _get_filters {
    my ($self, $filters, $depends, $table, $filter, $table_from, $date) = @_;

    my $clone_filter = clone($filter);
    rmap_all {$_ = ref($_) eq 'REF' || ref($_) eq 'SCALAR' ? $$_ : $_} $clone_filter;
    my $sha1 = sha1(to_json($clone_filter));

    return if $filters->{$table}{$sha1};
    $filters->{$table}{$sha1} = $filter;

    my @relationships = ('direct_relationship');
    push(@relationships, 'inverse_relationship')
      if not in_array($table, [@$ONLY_DIRECT_RELATIONSHIPS_TABLES, keys(%$NON_EXISTS_KEYS)]);

    my %fields = ();
    foreach my $fk (map {values(%$_)} @{$depends->{$table}}{@relationships}) {
        foreach (@{$fk->[0]}) {
            $fields{$_} = '';
        }
    }

    return unless %fields;

    my $query = $self->partner_db->query->select(
        table  => $self->partner_db->$table,
        fields => \%fields,
        filter => $filter
    )->distinct;

    my $rows = $query->get_all();

    return TRUE unless @$rows;

    foreach my $relationship (@relationships) {
        foreach my $table_name (sort keys(%{$depends->{$table}{$relationship} // {}})) {
            my $fk = $depends->{$table}{$relationship}{$table_name};

            my $table_filter;

            if (@{$fk->[2]} > 1) {
                my @sub_filters = ();
                my %unique      = ();
                foreach my $row (@$rows) {
                    my $key = join($;, map {$row->{$fk->[0][$_]}} 0 .. $#{$fk->[0]});

                    unless (exists($unique{$key})) {
                        $unique{$key} = ['AND', [map {[$fk->[2][$_], '=', \$row->{$fk->[0][$_]}]} 0 .. $#{$fk->[2]}]];
                    }
                }

                $table_filter = ['OR', [values(%unique)]];
            } else {
                $table_filter = [$fk->[2][0], 'IN', \array_uniq(map {$_->{$fk->[0][0]} // ()} @$rows)];
            }

            if ($table_name =~ /^statistics/) {
                $table_filter = ['AND', [['dt', '>=', \$date], $table_filter]];
            }

            $self->_get_filters($filters, $depends, $table_name, $table_filter, $table, $date);
        }
    }
}

sub _get_sql {
    my ($self, $table_name, $filter) = @_;

    my $table = $self->partner_db->$table_name;
    my $query = $self->partner_db->query->select(table => $table, fields => {});

    my $filter_expr = $self->partner_db->filter($filter)->expression();

    local $query->{'__WITHOUT_TABLE_ALIAS__'} = TRUE;
    my ($filter_sql) = $query->_field_to_sql(undef, $filter_expr, $query->_get_table($table));

    $filter_sql =~ s/\n/ /g;
    $filter_sql =~ s/\s+/ /g;
    $filter_sql =~ s/`/\\`/g;

    return $filter_sql;
}

TRUE;
