package Application::Model::DSP;

=encoding UTF-8

=cut

use qbit;
use QBit::Array;

use base qw(Application::Model::Common RestApi::MultistateModel Application::Model::ValidatableMixin);

use PiConstants qw(
  $DSP_MAIL
  $DSP_DIRECT_ID
  @VIDEO_INTERSTITIAL_DSPS
  $ADINSIDE_USER_ID
  $DSP_MEDIA_TYPE_ID
  $DSP_VIDEO_TYPE_ID
  $DSP_MOBILE_TYPE_ID
  $DSP_BANNER_FORMAT_ID
  $DSP_NATIVE_FORMAT_ID
  $DSP_VIDEO_FORMAT_ID
  @DSP_MODE
  );

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::IncorrectParams;

use Utils::Logger qw(INFO WARN);

sub accessor      {'dsp'}
sub db_table_name {'dsp'}

sub get_product_name {gettext('DSP')}

sub get_structure_model_accessors {
    my ($self) = @_;

    return {
        all_pages                    => 'Application::Model::AllPages',
        api_http_bk                  => 'QBit::Application::Model::API::Yandex::HTTPBK',
        api_selfservice              => 'Application::Model::API::Yandex::SelfService',
        block_dsps                   => 'Application::Model::Block::DSPs',
        block_dsps_unmoderated       => 'Application::Model::Block::DSPsUnmoderated',
        context_on_site_adblock      => 'Application::Model::Product::AN::ContextOnSite::AdBlock',
        context_on_site_campaign     => 'Application::Model::Product::AN::ContextOnSite::Campaign',
        context_on_site_rtb          => 'Application::Model::Product::AN::ContextOnSite::RTB',
        indoor_block                 => 'Application::Model::Block::InDoor',
        internal_context_on_site_rtb => 'Application::Model::Product::InternalAN::InternalContextOnSite::RTB',
        kv_store                     => 'QBit::Application::Model::KvStore',
        mail_notification            => 'Application::Model::MailNotification',
        mobile_app_rtb               => 'Application::Model::Product::AN::MobileApp::RTB',
        outdoor_block                => 'Application::Model::Block::OutDoor',
        partner_db                   => 'Application::Model::PartnerDB',
        rbac                         => 'Application::Model::RBAC',
        video_an_site_fullscreen     => 'Application::Model::Product::VideoAN::Site::Fullscreen',
        video_an_site_inpage         => 'Application::Model::Product::VideoAN::Site::InPage',
        video_an_site_instream       => 'Application::Model::Product::VideoAN::Site::InStream',
        users                        => 'Application::Model::Users',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    return [
        {
            name        => 'dsp',
            description => d_gettext('Rights to use dsp'),
            rights      => {
                do_dsp_action_test                      => d_gettext('Right to use DSP testing tool'),
                dsp_add_other                           => d_gettext('Right to add dsp for other user'),
                dsp_edit_all                            => d_gettext('Right to edit all dsp'),
                dsp_view                                => d_gettext('Right to view self dsp'),
                dsp_view_action_log                     => d_gettext('Right to view dsp log'),
                dsp_view_all                            => d_gettext('Right to view all dsp'),
                dsp_view_field__data_key                => d_gettext('Right to view field "data_key"'),
                dsp_view_field__id                      => d_gettext('Right to view field "id"'),
                dsp_view_field__login                   => d_gettext('Right to view field login'),
                dsp_view_field__multistate              => d_gettext('Right to view field multistate'),
                dsp_view_field__patch                   => d_gettext('Right to view field patch'),
                dsp_view_field__postmoderated           => d_gettext('Right to view field "postmoderated"'),
                dsp_view_field__short_caption           => d_gettext('Right to view field "short_caption"'),
                dsp_view_field__skipnoud                => d_gettext('Right to view field "skipnoud"'),
                dsp_view_field__unmoderated_rtb_auction => d_gettext('Right to view field "unmoderated_rtb_auction"'),
                dsp_view_rtb_host_response              => d_gettext('Right to view rtb host response'),
                dsp_edit_field__postmoderated           => d_gettext('Right to edit field "postmoderated"'),
                dsp_edit_field__types                   => d_gettext('Right to edit field "types"'),
                dsp_edit_field__formats                 => d_gettext('Right to edit field "formats"'),
                dsp_edit_field__unmoderated_rtb_auction => d_gettext('Right to edit field "unmoderated_rtb_auction"'),
                dsp_statistics_responses                => d_gettext('Right to view DSP responses statistics'),
            }
        }
    ];
}

my %TYPE = (
    $DSP_MEDIA_TYPE_ID  => d_gettext('Media advertising'),
    $DSP_VIDEO_TYPE_ID  => d_gettext('Video advertising'),
    $DSP_MOBILE_TYPE_ID => d_gettext('Mobile advertising'),
);

my %FORMAT = (
    $DSP_BANNER_FORMAT_ID => d_gettext('Banner advertising'),
    $DSP_NATIVE_FORMAT_ID => d_gettext('Native advertising'),
    $DSP_VIDEO_FORMAT_ID  => d_gettext('Video advertising'),
);

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        id => {
            api          => TRUE,
            type         => 'number',
            default      => TRUE,
            db           => TRUE,
            pk           => TRUE,
            type         => 'number',
            check_rights => 'dsp_view_field__id',
            adjust_type  => 'str',
        },
        owner_id => {
            api  => TRUE,
            type => 'number',
            db   => TRUE,
        },
        short_caption => {
            api          => TRUE,
            type         => 'string',
            default      => TRUE,
            db           => TRUE,
            check_rights => 'dsp_view_field__short_caption',
            need_check   => {
                type    => 'scalar',
                len_max => 32,
            }
        },
        display_name => {
            api        => TRUE,
            type       => 'string',
            db         => TRUE,
            need_check => {
                type    => 'scalar',
                len_max => 32,
                len_min => 3,
            }
        },
        multistate => {
            api          => TRUE,
            type         => 'number',
            db           => TRUE,
            check_rights => 'dsp_view_field__multistate'
        },
        types => {
            api        => TRUE,
            type       => 'complex',
            depends_on => ['id'],
            get        => sub {
                $_[0]->{'__DSP_TYPES__'}->{$_[1]->{'id'}} // [];
            },
            adjust_type => 'array_str',
            need_check  => {
                type  => 'array',
                check => sub {
                    my ($qv, $value) = @_;

                    throw Exception::Validation::BadArguments gettext('Types not found')
                      unless $qv->app->check_type($value);
                  }
            }
        },
        formats => {
            api        => TRUE,
            type       => 'complex',
            depends_on => ['id'],
            get        => sub {
                $_[0]->{'__DSP_FORMATS__'}->{$_[1]->{'id'}} // [];
            },
            need_check => {
                type     => 'array',
                optional => TRUE,
            },
            adjust_type => 'array_str',
        },
        type_captions => {
            api        => TRUE,
            type       => 'complex',
            depends_on => 'types',
            get        => sub {
                return [map {$TYPE{$_}()} @{$_[1]->{'types'}}];
            },
        },
        format_captions => {
            api        => TRUE,
            type       => 'complex',
            depends_on => 'formats',
            get        => sub {
                return [map {$FORMAT{$_}()} @{$_[1]->{'formats'}}];
            },
        },
        url => {
            api        => TRUE,
            type       => 'string',
            db         => TRUE,
            need_check => {
                type    => 'scalar',
                len_max => 1024,
            }
        },
        test_url => {
            api        => TRUE,
            type       => 'string',
            db         => TRUE,
            need_check => {
                optional => TRUE,
                type     => 'scalar',
                len_max  => 1024,
            }
        },
        skipnoud => {
            api          => TRUE,
            type         => 'boolean',
            db           => TRUE,
            check_rights => 'dsp_view_field__skipnoud'
        },
        tag => {
            api        => TRUE,
            type       => 'string',
            db         => TRUE,
            need_check => {
                optional => TRUE,
                type     => 'scalar',
                len_max  => 32,
            }
        },
        data_key => {
            api          => TRUE,
            type         => 'string',
            db           => TRUE,
            check_rights => 'dsp_view_field__data_key',
            need_check   => {
                type    => 'scalar',
                len_max => 255,
            }
        },
        show_probability => {
            api        => TRUE,
            type       => 'number',
            db         => TRUE,
            need_check => {
                type => 'int_un',
                max  => 100,
            }
        },
        postmoderated => {
            api          => TRUE,
            type         => 'boolean',
            db           => TRUE,
            check_rights => 'dsp_view_field__postmoderated'
        },
        unmoderated_rtb_auction => {
            api          => TRUE,
            type         => 'boolean',
            db           => TRUE,
            check_rights => 'dsp_view_field__unmoderated_rtb_auction'
        },
        patch => {
            api        => TRUE,
            db         => TRUE,
            label      => d_gettext('DSP patch'),
            type       => 'string',
            need_check => {type => 'json', optional => TRUE},
            get        => sub {
                if ($_[1]->{'patch'}) {
                    utf8::decode($_[1]->{'patch'});
                    return to_json(from_json($_[1]->{'patch'}), pretty => 1,);
                } else {
                    return $_[1]->{'patch'};
                }
            },

        },
        is_ssp_allowed => {
            api  => TRUE,
            type => 'boolean',
            db   => TRUE,
        },
        public_id => {
            api               => TRUE,
            type              => 'number',
            forced_depends_on => 'id',
            get               => sub {
                return $_[1]->{'id'};
            },
        },
        multistate_name => {
            api        => TRUE,
            type       => 'string',
            depends_on => 'multistate',
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
        },
        login => {
            api               => TRUE,
            type              => 'string',
            forced_depends_on => ['owner_id'],
            get               => sub {
                $_[0]->{'USERS'}->{$_[1]->{'owner_id'}} // '';
            },
        },
        is_deleted => {
            api               => TRUE,
            type              => 'boolean',
            forced_depends_on => ['multistate'],
            get               => sub {
                $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'deleted');
              }
        },
        is_created_in_bk => {
            api               => TRUE,
            type              => 'boolean',
            forced_depends_on => ['multistate'],
            get               => sub {
                $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'created_in_bk');
              }
        },
        works_on_all_platforms => {
            api               => TRUE,
            type              => 'boolean',
            forced_depends_on => ['multistate'],
            get               => sub {
                $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'working_on_all_platforms');
              }
        },
        is_dsp_statistics_responses_available => {
            api               => TRUE,
            type              => 'boolean',
            forced_depends_on => ['multistate'],
            get               => sub {
                $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'created_in_bk')
                  && !$_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'deleted');
              }
        },
        actions => {
            api               => TRUE,
            forced_depends_on => [qw(id multistate types)],
            get               => sub {
                $_[0]->model->get_actions($_[1]);
            },
            type => 'complex',
        },
        editable_fields => {
            api               => TRUE,
            forced_depends_on => [qw(id multistate tag)],
            get               => sub {
                return $_[0]->model->get_editable_fields($_[1]);
            },
            type => 'complex',
        },
        available_fields => {
            api               => TRUE,
            forced_depends_on => [qw(id multistate)],
            label             => d_gettext('Available fields'),
            get               => sub {
                return $_[0]->model->get_available_fields($_[1]);
            },
            type => 'complex',
        },
        fields_depends => {
            api        => TRUE,
            depends_on => [qw(id)],
            get        => sub {
                return $_[0]->model->get_fields_depends();
            },
            type => 'complex',

        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            id                      => {type => 'number',     label => d_gettext('ID')},
            owner_id                => {type => 'number',     label => d_gettext('ID owner')},
            short_caption           => {type => 'text',       label => d_gettext('Short caption')},
            tag                     => {type => 'text',       label => d_gettext('Tag')},
            multistate              => {type => 'multistate', label => d_gettext('Status')},
            unmoderated_rtb_auction => {type => 'boolean',    label => d_gettext('Unmoderated RTB auction')},
            owner                   => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'owner_id',
                label          => d_gettext('Login'),
            },
            types => {
                db_filter => sub {
                    my $filter =
                      $_[0]->{'__DB_FILTER__'}{$_[1]->[0]}->as_filter(['type_id' => $_[1]->[1] => $_[1]->[2]], $_[2]);
                    return [
                        id => '= ANY' => $_[0]->partner_db->query->select(
                            table  => $_[0]->partner_db->dsp_type,
                            fields => ['dsp_id'],
                            filter => $filter
                        )
                    ];
                },
                type   => 'dictionary',
                label  => gettext('Types'),
                values => sub {
                    [sort {$a->{'id'} <=> $b->{'id'}} map {{id => $_, label => $TYPE{$_}->()}} keys(%TYPE)];
                  }
            },
            formats => {
                db_filter => sub {
                    my $filter =
                      $_[0]->{'__DB_FILTER__'}{$_[1]->[0]}->as_filter(['format_id' => $_[1]->[1] => $_[1]->[2]], $_[2]);
                    return [
                        id => '= ANY' => $_[0]->partner_db->query->select(
                            table  => $_[0]->partner_db->dsp_format,
                            fields => ['dsp_id'],
                            filter => $filter
                        )
                    ];
                },
                type   => 'dictionary',
                label  => gettext('Formats'),
                values => sub {
                    [sort {$a->{'id'} <=> $b->{'id'}} map {{id => $_, label => $FORMAT{$_}->()}} keys(%FORMAT)];
                  }
            },
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            (
                $self->check_short_rights('view_field__login')
                ? {name => 'owner.login', label => gettext('Login'), login_filter => 'dsp=1'}
                : ()
            ),
            (
                $self->check_rights('users_view_field__client_id')
                ? {
                    name  => 'owner.client_id',
                    label => gettext('Client ID'),
                    hint  => gettext('The unique identifier of a partner in Balance')
                  }
                : ()
            ),
            (
                $self->check_rights('users_view_field__contract_id')
                ? {
                    name  => 'owner.contract_id',
                    label => gettext('Contract ID'),
                  }
                : ()
            ),
            {
                name  => 'types',
                label => gettext('Type'),
            },
            {
                name  => 'formats',
                label => gettext('Format'),
            }
        ],
        [
            ($self->check_short_rights('view_field__id') ? {name => 'id', label => gettext('DSP ID')} : ()),
            (
                $self->check_short_rights('view_field__short_caption')
                ? {name => 'short_caption', label => gettext('Connection')}
                : ()
            ),
            (
                $self->check_short_rights('view_field__multistate') ? {name => 'multistate', label => gettext('Status')}
                : ()
            )
        ]
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_gettext('New'),
        multistates => [
            [created_in_bk              => d_gettext('Created in BK')],
            [created_in_pi              => d_gettext('Task\'s created in PI')],
            [linked_in_balance          => d_gettext('Linked in Balance')],
            [working                    => d_gettext('DSP is working')],
            [need_approve               => d_gettext('Need approve DSP')],
            [edited                     => d_gettext('Edited DSP')],
            [deleted                    => d_gettext('Deleted')],
            [working_on_all_platforms   => d_gettext('Working on all platforms')],
            [working_on_yandex_services => d_gettext('Working on Yandex services')],
            [need_create_in_pi          => d_gettext('Need create in PI')],
            [not_need_create_in_pi      => d_gettext('Not need create in PI')],
        ],
        right_actions => {
            add                            => d_gettext('Add new dsp'),
            create_in_bk                   => d_gettext('Create dsp in BK'),
            create_in_pi                   => d_gettext('Create task in PI'),
            link_in_balance                => d_gettext('Link in Balance'),
            start                          => d_gettext('Turn on the DSP in BS'),
            stop                           => d_gettext('Turn off the DSP in BS'),
            request_for_approval           => d_gettext('Request for approval'),
            approve                        => d_gettext('Approve'),
            edit                           => d_gettext('Edit DSP'),
            test                           => d_gettext('Sandbox'),
            start_on_all_platforms         => d_gettext('Turn on the DSP on AdvNet platforms'),
            stop_on_all_platforms          => d_gettext('Turn off the DSP on Advnet platforms'),
            start_on_yandex_services       => d_gettext('Turn on the DSP on Yandex services'),
            stop_on_yandex_services        => d_gettext('Turn off the DSP on Yandex services'),
            delete                         => d_gettext('Delete'),
            restore                        => d_gettext('Restore'),
            set_flag_need_create_in_pi     => d_gettext('Set flags need create in PI'),
            set_flag_not_need_create_in_pi => d_gettext('Set flags not need create in PI'),
            resend_to_bk                   => d_gettext('Resend to BS'),
            start_on_mobile_block_native   => d_gettext('Start the Native DSP on Mobile Blocks'),
            start_on_mobile_block_banner   => d_gettext('Start the Banner DSP on Mobile Blocks'),
            start_on_mobile_block_video    => d_gettext('Start the Video DSP on Mobile Blocks'),
            stop_on_mobile_block_native    => d_gettext('Stop the Native DSP on Mobile Blocks'),
            stop_on_mobile_block_banner    => d_gettext('Stop the Banner DSP on Mobile Blocks'),
            stop_on_mobile_block_video     => d_gettext('Stop the Video DSP on Mobile Blocks'),
        },
        right_group        => [dsp_action => d_gettext('Right to manage DSP')],
        right_name_prefix  => 'dsp_action_',
        multistate_actions => [
            {
                action    => 'create_in_bk',
                from      => 'not created_in_bk and not need_approve and not deleted',
                set_flags => ['created_in_bk'],
            },
            {
                action      => 'create_in_pi',
                from        => 'need_create_in_pi',
                set_flags   => ['created_in_pi'],
                reset_flags => ['need_create_in_pi'],
            },
            {
                action    => 'link_in_balance',
                from      => 'not linked_in_balance and not need_approve and not deleted',
                set_flags => ['linked_in_balance'],
            },
            {
                action => 'start',
                from =>
'created_in_bk and (created_in_pi or not_need_create_in_pi) and linked_in_balance and not working and not deleted',
                set_flags => ['working']
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working']
            },
            {
                action    => 'request_for_approval',
                from      => '__EMPTY__',
                set_flags => ['need_approve']
            },
            {
                action      => 'approve',
                from        => 'need_approve and not deleted',
                reset_flags => ['need_approve']
            },
            {
                action    => 'edit',
                from      => 'not need_approve and not deleted',
                set_flags => ['edited']
            },
            {
                action => 'start_on_all_platforms',
                from =>
'created_in_bk and (created_in_pi or not_need_create_in_pi) and linked_in_balance and not working_on_all_platforms and not deleted',
                set_flags => ['working_on_all_platforms']
            },
            {
                action      => 'stop_on_all_platforms',
                from        => 'working_on_all_platforms',
                reset_flags => ['working_on_all_platforms']
            },
            {
                action => 'start_on_yandex_services',
                from =>
'created_in_bk and created_in_pi and linked_in_balance and not working_on_yandex_services and not deleted',
                set_flags => ['working_on_yandex_services']
            },
            {
                action      => 'stop_on_yandex_services',
                from        => 'working_on_yandex_services',
                reset_flags => ['working_on_yandex_services']
            },
            {
                action => 'test',
                from   => 'created_in_bk and not deleted',
            },
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action => 'delete',
                from =>
                  'not deleted and not working and not working_on_all_platforms and not working_on_yandex_services',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
            {
                action    => 'set_flag_need_create_in_pi',
                from      => 'not need_create_in_pi and not need_approve and not deleted and not created_in_pi',
                set_flags => ['need_create_in_pi']
            },
            {
                action    => 'set_flag_not_need_create_in_pi',
                from      => 'not not_need_create_in_pi and not need_approve and not deleted and not created_in_pi',
                set_flags => ['not_need_create_in_pi']
            },
            {
                action => 'resend_to_bk',
                from   => 'created_in_bk',
            },
            {
                action => 'start_on_mobile_block_native',
                from   => 'working_on_all_platforms or working_on_yandex_services',
            },
            {
                action => 'start_on_mobile_block_banner',
                from   => 'working_on_all_platforms or working_on_yandex_services',
            },
            {
                action => 'start_on_mobile_block_video',
                from   => 'working_on_all_platforms or working_on_yandex_services',
            },
            {
                action => 'stop_on_mobile_block_native',
                from   => 'working_on_all_platforms or working_on_yandex_services',
            },
            {
                action => 'stop_on_mobile_block_banner',
                from   => 'working_on_all_platforms or working_on_yandex_services',
            },
            {
                action => 'stop_on_mobile_block_video',
                from   => 'working_on_all_platforms or working_on_yandex_services',
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('login')) {
        $fields->{'USERS'} = {
            map {$_->{'id'} => $_->{'login'}} @{
                $self->users->get_all(
                    fields => [qw(id login)],
                    filter => {id => array_uniq(map {$_->{'owner_id'}} @$result)},
                )
              }
        };
    }

    if ($fields->need('types')) {
        foreach (
            @{
                $self->partner_db->query->select(
                    table  => $self->partner_db->dsp_type,
                    fields => [qw(dsp_id type_id)],
                    filter => ['dsp_id' => '=' => \array_uniq(map {$_->{'id'}} @$result)],
                  )->get_all()
            }
          )
        {
            $fields->{'__DSP_TYPES__'}{$_->{'dsp_id'}} //= [];
            push(@{$fields->{'__DSP_TYPES__'}{$_->{'dsp_id'}}}, $_->{'type_id'});
        }
    }

    if ($fields->need('formats')) {
        foreach (
            @{
                $self->partner_db->query->select(
                    table  => $self->partner_db->dsp_format,
                    fields => [qw(dsp_id format_id)],
                    filter => ['dsp_id' => '=' => \array_uniq(map {$_->{'id'}} @$result)],
                  )->get_all()
            }
          )
        {
            $fields->{'__DSP_FORMATS__'}{$_->{'dsp_id'}} //= [];
            push(@{$fields->{'__DSP_FORMATS__'}{$_->{'dsp_id'}}}, $_->{'format_id'});
        }
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    if ($obj && $self->check_multistate_flag($obj->{'multistate'}, 'deleted')) {
        return {};
    }

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => [
                qw( id multistate multistate_name login short_caption data_key
                  skipnoud postmoderated unmoderated_rtb_auction display_name
                  is_ssp_allowed is_dsp_statistics_responses_available)
            ],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    return {} unless $self->check_rights('do_dsp_action_add');
    my %fields = map {$_ => TRUE} qw(
      types
      short_caption
      url
      display_name
      formats
      test_url
      );

    foreach (qw(postmoderated is_ssp_allowed)) {
        $fields{$_} = TRUE if $self->check_short_rights('edit_field__' . $_);
    }

    $fields{'login'} = TRUE if $self->check_short_rights('add_other');

    return \%fields;
}

sub hook_fields_validation {
    my ($self, $obj) = @_;

    $self->SUPER::hook_fields_validation($obj);

    if ($self->hook_stash->mode('add')) {
        throw Exception::Validation::BadArguments gettext("Username must belong to the DSP partner")
          unless $self->check_rights('do_dsp_action_add');
    }
}

sub get_editable_fields {
    my ($self, $object) = @_;

    return {}
      unless $self->check_rights('dsp_edit_all')
          && $self->check_action($object, 'edit');
    my %res = map {$_ => 1} qw(
      short_caption
      url
      test_url
      skipnoud
      data_key
      show_probability
      display_name
      );

    foreach (qw(postmoderated types unmoderated_rtb_auction patch formats is_ssp_allowed)) {
        $res{$_} = TRUE if $self->check_rights('dsp_edit_field__' . $_);
    }

    $res{'tag'} = TRUE unless defined($self->{'tag'});

    return \%res;
}

sub api_can_add {TRUE}

sub get_fields_depends {
    return {
        depends => {
            types   => [qw(type_captions)],
            formats => [qw(format_captions)],
        },
    };
}

sub hook_post_validation {
    my ($self, $opts) = @_;

    if (exists($opts->{'show_probability'})) {
        throw Exception::Validation::BadArguments gettext('Value must be an integer from 1 to 100')
          unless $opts->{'show_probability'} =~ /^\d+$/
              && $opts->{'show_probability'} > 0
              && $opts->{'show_probability'} <= 100;
    }
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    my $dsp_id = $self->hook_stash->get('id')->{id};

    if ($self->hook_stash->mode('add')) {
        $self->partner_db->dsp_type->add_multi([map {{dsp_id => $dsp_id, type_id => $_}} @{$opts->{'types'}}]);
        if (in_array($DSP_MOBILE_TYPE_ID, $opts->{'types'}) && $opts->{'formats'} && @{$opts->{'formats'}}) {
            $self->partner_db->dsp_format->add_multi(
                [map +{dsp_id => $dsp_id, format_id => $_}, @{$opts->{'formats'}}]);
        }
    }
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    my $dsp_id = $self->hook_stash->get('id')->{id};
    my $dsp = $self->_get_object_fields($dsp_id, ['multistate']);

    my $dsp_types;
    my $dsp_formats = delete($opts->{'formats'});
    my $has_mobile;
    if (exists($opts->{'types'})) {
        $dsp_types = delete($opts->{'types'});

        $self->partner_db->dsp_type->delete($self->partner_db->filter({dsp_id => $dsp_id}));

        $self->partner_db->dsp_type->add_multi([map {{dsp_id => $dsp_id, type_id => $_}} @$dsp_types]);
        $has_mobile = in_array($DSP_MOBILE_TYPE_ID, $dsp_types);
    } else {
        $has_mobile = @{
            $self->partner_db->dsp_type->get_all(
                filter => [AND => [[dsp_id => '=' => \$dsp_id], [type_id => '=' => \$DSP_MOBILE_TYPE_ID],]]
            )
          };
    }

    if ($dsp_formats || !$has_mobile) {
        $self->partner_db->dsp_format->delete($self->partner_db->filter({dsp_id => $dsp_id}));
    }
    if ($dsp_formats && $has_mobile) {
        $self->partner_db->dsp_format->add_multi([map {{dsp_id => $dsp_id, format_id => $_}} @$dsp_formats]);
    }

    if ($self->need_update_in_bk($dsp, %$opts)) {
        $self->api_http_bk->edit_dsp($dsp_id, %$opts, (defined($dsp_types) ? (dsp_types => $dsp_types) : ()),);

        $self->api_http_bk->dsp_lb($self->_get_data_for_bk($dsp, %$opts));
    }
}

sub hook_processing_after_insert {
    my ($self, $obj) = @_;

    my $dsp_id = $self->hook_stash->get('id')->{id};

    my $tmp_rights = $self->app->add_tmp_rights('users_view_field__client_id');
    my $owner = $self->users->get($obj->{'owner_id'}, fields => [qw(id login email client_id)]);
    undef $tmp_rights;

    $self->do_action($dsp_id, 'add', %$obj);
    $self->do_action($dsp_id, 'request_for_approval');

    $self->mail_notification->add(
        type    => 0,
        user_id => 0,
        opts    => {
            check_alive => FALSE,
            subject =>
              gettext('DSP Partner "%s" has registered new DSP "%s"', $owner->{'login'}, $obj->{'short_caption'}),
            to     => $DSP_MAIL,
            values => {
                message_body => gettext(
                    "DSP partner has successfully registered new DSP.

Partner login: %s
Short name in the interface: %s
Types: %s
E-Mail: %s.

Several id's in different systems have been created for this partner:
- passport (PARTNER ID): %s
- partner interface and ad server (DSP ID): %s
- balance (CLIENT ID): %s

Manager must confirm DSP creation in the system",
                    $owner->{'login'}, $obj->{'short_caption'},
                    join(', ', map {$TYPE{$_}()} @{$obj->{'types'}}),
                    $owner->{'email'}, $owner->{'id'}, $dsp_id,
                    $owner->{'client_id'}
                ),
                plain_text_wrapper => TRUE,
            },
        },
    );

    $self->_send_dsp_for_billing_to_lb($dsp_id, $owner);
}

sub on_action_delete {
    my ($self, $dsp) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          block_dsps_edit
          )
    );

    $self->block_dsps->edit({dsp_id => $dsp->{'id'}}, {is_deleted => 1});

    return TRUE;
}

sub on_action_approve {
    my ($self, $dsp, %opts) = @_;

    my @missed_req_fields = grep {!defined($opts{$_})} qw(short_caption url tag);
    throw Exception::Validation::BadArguments gettext('Missed required fields: %s', join(', ', @missed_req_fields))
      if @missed_req_fields;

    $self->do_action($dsp->{'id'}, 'edit', map {$_ => $opts{$_}} qw(short_caption url test_url tag));

    $dsp = $self->_get_object_fields($dsp, [qw(types)]);
    if (in_array(0, $dsp->{'types'})) {
        $self->do_action($dsp->{'id'}, 'set_flag_need_create_in_pi');
        my $tmp_rights = $self->app->add_tmp_rights('do_dsp_action_create_in_pi');
        $self->do_action($dsp->{'id'}, 'create_in_pi');
    } else {
        $self->do_action($dsp->{'id'}, 'set_flag_not_need_create_in_pi');
    }
}

sub on_action_create_in_bk {
    my ($self, $dsp) = @_;

    $dsp = $self->_get_object_fields(
        $dsp,
        [
            qw(id tag url test_url types postmoderated short_caption unmoderated_rtb_auction is_ssp_allowed works_on_all_platforms)
        ]
    );

    $dsp->{'works_on_all_platforms'} = $dsp->{'works_on_all_platforms'} ? 1 : 0;

    my $data = $self->api_http_bk->add_or_get_if_exists_dsp(
        id                      => $dsp->{'id'},
        tag                     => $dsp->{'tag'},
        use_pnocsy              => $self->_get_use_pnocsy_value($dsp->{'tag'}),
        url                     => $dsp->{'url'},
        disabled                => TRUE,
        test_url                => $dsp->{'test_url'},
        dsp_types               => $dsp->{'types'},
        postmoderated           => $dsp->{'postmoderated'},
        short_caption           => $dsp->{'short_caption'},
        unmoderated_rtb_auction => $dsp->{'unmoderated_rtb_auction'},
        is_ssp_allowed          => $dsp->{'is_ssp_allowed'},
        works_on_all_platforms  => $dsp->{'works_on_all_platforms'},
    );

    $self->do_action($dsp->{'id'}, 'edit', data_key => $data->{'data_key'})
      if $data;

    $self->api_http_bk->dsp_lb($self->_get_data_for_bk($dsp, disabled => TRUE,));

    return TRUE;
}

sub on_action_link_in_balance {
    my ($self, $dsp) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id));
    my $user = $self->partner_db->users->get($dsp->{owner_id}, fields => [qw(id client_id)]);
    undef($tmp_rights);

    $self->app->api_balance->link_dsp_to_client_with_confirm(
        user_id   => $user->{id},
        DSP_ID    => $dsp->{'id'},
        CLIENT_ID => $user->{client_id}
    );

    return TRUE;
}

sub can_action_resend_to_bk {
    my ($self, $dsp) = @_;

    return $self->api_http_bk->get_option('allow_dsp_via_logbroker');
}

sub on_action_resend_to_bk {
    my ($self, $dsp) = @_;

    $self->api_http_bk->dsp_lb($self->_get_data_for_bk($dsp));
}

sub on_action_restore {
    my ($self, $dsp) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          block_dsps_edit
          )
    );

    $self->block_dsps->edit({dsp_id => $dsp->{'id'}}, {is_deleted => 0});

    return TRUE;
}

sub api_can_delete {FALSE}

sub on_action_start {
    my ($self, $dsp) = @_;

    $self->api_http_bk->start_dsp($dsp->{'id'});
    $self->api_http_bk->dsp_lb($self->_get_data_for_bk($dsp, disabled => FALSE,));
}

sub can_action_start_on_all_platforms {
    my ($self, $obj) = @_;

    $obj = $self->_get_object_fields($obj, [qw(types)]);

    return
      scalar(@{arrays_intersection([$DSP_MEDIA_TYPE_ID, $DSP_VIDEO_TYPE_ID, $DSP_MOBILE_TYPE_ID], $obj->{'types'})});
}

sub on_action_start_on_all_platforms {
    my ($self, $obj) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(video_an_site_view_all mobile_app_settings_view_all context_on_site_campaign_view_all users_view_all));

    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    $self->$_->turn_on_dsp($obj) foreach (
        qw(
        context_on_site_rtb
        video_an_site_instream
        video_an_site_inpage
        video_an_site_fullscreen
        indoor_block
        outdoor_block
        mobile_app_rtb
        )
    );
}

sub can_action_start_on_mobile_block_banner {
    my ($self, $obj) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    return in_array($DSP_MOBILE_TYPE_ID, $obj->{types}) && in_array($DSP_BANNER_FORMAT_ID, $obj->{formats});
}

sub on_action_start_on_mobile_block_banner {
    my ($self, $obj) = @_;
    $self->_start_mobile_block($obj);
}

sub can_action_start_on_mobile_block_native {
    my ($self, $obj) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    return in_array($DSP_MOBILE_TYPE_ID, $obj->{types}) && in_array($DSP_NATIVE_FORMAT_ID, $obj->{formats});
}

sub on_action_start_on_mobile_block_native {
    my ($self, $obj) = @_;
    $self->_start_mobile_block($obj);
}

sub can_action_start_on_mobile_block_video {
    my ($self, $obj) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    return in_array($DSP_MOBILE_TYPE_ID, $obj->{types}) && in_array($DSP_VIDEO_FORMAT_ID, $obj->{formats});
}

sub on_action_start_on_mobile_block_video {
    my ($self, $obj) = @_;
    $self->_start_mobile_block($obj);
}

sub can_action_start_on_yandex_services {
    my ($self, $obj) = @_;

    $obj = $self->_get_object_fields($obj, [qw(types)]);

    return in_array(0, $obj->{'types'});
}

sub on_action_start_on_yandex_services {
    my ($self, $dsp) = @_;

    $dsp = $self->_get_object_fields($dsp, [qw(id multistate tag types)]);
    $self->internal_context_on_site_rtb->turn_on_dsp($dsp);
}

sub on_action_stop {
    my ($self, $dsp) = @_;

    $self->api_http_bk->stop_dsp($dsp->{'id'});
    $self->api_http_bk->dsp_lb($self->_get_data_for_bk($dsp, disabled => TRUE,));
}

sub on_action_stop_on_all_platforms {
    my ($self, $obj) = @_;

    $self->turn_off_dsp($obj->{'id'});
}

sub can_action_stop_on_mobile_block_banner {
    my ($self, $obj) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    return in_array($DSP_MOBILE_TYPE_ID, $obj->{types}) && !in_array($DSP_BANNER_FORMAT_ID, $obj->{formats});
}

sub on_action_stop_on_mobile_block_banner {
    my ($self, $obj) = @_;
    unless ($self->can_action_start_on_mobile_block_video($obj)) {
        $self->_stop_mobile_block($obj, ['banner', 'interstitial']);
    }
}

sub can_action_stop_on_mobile_block_native {
    my ($self, $obj) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    return in_array($DSP_MOBILE_TYPE_ID, $obj->{types}) && !in_array($DSP_NATIVE_FORMAT_ID, $obj->{formats});
}

sub on_action_stop_on_mobile_block_native {
    my ($self, $obj) = @_;
    $self->_stop_mobile_block($obj, ['native']);
}

sub can_action_stop_on_mobile_block_video {
    my ($self, $obj) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    return in_array($DSP_MOBILE_TYPE_ID, $obj->{types}) && !in_array($DSP_VIDEO_FORMAT_ID, $obj->{formats});
}

sub on_action_stop_on_mobile_block_video {
    my ($self, $obj) = @_;
    unless ($self->can_action_start_on_mobile_block_banner($obj)) {
        $self->_stop_mobile_block($obj, ['banner', 'interstitial']);
    }
}

sub on_action_stop_on_yandex_services {
    my ($self, $dsp) = @_;

    $self->turn_off_dsp($dsp->{'id'});
}

sub api_available_actions {
    my ($self) = @_;

    return (
        sort qw(approve edit delete restore start stop test
          start_on_all_platforms start_on_yandex_services
          stop_on_all_platforms stop_on_yandex_services
          start_on_mobile_block_banner start_on_mobile_block_native start_on_mobile_block_video
          stop_on_mobile_block_banner stop_on_mobile_block_native stop_on_mobile_block_video
          statistics_responses
          )
    );
}

sub api_can_edit {TRUE}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_owner($filter);

    return $filter;
}

sub check_dsp_states_in_pi_and_bs {
    my ($self) = @_;

    my $pi_dsps = $self->get_all(
        fields => ['id', 'multistate'],
        filter => {multistate => 'created_in_bk'},
    );

    my %pi_dsp_states =
      map {
        $_->{'id'} => {
            deleted => $self->check_multistate_flag($_->{'multistate'}, 'deleted'),
            status  => $self->check_multistate_flag($_->{'multistate'}, 'working') ? 1 : 0
          }
      } @{$pi_dsps};

    my %pi_dsp_multistates = map {$_->{'id'} => $_->{'multistate'},} @{$pi_dsps};

    my $bs_dsps = $self->api_http_bk->get_dsp_info;

    foreach my $bs_dsp (@{$bs_dsps}) {
        my $dsp_id = $bs_dsp->{'DSPID'};

        # TODO: Sentry formatting
        unless (exists($pi_dsp_states{$dsp_id})) {
            WARN(join "\n", "## DSP ID $dsp_id", "PI don't have DSP ID $dsp_id", "");
            next;
        }

        unless ($pi_dsp_states{$dsp_id}->{'status'} == $bs_dsp->{'Status'}) {
            WARN(
                join "\n",
                "## DSP ID $dsp_id",
                "DSP ID $dsp_id has different state in PI and BS. Details:",
                "PI multistate: " . $pi_dsp_multistates{$dsp_id},
                "PI multistate name:",
                $self->get_multistate_name($pi_dsp_multistates{$dsp_id}),
                "",
                "BS info:",
                QBit::Log::_get_pretty_dump($bs_dsp),
                ""
            );
        }

        delete($pi_dsp_states{$dsp_id});
    }

    WARN("## DSP ID $_\nBS don't have DSP ID $_\n")
      foreach (grep {!$pi_dsp_states{$_}->{'deleted'}} keys(%pi_dsp_states));
}

sub check_type {
    my ($self, $types) = @_;

    foreach (@$types) {
        return FALSE unless exists($TYPE{$_});
    }

    return TRUE;
}

sub check_format {
    my ($self, $formats) = @_;

    foreach (@$formats) {
        return FALSE unless exists($FORMAT{$_});
    }

    return TRUE;
}

sub get_all_dsp_for_testing {
    my ($self, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights("dsp_view_field__multistate");

    my $dsp_list = $self->get_all(
        fields => ['id', 'url', 'multistate', 'short_caption', 'test_url', 'tag'],
        ($opts{'limit'} ? (limit => $opts{'limit'}) : ())
    );

    undef($tmp_rights);

    my @result = ();

    foreach (@$dsp_list) {
        if ($self->check_multistate_action($_->{'multistate'}, 'test')) {
            push(@result, $_);
        } else {
            push(@result, {%$_, disabled => TRUE,}) if $opts{'with_disabled'};
        }
    }

    return \@result;
}

sub get_unmoderated_dsps {
    my ($self, $list) = @_;

    # деволтный фильтр
    my @filters = ({unmoderated_rtb_auction => TRUE});

    # фильтр на основе пришедшего списка (иначе все)
    push @filters, [id => IN => $list] if (ref $list eq 'ARRAY');

    # итоговый фильтр
    my $filter = (@filters == 1 ? $filters[0] : [AND => \@filters]);

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          dsp_view_all
          dsp_view_field__id
          dsp_view_field__short_caption
          dsp_view_field__unmoderated_rtb_auction
          )
    );
    my $dsp_list = $self->get_all(
        fields   => ['id'],
        filter   => $filter,
        order_by => [qw(short_caption)],
    );

    return [map {$_->{id}} @$dsp_list];
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'dsp_mode'}) {
        $result{'dsp_mode'} = \@DSP_MODE;
    }

    if ($need_fields->{'type'}) {
        $result{'type'} = {map {$_ => $TYPE{$_}->()} keys %TYPE};
    }

    if ($need_fields->{'format'}) {
        $result{'format'} = {map {$_ => $FORMAT{$_}->()} keys %FORMAT};
    }

    return \%result;
}
# TODO: вынести в общий модуль
sub get_fields_cnt {
    my ($self, %opts) = @_;

    return $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => {map {("cnt_$_") => {count => [{distinct => [$_]}]}} @{$opts{'fields'}}},
        filter => [
            id => ' = ANY' => $self->query(
                fields => $self->_get_fields_obj(['id']),
                filter => $self->get_db_filter($opts{'filter'})
            )
        ]
    )->get_all()->[0];
}

sub get_types {
    my ($self) = @_;

    my $types = clone(\%TYPE);

    return [sort {$a->{'id'} <=> $b->{'id'}} map {{id => $_, caption => $types->{$_}()}} keys(%$types)];
}

sub get_formats {
    my ($self) = @_;

    my $formats = clone(\%FORMAT);

    return [sort {$a->{'id'} <=> $b->{'id'}} map {{id => $_, caption => $formats->{$_}()}} keys(%$formats)];
}

sub need_update_in_bk {
    my ($self, $dsp, %opts) = @_;

    return FALSE unless $self->check_multistate_flag($dsp->{'multistate'}, 'created_in_bk');

    my %need_update = map {$_ => TRUE} qw(
      patch
      postmoderated
      short_caption
      show_probability
      skipnoud
      test_url
      types
      unmoderated_rtb_auction
      url
      );

    foreach (keys(%opts)) {
        return TRUE if $need_update{$_};
    }

    return FALSE;
}

sub turn_off_dsp {
    my ($self, $dsp_id) = @_;

    my @need_update =
      map {$_->{page_id}}
      @{$self->block_dsps->get_all(fields => ['page_id'], filter => {dsp_id => $dsp_id}, distinct => TRUE)};

    if (@need_update) {
        my $filter = $self->partner_db->filter({dsp_id => $dsp_id});
        $self->block_dsps->partner_db_table()->delete($filter);
        $self->block_dsps_unmoderated->partner_db_table()->delete($filter);

        $self->all_pages->mark_pages_for_async_update(page_ids => \@need_update, use_separate_queue => TRUE);
    }
}

sub _get_data_for_bk {
    my ($self, $dsp, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          dsp_view_field__display_name
          dsp_view_field__multistate
          dsp_view_field__patch
          dsp_view_field__short_caption
          dsp_view_field__skipnoud
          dsp_view_field__unmoderated_rtb_auction
          )
    );
    my $obj = $self->_get_object_fields(
        $dsp,
        [
            qw(
              display_name
              id
              multistate
              patch
              postmoderated
              short_caption
              show_probability
              skipnoud
              tag
              types
              formats
              unmoderated_rtb_auction
              url
              is_ssp_allowed
              works_on_all_platforms
              )
        ]
    );
    my $multistate = delete $obj->{multistate};
    $obj->{disabled}   = !$self->check_multistate_flag($multistate, 'working');
    $obj->{dsp_types}  = delete $obj->{types};
    $obj->{use_pnocsy} = $self->_get_use_pnocsy_value($obj->{tag});
    $obj = {%$obj, %opts};
    $obj->{$_} = ($obj->{$_} ? 1 : 0)
      foreach (
        qw(disabled postmoderated skipnoud unmoderated_rtb_auction use_pnocsy is_ssp_allowed works_on_all_platforms));

    return $obj;
}

sub _get_use_pnocsy_value {
    my ($self, $tag) = @_;

    throw Exception::IncorrectParams gettext("Tag should be defined") unless defined($tag);

    if (in_array($tag, [qw(yabs awaps local dc_dsp_test yamarket)])) {
        return 0;
    } else {
        return 1;
    }
}

sub _start_mobile_block {
    my ($self, $obj) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    my $tmp_rights = $self->app->add_tmp_rights(qw(mobile_app_settings_view_all));
    $self->$_->turn_on_dsp($obj) foreach (
        qw(
        mobile_app_rtb
        )
    );
}

sub _stop_mobile_block {
    my ($self, $obj, $block_type) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id multistate tag types formats)]);
    my $tmp_rights = $self->app->add_tmp_rights(qw(mobile_app_settings_view_all));
    $self->$_->stop_dsp_on_mobile_block($obj, $block_type) foreach (
        qw(
        mobile_app_rtb
        )
    );
}

sub _send_dsp_for_billing_to_lb {
    my ($self, $dsp_id, $owner, $update_dt) = @_;

    my $tm      = Time::HiRes::time;
    my $version = int($tm * 1_000);
    $update_dt //= curdate(oformat => "db_time");
    $self->api_selfservice->logbroker(
        topic => $self->get_option('dsp_for_billing_topic'),
        data  => [
            {
                Version => 0 + $version,
                ID      => 0 + $dsp_id,
                Object  => {
                    dsp_id       => 0 + $dsp_id,
                    client_id    => 0 + $owner->{client_id},
                    operator_uid => 0 + $owner->{id},
                    update_dt    => $update_dt,
                },
            }
        ],
    );
}

TRUE;
