package Application::Model::DSP::Documentation;

=encoding UTF-8

=cut

use qbit;

use base qw(Application::Model::Common);

use Exception::Denied;

sub accessor      {'dsp_documentation'}
sub db_table_name {'dsp_documentation'}

__PACKAGE__->model_accessors(
    partner_db => 'Application::Model::PartnerDB',
    users      => 'Application::Model::Users',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'dsp_documentation',
            description => d_gettext('Rights to use documentation'),
            rights      => {dsp_documentation_view => d_gettext('Right to view documentation'),},
        }
    ]
);

__PACKAGE__->model_fields(
    id          => {default => 1, db => TRUE, pk => TRUE, type => 'number'},
    name        => {default => 1, db => TRUE},
    description => {default => 1, db => TRUE},
    language    => {default => 1, db => TRUE},
    filename    => {default => 1, db => TRUE},
    content     => {default => 1, db => TRUE},
    multistate  => {default => 1, db => TRUE},
    size            => {default => 1, db => TRUE, db_expr => {length => ['content']}},
    multistate_name => {
        depends_on => ['multistate'],
        get        => sub {
            $_[0]->model->get_multistate_name($_[1]->{'multistate'});
        },
    },
    size_format => {
        depends_on => ['size'],
        get        => sub {
            my $bytes = $_[1]->{'size'};
            my $kb = sprintf('%.02f', $bytes / 1024);
            return $bytes . ' B' if $kb =~ /^0\.00/;
            my $mb = sprintf('%.02f', $bytes / (1024 * 1024));
            return $kb . ' KB' if $mb =~ /^0\.00/;
            return $mb . ' MB';
        },
    },
    actions => {
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
          }
    }
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id          => {type => 'number',     label => d_gettext('ID')},
        name        => {type => 'text',       label => d_gettext('Name')},
        description => {type => 'text',       label => d_gettext('Description')},
        language    => {type => 'text',       label => d_gettext('Language')},
        filename    => {type => 'text',       label => d_gettext('Name file')},
        multistate  => {type => 'multistate', label => d_gettext('Status')},
    },
);

__PACKAGE__->multistates_graph(
    empty_name    => 'New',
    multistates   => [[edited => d_gettext('Edited file')], [deleted => d_gettext('Deleted file')],],
    right_actions => {
        add    => d_gettext('Add file'),
        edit   => d_gettext('Edit file'),
        delete => d_gettext('Delete file'),
    },
    right_group        => [dsp_documentation_action => d_gettext('Right to manage dsp documentation')],
    right_name_prefix  => 'dsp_documentation_',
    multistate_actions => [
        {
            action => 'add',
            from   => '__EMPTY__',
        },
        {
            action    => 'edit',
            from      => 'not deleted',
            set_flags => ['edited'],
        },
        {
            action    => 'delete',
            from      => 'not deleted',
            set_flags => ['deleted'],
        },
    ],
);

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_rights($self->get_rights_by_actions('add'));

    my $id;
    $self->partner_db->transaction(
        sub {
            $id = $self->partner_db->dsp_documentation->add(\%opts);
            $self->do_action($id, 'add');
        }
    );
    return $id;
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;

    $self->partner_db->dsp_documentation->edit($obj, \%opts);
}

sub query_filter {
    my ($self, $filter) = @_;

    throw Exception::Denied unless $self->check_short_rights('view');

    return $filter;
}

TRUE;
