package Application::Model::DSP::StatisticsResponse;

use qbit;

use base qw(Application::Model::Statistics::Update);

use Exception::Denied;
use Exception::Validation::BadArguments;

sub accessor      {'statistics_dsp_response'}
sub db_table_name {'statistics_dsp_response'}

__PACKAGE__->model_accessors(
    partner_db  => 'Application::Model::PartnerDB',
    dsp         => 'Application::Model::DSP',
    api_http_bk => 'QBit::Application::Model::API::Yandex::HTTPBK',
    dsp_testing => 'Application::Model::DSP::Testing',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'can_be_statistics_reloaded',
            description => d_gettext('Rights for reloading statistics'),
            rights =>
              {statistics_dsp_response_can_be_reloaded => d_gettext('Right to reloading statistics by DSP response'),}
        }
    ]
);

sub can_be_reloaded {
    my ($self) = @_;

    return $self->check_short_rights('can_be_reloaded');
}

sub get_data_for_dsp_errors_widget {
    my ($self, $periods) = @_;

    throw Exception::Denied unless $self->check_rights('dsp_statistics_responses');

    my @result = ();

    my @dsps = map {$_->{'id'}} @{$self->dsp->get_all(fields => ['id'])};

    foreach (@$periods) {
        my $period = $_;
        $period = [$period->{'period__from'}, $period->{'period__to'}] if $period->{'period__name'} eq 'other';
        if (ref($period) ne 'ARRAY') {
            $period = [name2dates($period->{'period__name'}, '', '', iformat => 'db', oformat => 'db')];
        }

        unless (ref($period) eq 'ARRAY'
            && check_date($period->[0], iformat => 'db')
            && check_date($period->[1], iformat => 'db')
            && $period->[0] le $period->[1])
        {
            push(@result, {status => 'error', data => gettext('Invalid period')});
            next;
        }

        my $errors_count = $self->get_statistics(
            fields => {errors_count => {'SUM' => ['errors_sum']}},
            filter => ['AND', [['dsp_id', 'IN', \\@dsps], ['dt', '>=', \$period->[0]], ['dt', '<=', \$period->[1]]]]
        )->[0]{'errors_count'};

        push(@result, {status => 'ok', data => $errors_count // 0});
    }

    return {data => \@result};
}

sub get_statistics {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_rights('dsp_statistics_responses');

    my $filter = $self->partner_db->filter($opts{'filter'});

    my $select = $self->partner_db->query->select(
        table  => $self->_get_stat_table(),
        fields => $opts{'fields'},
        filter => $filter,
    );

    $select->order_by(@{$opts{'order_by'}}) if exists($opts{'order_by'});

    $select->limit($opts{'limit'}) if exists($opts{'limit'});

    return $select->get_all;
}

sub _fix_delete_filter {
    my ($self, $filter, %opts) = @_;

    my @dsp_ids = $self->_filter_opts2dsp_ids(%opts);

    $filter->and({dsp_id => \@dsp_ids}) if @dsp_ids;
}

sub _filter_opts2dsp_ids {
    my ($self, %opts) = @_;

    my $unknown_opts = arrays_difference([sort keys(%opts)], [qw(logins)]);

    throw Exception::Validation::BadArguments ngettext(
        'Unknown filter option "%s"',
        'Unknown filter options "%s"',
        scalar(@$unknown_opts),
        join(', ', @$unknown_opts)
    ) if @$unknown_opts;

    my @res = @{$opts{'dsp_ids'} // []};

    if ($opts{'logins'}) {
        my $dsps = $self->dsp->get_all(
            fields => ['id'],
            filter => [AND => [[owner => 'MATCH' => [login => '=' => $opts{'logins'}]]]]
        );

        # добавляем 0, чтобы не удалить всю статистику, если нет пейджей для логинов
        push(@res, @$dsps ? (map {$_->{'id'}} @$dsps) : 0);
    }

    return @res;
}

sub _filter_stat {
    my ($self, $data, %opts) = @_;

    my @ids = $self->_filter_opts2dsp_ids(%opts);

    return $data unless @ids;

    my %dsp_ids = map {$_ => TRUE} @ids;

    return [grep {$dsp_ids{$_->{'dsp_id'}}} @$data];
}

sub _get_last_update_day {
    return 3;
}

sub _get_stat_from_balance {
    my ($self, %opts) = @_;

    throw gettext("Expecting 'from'") unless defined $opts{'from'};
    throw gettext("Expecting 'to'")   unless defined $opts{'to'};

    my %known_dsp = map {$_->{'id'} => TRUE} @{$self->dsp->get_all(fields => ['id'])};

    my @statistics = map {
        {
            hash_transform(
                $_,
                [],
                {
                    DSPID      => 'dsp_id',
                    PageID     => 'campaign_id',
                    UpdateDate => 'dt',
                    ErrorType  => 'error_type',
                    ErrorsSum  => 'errors_sum',
                }
              )
        }
      } grep {
        $known_dsp{$_->{'DSPID'}}
      } @{$self->api_http_bk->get_dsp_statistics_responses(undef, $opts{'from'}, $opts{'to'})};

    return \@statistics;
}

TRUE;
