package Application::Model::DSP::Testing;

=encoding UTF-8

Модель для работы с инструментом тестирования DSP.

Процесс тестирования DSP выглядит так:

 * Пользователь указывает параметры с которыми он хочет отправить тестовый
   запрос и нажимает на кнопку
 * Мы эти парамеры преобразовываем в формат, который хочет видеть БК (это url
   и некоторое количество http headers)
 * Отправляем запрос в БК и тут же получаем ответ
 * Распарсиваем ответ, и часть ответа показываем пользователю

Вся эта логика спрятана в этом классе. На вход объект получает даныне от
пользователя, на выходе отбеъкт отдает данные которые можно показывать
пользователю.

=cut

use qbit;

use base qw(QBit::Application::Model::API::HTTP);

use JavaScript::V8;
use MIME::Base64 qw(encode_base64 decode_base64);

use Exception::Validation::BadArguments;

sub accessor {'dsp_testing'}

__PACKAGE__->model_accessors(dsp => 'Application::Model::DSP',);

sub call {
    my ($self, $method, $headers, %params) = @_;

    $self->{'__LWP__'}->default_headers($headers) if $headers;

    return $self->SUPER::call($method, %params);
}

=head2 get_dsp_testing_results

На входе - hashref с данными которые ввел пользователь.

На выходе - hashref с данными которые можно показывать пользователю.

=cut

sub get_dsp_testing_results {
    my ($self, %user_params) = @_;

    my $dsp_id = $user_params{'dsp_id'};
    return {error => {form => gettext("Expected dsp_id")}} unless $dsp_id;

    my $dsp = $self->dsp->get($dsp_id, fields => ['id']);
    return {error => {form => gettext("DSP with ID %s not found", $dsp_id)}}
      unless (ref($dsp) eq 'HASH' and $dsp->{'id'} == $dsp_id);

    # Прямо сейчас мы не можем проверять Яндекс.Директ, так как там есть
    # какая-то специфика. Если понадобится тестирования DSP Директ, то нужно
    # сходить в БК, выяснить особенности и реализовать эти особенности в коде
    return {error => {form => gettext("This can't test DSP 'Yandex.Direct'")}} if $dsp_id == 1;

    my $headers = $self->__get_headers_for_bs(\%user_params);

    my $response;
    try {
        $response =
          $self->call($user_params{'site_id'}, $headers, charset => 'UTF-8', 'imp-id' => $user_params{'imp_id'});
    }
    catch {
        $response = shift->{'response'};
    };

    if (ref($response) && $response->isa('HTTP::Response')) {
        return {error => {fields => {site_referer => gettext('Referrer does not match PageID')}}}
          if $response->code == 403 && $response->{'_content'} =~ m/Bad\s+partner\/domain/i;

        return {error => {form => gettext('User\'s region is banned in PageImp on PageID/ImpID')}}
          if $response->code == 404 && $response->{'_content'} =~ m/Invalid\s+region\s+for\s+PageID\/ImpID/i;

        return {error => {form => gettext("Error in the input data")}};
    } else {
        my $user_answer = $self->__get_answer_for_user(
            bs_response => $response,
            dsp_id      => $dsp_id,
        );

        return $user_answer;
    }
}

sub get_errors {
    my ($self) = @_;

    return (
        gettext('No HTTP response code 200 and HTTP 204'),
        gettext('Invalid json'),
        gettext('Invalidly rate'),
        gettext('Invalid value in field "ADM"'),
        gettext('Unsuitable characteristics of creative'),
        gettext('Invalid token test'),
        gettext('HTTP 204'),
        gettext('Verification token is blocked'),
        undef, undef,
        gettext('Crid absent, has the wrong format or is incorrect'),
        gettext('In dc_params to present not covered by the protocol'),
        gettext('One or more object keys dc_params invalid (eg an empty string)'),
        gettext(
'Invalid value or object type. The following causes: dc_params, creative_params, data_params are not objects; object price without number; not the right data type of objects click_url, target_url, image_url, text'
        ),
        gettext('No click_url or target_url, or one of them (both parameters are required)'),
        gettext('Error in data_params'),
        gettext('Error in ObjN data_params'),
        gettext('Error in subobjects ObjN data_params'),
        gettext('One or more ObjN rejected moderation'),
        gettext('Crid rejected moderation'),
        gettext('Term of use Crid expired'),
        gettext('Domain is blocked for the site or is not permitted for this crid'),
        gettext('Show for the site or region is not allowed'),
        gettext('ObjN not moderation. The DSP is not allowed post moderation'),
        gettext('One or more ObjN banned'),
        gettext('ObjN data_params is not allowed for the DSP'),
        gettext('Invalid currency'),
        gettext('Invalid ad type'),
        gettext('Invalid image url'),
        gettext('Incorrect flag'),
        gettext('Invalid categories'),
        gettext('URL not proccessed'),
    );
}

=begin comment __get_answer_for_user

На входе - объект HTTP::Response с ответом БК, на выходе hashref с данными,
которые можно отдавать пользователю.

=cut

sub __get_answer_for_user {
    my ($self, %opts) = @_;

    my $dsp_id = delete $opts{'dsp_id'};
    return {error => {form => gettext("Expected dsp_id")}} unless $dsp_id;

    my $bs_response = delete $opts{'bs_response'};
    throw Exception::Validation::BadArguments gettext("Expected bs_response") unless $bs_response;

    my $bs_data = $self->__get_fixed_bs_data($bs_response);

    return {error => {form => gettext("Incorrect parameters")}} if $bs_data->{'error'};

    my ($request, $response, $event_flags, $partner_price, $min_price);

    if ($bs_data->{'rtb'}{'debug'}{'dsp'} && ref($bs_data->{'rtb'}{'debug'}{'dsp'}) eq 'ARRAY') {
        foreach (@{$bs_data->{'rtb'}{'debug'}{'dsp'}}) {
            if ($_->{'dsp_id'} == $dsp_id) {
                $_->{'request'} =~ tr/-_/+\//;
                $request = decode_base64($_->{'request'});
                utf8::decode($request);
                $_->{'response'} =~ tr/-_/+\//;
                $response = decode_base64($_->{'response'});
                utf8::decode($response);
                $event_flags   = $_->{'event_flags'};
                $partner_price = $_->{'partner_price'};
                $min_price     = $_->{'min_price'};
                last;
            }
        }
    }

    return {error => {form => gettext("Please, contact support to connect your DSP to sandbox")}}
      unless defined($event_flags);

    my @error;

    push(@error, gettext('DSP rate less then CPM limit campaign')) if $event_flags == 0 && $partner_price < $min_price;

    my $multi_errors = {2**1 | 2**4 => gettext('Invalid format of field "properties"')};

    foreach my $error_code (keys(%$multi_errors)) {
        if (($event_flags & $error_code) == $error_code) {
            push(@error, $multi_errors->{$error_code});
            $event_flags &= ~int($error_code);
        }
    }

    my @error_list = $self->get_errors();

    for my $i (0 .. @error_list) {
        next unless $error_list[$i];
        push(@error, "($i) " . $error_list[$i]) if $event_flags & (2**$i);
    }

    my $answer_for_user = {
        error                  => \@error,
        rtb_host_response_json => to_json($bs_data, pretty => TRUE),
        request  => $request  // '',
        response => $response // '',
        can_view_rtb_host_response => $self->app->check_rights('dsp_view_rtb_host_response'),
    };

    return $answer_for_user;
}

=begin comment __get_fixed_bs_data

Те данные, которые возвращает БК в общем случае не являются валидным JSON-ом,
но эти данные гарантировано являются валидным JavaScript.

=cut

sub __get_fixed_bs_data {
    my ($self, $data) = @_;

    # Данные которые возвращает БК выглядят так:
    #
    #   ('{ ... }')
    #
    # Чтобы можно было распрасить эту штуку отрезаем начальные и конечные
    # символы
    $data =~ s/^\('//;
    $data =~ s/'\)$//;

    my $context = JavaScript::V8::Context->new();

    my $result = $context->eval(
        qq|
        var j=$data;
        JSON.stringify(j);
    |
    );

    if (!$@) {
        my $href = from_json($result);
        return $href;
    } else {
        return {error => TRUE};
    }
}

=begin comment __get_headers_for_bs

Возвращает объект HTTP::Headers в котором содержится все данные в том виде как
их хочет видеть БК.

    my $headers = $self->__get_headers_for_bs( $user_params );

Пример обращения с БК с помощью curl:

    curl --config pi_rtb_request.conf

При этом в файле pi_rtb_request.conf должно быть:

    url = "http://bs-rtb-sandbox.bshive.yandex.net/meta/92550?callback=Ya%5B1358589606637%5D&rnd=752342&charset=UTF-8&skip-token=yabs.&imp-id=1"
    header = "Host: bs-meta.yandex.ru"
    header = "User-Agent: Mozilla/5.0 (Windows NT 5.1) AppleWebKit/537.17 (KHTML, like Gecko) Chrome/24.0.1312.52 Safari/537.17"
    header = "Referer: http://mail.yandex.ru/neo2/"
    header = "Cookie: yandexuid=2566469731340134007; yabs-dsp=sociomantic.MTU4ODUzNDU0ODEzMDMyNTIwNjQ=; yabs-exp-sid=126cbf576b8b4567;"
    header = "X-Real-IP: 95.73.252.226"

=cut

sub __get_headers_for_bs {
    my ($self, $user_params) = @_;

    my $cookie = 'yandexuid=' . $user_params->{'device_yandexuid'};

    if ($user_params->{'device_userdata'}) {
        utf8::encode($user_params->{'device_userdata'});
        my $userdata = encode_base64($user_params->{'device_userdata'});
        $userdata =~ tr/\+\//-_/;
        $cookie .= "; yabs-dsp=" . $user_params->{'tag'} . ".$userdata";
    }

    $cookie .= "; yabs-exp-sid=126cbf576b8b4567;";

    my $headers = HTTP::Headers->new();
    $headers->header(
        'Host'          => 'bs-meta.yandex.ru',
        "User-Agent"    => $user_params->{'device_ua'},
        "Referer"       => $user_params->{'site_referer'},
        "Cookie"        => $cookie,
        "X-Real-IP"     => $user_params->{'device_ip'},
        "Dis-Dsp-Token" => $user_params->{'moderation'},
        "Only-Dsp-Id"   => $user_params->{'dsp_id'}
    );

    return $headers;
}

TRUE;
