package Application::Model::DesignTemplates;

use qbit;
use Utils::HookStash;

use Template;

use base qw(
  Application::Model::Common
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::UpdateTime
  );

use PiConstants (qw($DESIGN_TYPES $DEFAULT_DESIGN_TYPE));

use Exception::DB::DuplicateEntry;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validation::BadArguments::EmptyValue;

sub accessor      {'design_templates'}
sub db_table_name {'design_templates'}

sub get_product_name {gettext('design_templates')}

sub get_opts_schema_name {'design_templates_opts'}

sub get_structure_model_accessors {
    my ($self) = @_;

    return {
        partner_db   => 'Application::Model::PartnerDB',
        api_http_mol => 'Application::Model::API::Yandex::HTTPMOL',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        map {$self->get_description_right($_)}
          qw(
          view
          edit
          view_action_log
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        id => {default => TRUE, db => TRUE, type => 'number', pk => TRUE, api => 1,},
        page_id =>
          {default => TRUE, db => TRUE, type => 'number', need_check => {type => 'int_un', min => 1}, api => 1,},
        block_id => {
            default    => TRUE,
            db         => TRUE,
            type       => 'number',
            need_check => {type => 'int_un', min => 0, optional => TRUE},
            api        => 1,
        },
        public_id => {
            db      => TRUE,
            db_expr => 'id',
            type    => 'string',
            api     => 1,
        },
        caption => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            need_check => {
                len_min => 1,
                len_max => 255,
            },
            api => 1,
        },
        type => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            need_check => {
                in  => [sort values %$DESIGN_TYPES],
                msg => gettext("Incorrect value for field \"%s\"", "type")
            },
            api => 1,
        },
        design_settings => {default => TRUE, from_opts => 'from_hash', type => 'complex', api => 1,},
        filter_tags     => {
            default    => TRUE,
            from_opts  => 'from_hash',
            type       => 'array',
            sub_type   => 'string',
            need_check => {
                optional => TRUE,
                type     => 'array',
                all      => {len_min => 1}
            },
            api => 1,
        },
        is_custom_format_direct => {
            default    => TRUE,
            from_opts  => 'from_hash',
            type       => 'boolean',
            need_check => {
                optional => TRUE,
                type     => 'boolean',
            },
            api => 1,
        },
        multistate      => {db => TRUE, type => 'number', api => 1,},
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1,
        },
        #API
        fields_depends => {
            depends_on => ['id'],
            get        => sub {{}},
            type       => 'complex',
            api        => 1,
        },
        status => {
            depends_on => ['id'],
            get        => sub {'sync'},
            type       => 'string',
            api        => 1,
        },
        shows_percent => {
            depends_on => ['id', 'page_id', 'block_id'],
            get        => sub {
                my $stats = $_[0]{'__STATS__'};
                my $s     = $stats->{by_design}{$_[1]{id}};
                if (my $show = $s->{Shows} and my $sum = $stats->{by_block}{$s->{key}}) {
                    return 100 * $show / $sum;
                } else {
                    return 0;
                }
            },
            type => 'number',
            api  => 1,
        },
        delta_cpm => {
            depends_on => ['id', 'page_id', 'block_id'],
            get        => sub {
                my $stats = $_[0]{'__STATS__'};
                my $s     = $stats->{by_design}{$_[1]{id}};
                if (my $delta = $s->{DeltaRealPrice} and my $price = $s->{RealPrice}) {
                    return 100 * $delta / $price;
                } else {
                    return 0;
                }
            },
            type => 'number',
            api  => 1,
        },
        stat_last_date => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_check => {
                optional => TRUE,
                type     => 'date',
            },
        },
    };
}

sub get_structure_model_filter {

    return {
        db_accessor => 'partner_db',
        fields      => {
            id             => {type => 'number',     label => d_gettext('ID')},
            page_id        => {type => 'number',     label => d_gettext('Page ID')},
            block_id       => {type => 'number',     label => d_gettext('Block ID')},
            caption        => {type => 'text',       label => d_gettext('Caption')},
            type           => {type => 'text',       label => d_gettext('Type')},
            multistate     => {type => 'multistate', label => d_gettext('Multistate')},
            stat_last_date => {type => 'json'},
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => 'id',         label => gettext('Template ID')},
        {name => 'multistate', label => gettext('Status')},
        {name => 'caption',    label => gettext('Caption')},
        {name => 'type',       label => gettext('Type')},
        {name => 'page_id',    label => gettext('Page ID')},
        {name => 'block_id',   label => gettext('Block ID')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {

        empty_name    => 'new',
        multistates   => [[deleted => d_gettext('Deleted')]],
        right_actions => {
            add     => d_gettext('Add'),
            edit    => d_gettext('Edit'),
            delete  => d_gettext('Delete'),
            restore => d_gettext('Restore'),
        },
        right_group        => [design_templates => d_gettext('Rights for design templates')],
        right_name_prefix  => 'design_templates_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('shows_percent') or $fields->need('delta_cpm')) {
        my %ids;
        for my $row (@$result) {
            $ids{$row->{page_id}}{$row->{block_id}} = 1;
        }

        my @filter = map +{
            PageID => {eq => [$_]},
            ImpID  => {eq => [sort keys %{$ids{$_}}]},
          },
          keys %ids;

        my $data = $self->api_http_mol->get_data(
            countable_fields => ["Shows", "DeltaRealPrice", "RealPrice"],
            currency         => "RUB",
            date_from   => date_sub(curdate(), day => 14, oformat => 'date_only_numbers'),
            date_to     => date_sub(curdate(), day => 1,  oformat => 'date_only_numbers'),
            filters_pre => {
                'DesignID' => {ne => [0]},
                '-and'     => \@filter
            },
            group_by => ["PageID", "ImpID", "DesignID"],
        );
        my %by_design;
        my %by_block;
        my $header = $data->{header};
        my %row;
        for my $row (@{$data->{data}}) {
            @row{@$header} = @$row;
            my $block_key = "$row{PageID}-$row{ImpID}";
            $by_block{$block_key} += $row{Shows};
            $by_design{$row{DesignID}} = {
                Shows          => $row{Shows},
                key            => $block_key,
                RealPrice      => $row{RealPrice},
                DeltaRealPrice => $row{DeltaRealPrice}
            };
        }
        $fields->{'__STATS__'} = {
            by_design => \%by_design,
            by_block  => \%by_block
        };
    }
}

sub get_actions_depends {
    [
        qw(
          id
          page_id
          block_id
          type
          multistate
          )
    ];
}

sub get_available_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} keys(%{$self->get_model_fields});

    return \%fields;
}

sub get_available_fields_depends {
    return [qw(id)];
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{$_} = TRUE foreach (qw(page_id block_id));

    return $fields;
}

sub get_editable_fields {
    my ($self, $format) = @_;

    return {} if $self->check_multistate_flag($format->{'multistate'}, 'deleted');

    my $fields = $self->_get_common_add_edit_fields();
    $fields->{stat_last_date} = TRUE;
    return $fields;
}

sub get_editable_fields_depends {
    return [qw(multistate)];
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              block_id
              caption
              design_settings
              filter_tags
              is_custom_format_direct
              page_id
              type
              )
        ]
    );

    return $fields;
}

sub hook_owner_processing { }

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;
    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{'block_id'} //= 0;
    $opts->{'type'} //= $DEFAULT_DESIGN_TYPE;
}

sub hook_fields_processing_before_validation {
    my ($self) = @_;

    my $opts = $self->hook_stash->get('opts');

    $self->SUPER::hook_fields_processing_before_validation($opts);

    $opts->{'type'} //= $DEFAULT_DESIGN_TYPE;
}

sub get_direct_limit {
    my ($self, $opts) = @_;
    my $limit;
    if ($opts->{type} eq $DESIGN_TYPES->{NATIVE}) {
        $limit = $opts->{design_settings}->{grid_columns} * $opts->{design_settings}->{grid_rows};
    } else {
        $limit = $opts->{design_settings}->{limit} if exists $opts->{design_settings}->{limit};
    }
    return $limit;
}

# API

sub api_available_actions {
    my ($self) = @_;

    return ();
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_current_user_pages($filter);

    return $filter;
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->register_rights($self->get_structure_rights_to_register());
    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
    $self->multistates_graph($self->get_structure_multistates_graph());
}

TRUE;
