package Application::Model::Documents;

use qbit;

use base qw(QBit::Application::Model);

use Application::Model::AgreementChecker::_Agreement;
use Utils::Logger qw(WARN);

use PiConstants qw(
  %CONTRACT_TYPE
  $FIRM_ID_YANDEX_EUROPE_AG
  $FIRM_ID_YANDEX_LTD
  $GAME_OFFER_SERVICE
  );

sub accessor {'documents'}

__PACKAGE__->model_accessors(
    api_balance => 'QBit::Application::Model::API::Yandex::Balance',
    memcached   => 'QBit::Application::Model::Memcached'
);

my %TYPES = (PARTNERS => {title => d_gettext('Advertising network')},);

my %COLLATERAL_TYPES = (
    # изменение срока действия
    2000 => {title => d_gettext('Change of period of validity'), change_fields => [qw(end_dt)]},
    # переход на УСН (вознаграждение не облагается НДС)
    2010 => {title => d_gettext('Change to simplified taxation system'), change_fields => [qw(nds)]},
    # изменение процента партнера
    2020 => {title => d_gettext('Change of contract percent'), change_fields => [qw(partner_pct agregator_pct)]},
    # подключение площадок МКБ
    2030 => {title => d_gettext('MKB suplement'), change_fields => [qw(mkb_price)]},
    # прочее
    2040 => {title => d_gettext('Others')},
    # расторжение договора
    2050 => {title => d_gettext('Termination of contract'), change_fields => [qw(end_dt)]},
    # о доменах
    2060 => {title => d_gettext('Domains suplement')},
    # о баннеромаркете
    2070 => {title => d_gettext('Banneromarket suplement')},
    # о поисковых формах
    2080 => {title => d_gettext('Search forms suplement'), change_fields => [qw(search_forms)]},
    # уведомление о закрытии
    2090 => {title => d_gettext('Notification of termination of contract'), change_fields => [qw(end_dt end_reason)]},
    # уведомление о смене НДС)
    2100 => {title => d_gettext('Notification of change taxation system'), change_fields => [qw(nds)]},
    # изменение способа выплат
    2110 => {title => d_gettext('Change of payment type'), change_fields => [qw(pay_to)]},
    # самозанятый
    2160 => {title => d_gettext('Set partner as selfemployed'), change_fields => [qw(selfemployed)]},
    # изменение налогообложения
    3010 => {title => d_gettext('Change of taxacation system'), change_fields => [qw(nds)]},
    # изменение процента партнера
    3020 => {title => d_gettext('Change of contract percent'), change_fields => [qw(partner_pct)]},
    # изменение номенклатуры
    3030 => {
        title         => d_gettext('Change of nomenclature'),
        change_fields => [qw(distribution_places product_search product_searchf product_options)]
    },
    # прочее
    3040 => {title => d_gettext('Other')},
    # расторжение договора
    3060 => {title => d_gettext('Termination of contract'), change_fields => [qw(end_dt tail_time)]},
    # изменение/добавление приложения загрузок
    3070 => {
        title         => d_gettext('Download applications suplement'),
        change_fields => [qw(products_download download_domains)]
    },
    # изменение/добавление приложения установок
    3080 =>
      {title => d_gettext('Installation applications suplement'), change_fields => [qw(install_price install_soft)]},
    # переход на NET-базу для вознаграждения
    3090 => {title => d_gettext('Change to NET reward'), change_fields => [qw(partner_pct reward_type)]},
);

my %AN_DOC_SETS = (
    1 => {title => d_gettext('First')},
    2 => {title => d_gettext('Old')},
    3 => {title => d_gettext('Without domains')},
    4 => {title => d_gettext('Without domains (extended)')},
);

my %CURRENCY_BYN = (
    933 => 'BYN',
    BYN => 'BYN',
);

my %CURRENCY_EUR = (
    978 => 'EUR',
    EUR => 'EUR',
);

my %CURRENCY_RUB = (
    643 => 'RUB',
    810 => 'RUB',
    RUB => 'RUB',
);

my %CURRENCY_USD = (
    840 => 'USD',
    USD => 'USD',
);

my %CURRENCY = (%CURRENCY_BYN, %CURRENCY_EUR, %CURRENCY_RUB, %CURRENCY_USD,);

sub get_active_contract {
    my ($self, $opts) = @_;

    my @live_contracts = @{$self->get_live_contracts(client_id => $opts->{client_id})};

    my $contract;
    if (@live_contracts == 0) {
        return;
    } elsif (@live_contracts > 1) {
        WARN scalar(@live_contracts) . " live contracts for client_id = $opts->{client_id}";
    }
    $contract = $live_contracts[0];
    my $agreement = Application::Model::AgreementChecker::_Agreement->new(
        contract    => $contract->{Contract},
        collaterals => $contract->{Collaterals},
    );

    $contract->{'Contract'}{'status'} = $self->_get_contract_status($contract->{'Contract'});
    # PI-9937
    delete $contract->{'Person'}{'yamoney_wallet'} unless (($contract->{'Contract'}{'pay_to'} // '') eq '2');

    # Возможные значение CONTRACT_TYPE в балансе
    # "CODE","VALUE"
    # "2","С Агрегатором"
    # "3","С Физ. лицом"
    # "1","С Юр. лицом"
    # "4","С нерезидентом"
    # "5","универсальный"
    # "6","РСЯ 2014"
    # "7","Новая оферта РСЯ"
    # "8","SSP"
    # "9", Оферта РСЯ для России (создана в апреле 2018)
    # "10", Лицензионный договор (создан в ноябре 2018 — BALANCE-25434)

    # Флаг is_oferta1 означает что пользователь работает под офертой для других стран (Франция, Германия, etc)
    # Текст этой оферты находится по адресу https://yandex.ru/legal/oferta_yan_ch/
    $contract->{'Contract'}{'is_oferta1'} = 1 if $contract->{'Contract'}{'contract_type'} == $CONTRACT_TYPE{NEW_OFFER};

    # Флаг is_oferta2 означает что пользователь работает под офертой для России
    # Текст этой оферты находится по адресу https://yandex.ru/legal/yan_service_offer/
    if ($contract->{'Contract'}{'contract_type'} == $CONTRACT_TYPE{OFFER}) {
        if ($contract->{'Contract'}{'firm'} == $FIRM_ID_YANDEX_EUROPE_AG) {
            $contract->{'Contract'}{'is_oferta1'} = 1;
        } else {
            $contract->{'Contract'}{'is_oferta2'} = 1;
        }
    }

    # Флаг allows_to_fill_part2 означает что пользователь работает под офертой для России,
    # и заполнил только первую часть анкеты, т.е. у него нет плательщика
    $contract->{'Contract'}{'allows_to_fill_part2'} = $agreement->allows_to_fill_part2
      if $agreement->allows_to_fill_part2;

    my $filter = {
        Person => {
            longname             => 1,
            name                 => 1,
            phone                => 1,
            email                => 1,
            fax                  => 1,
            legaladdress         => 1,
            representative       => 1,
            account              => 1,
            authority_doc_type   => 1,
            ben_account          => 1,
            ben_bank             => 1,
            bank                 => 1,
            bik                  => 1,
            iban                 => 1,
            inn                  => 1,
            kpp                  => 1,
            other                => 1,
            payoneer_wallet      => 1,
            person_account       => 1,
            postaddress          => 1,
            signer_person_name   => 1,
            signer_position_name => 1,
            swift                => 1,
            yamoney_wallet       => 1,
            ($self->check_rights('users_view_field__client_id') ? (client_id => 1) : ()),
            ($self->check_rights('users_view_field__id')        ? (id        => 1) : ()),
            type    => 1,
            country => 1,
        },
        Contract => {
            person_id   => 1,
            nds         => 1,
            firm        => 1,
            test_mode   => 1,
            external_id => 1,
            dt          => 1,
            is_signed   => 1,
            is_faxed    => 1,
            status      => 1,
            currency    => 1,
            vat         => 1,
            (
                $self->check_rights('view_active_contract_reward_type')
                ? (reward_type => 1)
                : ()
            ),
            is_oferta1           => 1,
            is_oferta2           => 1,
            allows_to_fill_part2 => 1,
            selfemployed         => 1,
            contract2_id         => 1,
            contract_type        => 1,
            pay_to               => 1,
        },
    };

    $contract = _filter_hash($contract, $filter) unless $opts->{'ignore_filter'};

    $contract->{'Person'}{'login'} = $opts->{'login'} if $self->check_rights('partner_acts_view_filter__login');

    return $contract;
}

sub get_live_contracts {
    my ($self, %opts) = @_;

    my @contracts =
      grep {
        Application::Model::AgreementChecker::_Agreement->new(
            contract    => $_->{Contract},
            collaterals => $_->{Collaterals},
          )->_is_live('today')
      } @{$self->get_contracts(client_id => $opts{client_id})};

    return \@contracts;
}

sub get_contracts {
    my ($self, %opts) = @_;

    my @contracts =
      grep {exists($TYPES{$_->{'Contract'}{'type'}})} @{$opts{contract_list} // $self->get_raw_contracts(%opts)};

    my $today = curdate(oformat => 'db');
    foreach my $contract (@contracts) {
        foreach my $collateral (sort {$a->{'dt'} cmp $b->{'dt'}} @{$contract->{'Collaterals'} || []}) {
            my $is_live_collateral = !(
                (
                      !$opts{skip_collateral_start_dt_check}
                    && $today lt $collateral->{'dt'}    # Время действия ещё не наступило
                )
                || $collateral->{'is_cancelled'}        # Аннулировано
                || (
                    $collateral->{'class'} eq 'COLLATERAL'    # Допник должен быть подписан
                    && !(
                        $collateral->{'is_signed'}            # оринигал
                        || $collateral->{'is_faxed'}          # или по факсу
                    )
                    && $collateral->{'collateral_type_id'} != 2050    # и это не допник о закрытии
                    && $collateral->{'collateral_type_id'} != 2090    # и это не допник о закрытии
                   )
            );

            if ($is_live_collateral) {
                # Перенос полей в договор
                $contract->{'Contract'}{$_} = $collateral->{$_}
                  foreach @{$COLLATERAL_TYPES{$collateral->{'collateral_type_id'}}->{'change_fields'} || []};
            }
        }
    }

    if ($opts{need_signed_offer_ru}) {
        return [
            grep {
                     $_->{'Contract'}{'is_signed'}
                  && $_->{'Contract'}{'contract_type'} == $CONTRACT_TYPE{OFFER}
                  && $_->{'Contract'}{'firm'} != $FIRM_ID_YANDEX_EUROPE_AG
                  && Application::Model::AgreementChecker::_Agreement->new(
                    contract    => $_->{Contract},
                    collaterals => $_->{Collaterals},
                  )->is_live_today()
              } @contracts
        ]->[0];
    } else {
        return [sort {$a->{'Contract'}{'dt'} cmp $b->{'Contract'}{'dt'}} @contracts];
    }
}

sub get_contracts_by_filter {
    my ($self, %opts) = @_;

    my $rc = $self->get_raw_contracts(%opts);
    my @contracts;
    for my $c (@$rc) {
        push @contracts, $c if _filter_cmp_hash($c, $opts{filter});
    }
    return @contracts;
}

sub get_raw_contracts {
    my ($self, %opts) = @_;

    my $contracts;
    try {
        $contracts = $self->api_balance->get_partner_contracts(ClientID => $opts{'client_id'});
    }
    catch {
        $contracts = $self->memcached->get(documents => $opts{'client_id'}) || throw $_[0];
    };

    $self->memcached->set(documents => $opts{'client_id'} => $contracts, 7 * 60 * 60 * 24);

    return $contracts;
}

sub _get_contract_status {
    my ($self, $contract) = @_;

    my $status;

    if ($contract->{'contract_type'} == $CONTRACT_TYPE{OFFER}) {
        if ($contract->{test_mode}) {
            $status = gettext('Offer test mode');
        } elsif ($contract->{is_signed}) {
            $status = gettext('Offer accepted');
        } else {
            WARN('Unknown contract status');
        }
    } else {
        if (!$contract->{'is_faxed'} && !$contract->{'is_signed'}) {
            $status = gettext('Sent by mail');
        } elsif (!$contract->{'is_signed'}) {
            $status = gettext('Got a copy, send the original');
        } else {
            $status = gettext('Signed');

            $status .= ' ' . format_date($contract->{'is_signed'}, '%d.%m.%Y', iformat => 'db')
              if defined($contract->{'is_signed'})
                  && check_date($contract->{'is_signed'}, iformat => 'db')
                  && $contract->{'contract_type'} != $CONTRACT_TYPE{NEW_OFFER};
        }
    }

    return $status;
}

sub _filter_hash {
    my ($hash, $filter) = @_;

    my $filtered = {};

    for my $key (keys %$filter) {
        if (ref($filter->{$key}) eq 'HASH') {
            throw Exception "key $key element is supposed to be hashref"
              unless ref($hash->{$key}) eq 'HASH';
            $filtered->{$key} = _filter_hash($hash->{$key}, $filter->{$key});
        } else {
            if (exists($hash->{$key})) {
                $filtered->{$key} = $hash->{$key};
            }
        }
    }

    return $filtered;
}

sub _filter_cmp_hash {
    my ($hash, $filter) = @_;

    my $result = TRUE;

    for my $key (keys %$filter) {
        if (ref($filter->{$key}) eq 'HASH') {
            if (ref($hash->{$key}) eq 'HASH') {
                $result &&= _filter_cmp_hash($hash->{$key}, $filter->{$key});
            } else {
                $result = FALSE;
            }
        } else {
            if (defined $hash->{$key}) {
                $result &&= $filter->{$key} eq $hash->{$key};
            } else {
                $result = FALSE;
            }
        }
    }

    return $result;
}

sub is_resident {
    my ($self, $contract) = @_;

    my $contract_type = $contract->{Contract}{contract_type} // '';
    my $currency      = $contract->{Contract}{currency}      // '';
    my $firm_id       = $contract->{Contract}{firm}          // '';
    my $test_mode     = $contract->{Contract}{test_mode}     // '';
    my $person_type   = $contract->{Person}{type}            // '';

    my $result = (
        (
            in_array($contract_type, [@CONTRACT_TYPE{qw(UR PH YAN_2014)}])
              || $contract_type eq $CONTRACT_TYPE{OFFER} && $firm_id eq $FIRM_ID_YANDEX_LTD
        )
          && ($person_type eq 'ph' || $person_type eq 'ur' || $test_mode)
          && $self->is_currency_rub($currency)
    ) ? TRUE : FALSE;

    return $result;
}

sub is_nonresident {
    my ($self, $contract) = @_;

    my $contract_type = $contract->{Contract}{contract_type} // '';
    my $firm_id       = $contract->{Contract}{firm}          // '';
    my $person_type   = $contract->{Person}{type}            // '';

    my $result = (
        (
            in_array($contract_type, [@CONTRACT_TYPE{qw(NONRESIDENT UNIVERSAL NEW_OFFER)}])
              || $contract_type eq $CONTRACT_TYPE{OFFER} && $firm_id eq $FIRM_ID_YANDEX_EUROPE_AG
        )
          && $person_type
          && $person_type ne 'ph'
          && $person_type ne 'ur'
    ) ? TRUE : FALSE;

    return $result;
}

sub is_currency_eur {
    my ($self, $currency) = @_;

    return (exists $CURRENCY_EUR{$currency // ''} ? TRUE : FALSE);
}

sub is_currency_rub {
    my ($self, $currency) = @_;

    return (exists $CURRENCY_RUB{$currency // ''} ? TRUE : FALSE);
}

sub is_currency_usd {
    my ($self, $currency) = @_;

    return (exists $CURRENCY_USD{$currency // ''} ? TRUE : FALSE);
}

sub get_currency_for_offer {
    my ($self, $currency, $default_currency) = @_;
    $default_currency //= 'RUB';
    return ($CURRENCY{$currency // ''} // $default_currency);
}

sub get_games_contracts {
    my ($self, $client_id, %opts) = @_;

    my @game_offers = $self->get_contracts_by_filter(
        client_id => $client_id,
        filter    => {
            Contract => {
                type     => 'SPENDABLE',
                services => {$GAME_OFFER_SERVICE => 1},
            }
        }
    );

    if ($opts{live_only}) {
        @game_offers = grep {
            Application::Model::AgreementChecker::_Agreement->new(
                contract    => $_->{Contract},
                collaterals => $_->{Collaterals},
              )->_is_live('today')
        } @game_offers;
    }

    return @game_offers;
}

1;
