package Application::Model::ExceptionDumper;

use qbit;

use base qw(QBit::Application::Model);

use Utils::Logger qw(ERROR);

sub accessor {'exception_dumper'}

my $_error_dump_link = '';

sub application_dump {
    my ($self) = @_;

    return sprintf('
        <div style="background-color: #EEEEEE; padding: 5px 10px; margin: 1px;">
            <h3 id="application_dump_part">Application dump:</h3>
            <pre id="application_dump_text">%s</pre>
        </div>',
        safe_html_encode(short_dumper($self)));
}

sub backtrace_info {
    my ($self, $exception) = @_;

    return sprintf('
        <div style="background-color: #FFFFCC; padding: 5px 10px; margin: 1px;">
            <h3 id="backtrace_part">Backtrace:</h3>
            <ol>
            %s
            </ol>
        </div>
    ',
        join(
            '',
            map {
                sprintf('
                  <li style="font-family: monospace; margin-bottom: 0.5em;">
                    <strong>%s</strong>
                    (<pre style="margin: 0px 0px 0px 2em; padding: 0px;">
%s
called at %s line %s
                    </pre>)
                  </li>
                  ',
                    safe_html_encode($_->{'subroutine'}),
                    join("\n", map {safe_html_encode(short_dumper($_, 1)) . ","} @{$_->{'args'}}),
                    safe_html_encode($_->{'filename'}),
                    $_->{'line'})
              } @{$exception->{'callstack'}}
            ));
}

sub basic_request_info {
    my ($self, $request) = @_;

    my $html = '';
    if ($request) {
        $html = sprintf('
            <div style="background-color: #EEAA77; padding: 5px 10px; margin: 1px;">
                <h3 id="request_part">Request:</h3>
                <pre id="basic_request_text"> %s </pre>
            </div>',
            safe_html_encode(short_dumper($request)));
    }

    return $html;
}

sub build_file {
    my ($self, $dir, $exception) = @_;
    $dir = $self->get_option('ApplicationPath') . '/' . $dir unless $dir =~ /^\//;

    my $filename = "dump_" . format_date(curdate(), '%Y%m%d_%H%M%S') . "${$}.html";
    $self->set_curr_dump_link($filename);

    require File::Path;
    File::Path::make_path($dir);
    writefile($dir . '/' . $filename, $self->exception2html($exception));

    eval {system("cp $dir/$filename $dir/last_error.html")};
}

sub dump_as_html_file {
    my ($self, $exception, %opts) = @_;

    {
        local $Log::Log4perl::caller_depth = $Log::Log4perl::caller_depth + 1;
        ERROR $exception if (!$ENV{TAP_VERSION} || $ENV{FORCE_LOGGER_TO_SCREEN});
    }
    my $stage = $self->get_option('stage', '');
    if ($stage eq 'dev' || $exception->{sentry}{error}) {
        my $dir = $self->get_option('error_dump_dir');

        if (defined($dir) && $dir ne '') {
            $self->build_file($dir, $exception);
        }
    }
}

sub env_info {
    my ($self) = @_;

    my $env = $self->app->sentry->_filtered_env();

    return sprintf('
        <div style="background-color: #CCCCFF; padding: 5px 10px; margin: 1px;">
            <h3 id="server_environment_part">Server enviroment:</h3>
            <table border="1" cellspacing="0" cellpadding="3" bordercolor="#CCCCCC">
            %s
            </table>
        </div>
    ',
        join(
            '',
            map {
                sprintf('
                  <tr>
                    <th align="right">%s</th>
                    <td>%s</td>
                  </tr>',
                    safe_html_encode($_),
                    safe_html_encode($env->{$_} || ''))
              } keys(%$env)
            ));
}

sub exception2html {
    my ($self, $exception) = @_;

    my $request_info = '';
    if ($self->app->can('request')) {
        my $request = $self->app->request;
        if (blessed($request) && $request->isa('QBit::WebInterface::Request')) {
            $request_info = $self->http_request_info($request);
        } else {
            $request_info = $self->basic_request_info($request);
        }
    }

    my ($mysql_engine_innodb_dump, $get_mysql_processlist, $is_db_error) = ('', '', 0);
    if (ref($exception) =~ /Exception::DB/) {
        $is_db_error              = 1;
        $mysql_engine_innodb_dump = $self->get_mysql_innodb_status();
        $get_mysql_processlist    = $self->get_mysql_processlist();
    }

    my @parts = (
        $self->server_info($is_db_error), $self->exception_info($exception),
        $request_info,                    $self->backtrace_info($exception),
        $self->http_headers(),            $self->env_info(),
        $mysql_engine_innodb_dump,        $get_mysql_processlist,

        # NOTE! Substitution loop
        # $self->application_dump(),
    );

    return sprintf('
<html>
    <head>
        <meta http-equiv="content-type" content="text/html; charset=UTF-8">
        <title>Fatal error</title>
    </head>
    <body bgcolor="#FFFFFF" text="#000000">
        %s
    </body>
</html>',
        join("\n", @parts));
}

sub exception_info {
    my ($self, $exception) = @_;

    my $sentry_info;
    if (defined($exception->{'sentry'})) {
        $sentry_info =
          $exception->{'sentry'}{'event_url'}
          ? (
            sprintf(
                qq[<a href="%s">%s</a>],
                $exception->{'sentry'}{'event_url'},
                safe_html_encode($exception->{'sentry'}{'event_url'})
            )
          )
          : safe_html_encode($exception->{'sentry'}{'error'});
    } else {
        $sentry_info = 'No info about Sentry event. This is unexpected behaviour, please check the logs.';
    }

    return sprintf('
        <div id="error_block" style="background-color: #FF7777; font-size: 110%%; padding: 5px 10px; margin: 1px;">
            <h3 id="exception_part"> %s</h3>
            <strong>Login:</strong> %s <br>
            <strong>Date:</strong> %s <br>
            <strong>Host:</strong> %s <br>
            <strong>App:</strong> %s <br>
            <strong>Package:</strong> %s<br>
            <strong>Filename:</strong> %s (line %s) <br>
            <strong>PID:</strong> %d <br>
            <strong>Sentry Info:</strong> %s <br>
            <h4><pre id="exception_text">%s</pre></h4>
        </div>',
        safe_html_encode(ref($exception)),
        safe_html_encode($self->get_option('cur_user', {})->{'login'} // 'unknown'),
        safe_html_encode(format_date(curdate(), '%c')),
        safe_html_encode($self->get_option('hostname')),
        safe_html_encode(ref($self->app)),
        safe_html_encode($exception->{'package'}),
        safe_html_encode($exception->{'filename'}),
        safe_html_encode($exception->{'line'}),
        $$,
        $sentry_info,
        safe_html_encode($exception->message()),
    );
}

sub get_curr_dump_link {
    my ($self) = @_;
    return $_error_dump_link;
}

sub http_headers {
    my ($self) = @_;

    my $headers = $self->get_option('http_headers', {});

    return sprintf('
        <div style="background-color: #CCFFCC; padding: 5px 10px; margin: 1px;">
            <h3 id="http_headers">HTTP Headers:</h3>
            <table border="1" cellspacing="0" cellpadding="3" bordercolor="#CCCCCC">
            %s
            </table>
        </div>
    ',
        join(
            '',
            map {
                sprintf('
                  <tr>
                    <th align="right">%s</th>
                    <td>%s</td>
                  </tr>',
                    safe_html_encode($_),
                    safe_html_encode($headers->{$_} || ''))
              } sort keys(%$headers)
            ));
}

sub http_request_info {
    my ($self, $request) = @_;

    my @headers = (
        [host              => 'Host'],
        [referer           => 'Referer'],
        ['user-agent'      => 'User agent'],
        ['remote-addr'     => 'Remote address'],
        [accept            => 'Accept'],
        ['accept-encoding' => 'Accept encoding'],
        ['accept-language' => 'Accept languages'],
        [cookie            => 'Cookie']
    );

    return sprintf('
        <div style="background-color: #EEAA77; padding: 5px 10px; margin: 1px;">
          <h3 id="request_part">Request:</h3>
          <table width="100%%">
            <tr>
              <th valign="top" align="right">Method</th>
              <td>%s</td>
            </tr>
            <tr>
                <th valign="top" align="right">URL</th>
                <td>%s</td>
            </tr>
            ',
        safe_html_encode($request->method),
        safe_html_encode($request->url))
      . join(
        '',
        map {
            sprintf('
                <tr>
                  <th valign="top" nowrap="nowrap" align="right">%s</th>
                  <td>%s</td>
                </tr>',
                safe_html_encode($_->[1]),
                safe_html_encode($request->http_header($_->[0]) || ''))
          } @headers
      )
      . '
          </table>
        </div>';
}

sub get_mysql_innodb_status {
    my ($self) = @_;

    my $list          = $self->app->partner_db->_get_all('SHOW ENGINE INNODB STATUS');
    my $innodb_status = $list->[0]->{'Status'};

    return sprintf('
        <div style="background-color: #CCFFFF; padding: 5px 10px; margin: 1px;">
            <h3 id="mysql_innodb_status">MySql innodb status:</h3>
            <pre>
            %s
            </pre>
        </div>
        ',
        safe_html_encode($innodb_status));
}

sub get_mysql_processlist {
    my ($self) = @_;

    my $list   = $self->app->partner_db->_get_all("show full processlist");
    my $header = [qw(Id User Host db Command Time State Info Rows_sent Rows_examined Rows_read)];

    return sprinttmpl('
        <div style="background-color: #CCFFCC; padding: 5px 10px; margin: 1px;">
            <h3 id="mysql_processlist_part">MySql processlist:</h3>
            <pre id="mysql_processlist_text">
                <table border="1" cellspacing="0" cellpadding="3" bordercolor="#CCCCCC">
                    <th>
                    [% FOREACH h = header %]
                        <td>[% h %]</td>
                    [% END %]
                    </th>
                    [% FOREACH r = list %]
                        <tr>
                        [% FOREACH h = header %]
                            <td>[% r.${h} | html %]</td>
                        [% END %]
                        </tr>
                    [% END %]
                </table>
            </pre>
        </div>
        ',
        list   => $list,
        header => $header,);
}

sub safe_html_encode {
    my $str = shift // '';

    $str =~ s/&/&amp/g;
    $str =~ s/</&lt/g;

    return $str;
}

sub server_info {
    my ($self, $is_db_error) = @_;

    my $db_links = $is_db_error
      ? q[
                <li><a href="#mysql_innodb_status">MySql innodb status</a>
                <li><a href="#mysql_processlist">MySql processlist</a>]
      : '';

    return sprintf('
        <div style="background-color: #CCFF99; padding: 5px 10px; margin: 1px;">
            <h3 id="server_part">Server: %s</h3>
            <ul>
                <li><a href="#request_part">Request</a>
                <li><a href="#backtrace_part">Backtrace</a>
                <li><a href="#http_headers">HTTP Headers</a>
                <li><a href="#server_environment_part">Server enviroment</a>
                %s
                <!-- <li><a href="#application_dump_part">Application dump</a> -->
            </ul>
        </div>',
        safe_html_encode($self->get_option('hostname')),
        $db_links);
}

sub set_curr_dump_link {
    my ($self, $error_dump_link) = @_;
    $_error_dump_link = $error_dump_link;
}

sub short_dumper {
    my ($data, $max_depth) = @_;

    return '' unless defined($data);

    local $Data::Dumper::Indent   = 1;
    local $Data::Dumper::Sortkeys = 1;
    local $Data::Dumper::Maxdepth = $max_depth if defined($max_depth);
    local $Data::Dumper::Varname  = '';
    local $Data::Dumper::Sortkeys = TRUE;

    my $dtext = Dumper($data);

    $dtext =~ s|bless\( \{[^}]+\}|bless({ ... }|gs;

    utf8::decode($dtext) unless utf8::is_utf8($dtext);

    $dtext =~ s/\\x\{([a-f0-9]{2,})\}/chr(hex($1))/ge;
    $dtext =~ s/^\s*\$1 = //s;
    $dtext =~ s/;\s*$//;

    return $dtext;
}

TRUE;
