package Application::Model::Frontend;

use qbit;

use QBit::Validator;

use base qw(QBit::Application::Model);

use Exception;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::API::HTTP;

sub accessor {'frontend'}

__PACKAGE__->model_accessors(
    api_blackbox             => 'QBit::Application::Model::API::Yandex::BlackBox',
    api_geobase              => 'QBit::Application::Model::API::Yandex::HTTPGeobase',
    lang_detect              => 'QBit::Application::Model::YandexLangDetect',
    statistics               => 'Application::Model::Statistics',
    partner_db               => 'Application::Model::PartnerDB',
    product_manager          => 'Application::Model::ProductManager',
    rbac                     => 'Application::Model::RBAC',
    vip_partners             => 'Application::Model::VIP_Partners',
    resources                => 'Application::Model::Resources',
    context_on_site_campaign => 'Application::Model::Product::AN::ContextOnSite::Campaign',
    search_on_site_campaign  => 'Application::Model::Product::AN::SearchOnSite::Campaign',
    cur_user                 => 'Application::Model::CurUser',
    memcached                => 'QBit::Application::Model::Memcached',
    simple_notification      => 'Application::Model::SimpleNotification',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'frontend',
            description => d_gettext('Right to manage Frontend'),
            rights      => {}
        }
    ]
);

=head1 echo
    ->echo('АБВ');

Возвращает ту же строку, которую в нее передали.

=cut

sub echo {
    my ($self, $string) = @_;

    return $string;
}

=head1 fail

    ->fail()

Метод осуществляет выход из процесса. Создан для того чтобы тестировать как
клиенты rosetta себя ведут во время такого варианта.

=cut

sub fail {
    my ($self) = @_;

    exit;
}

=head2 find_resource

На входе:

    ->find_resource(
        name => 'campaign_id',
        value => '41443',
    );

На выходе:

    {
        resource => 'context_on_site_campaign',
    }

Умеет работать с name = 'campaign_id', 'id', 'site_id' (все эти поля означают одно и тоже
- Page ID), 'public_id', 'login', 'dsp_id', 'tag_id'.

В случае ошибки может выдать:

    {
        error => 'no_data_for_zero_block',
        error_message => 'Для нулевого блока не существует данных',
    }

=cut

sub find_resource {    # TODO: remove if unused by frontend
    my ($self, %opts) = @_;

    my %pages_field_name = ();

    foreach (@{$self->product_manager->get_page_model_accessors()}) {
        $pages_field_name{$self->app->$_->get_page_id_field_name()} = TRUE;

        foreach my $block_model ($self->app->$_->get_block_models()) {
            $pages_field_name{$block_model->get_page_id_field_name()} = TRUE;
        }
    }

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                name  => {in   => [sort keys(%pages_field_name), qw(application_id public_id login dsp_id tag_id)],},
                value => {type => 'scalar'},
                relaxed => {type => 'scalar', optional => TRUE},
            },
        },
        throw => TRUE,
    );

    my $name  = $opts{'name'};
    my $value = $opts{'value'};

    my $answer;

    if ($name eq 'login') {
        $answer = {resource => 'users'};
    } elsif ($name eq 'application_id') {
        $answer = {resource => 'internal_mobile_app'};
    } elsif ($name eq 'dsp_id') {
        $answer = {resource => 'dsp'};
    } elsif ($name eq 'tag_id') {
        $answer = {resource => 'block_tags'};
    } elsif ($pages_field_name{$name}) {

        my $product_structure = $self->product_manager->get_product_structure('page_id', $value);

        throw Exception::Validation::BadArguments gettext('Not found') unless @$product_structure;

        $answer = {resource => $product_structure->[0]{'products'}[0]{'model'}};
    } elsif ($name eq 'public_id') {
        my $res = $self->product_manager->get_model_data_from_public_id($value, $opts{'relaxed'});

        $answer = {
            resource  => $res->{'accessor'},
            public_id => $res->{'public_id'}
        };
    } else {
        throw Exception gettext('Internal error. This should not happen.');
    }

    return $answer;
}

=head2 get_available_resources

Возвращает массив хешей. В каждом хеше содержится описание части интерфейса к которому
у пользователя есть доступ. На основании этих данных фронтенд создает меню. Этот метод
возвращает плоский список. Данные о иерархии содержатся на фронтенде. Так же фронтенд
содержит ссылки на страницы.

Вид доступа кодируется с помощью значений methods. GET означает что есть право читать
ресурс, а POST означает что есть право создавать новый объект в этом ресурсе.
Порядок элеметов в массиве methods не определен. В массиве methods может быть
либоиGET, либо POST; либо и GET, и POST. Ключ methods всегда есть и его
значение - это всегда массив, в котором есть не менее одного элемента.

Из ответа этого метода нельзя однознано определить какой perl моделе соответствует
ресурс (и вообще есть ли перловая модель для этого ресурса).

Пример данныых, которые отдает этот метод:

    [
        # Эта часть интерфейса соответствует модели 'context_on_site_rtb'
        {
            resource => 'context_on_site_rtb',
            methods => ['GET', 'POST'],
        },

        # Эта часть интерфейса соответствует удаленным элементам модели 'context_on_site_rtb'
        {
            model => 'context_on_site_rtb_deleted',
            methods => ['GET'],
        },

        # А это часть интрфейса, для которой нет отдельной модели
        {
            resource => 'drop_user',
            methods => ['GET'],
        },
        ...
    ]

Поля resource уникальны.

Порядок элементов в массиве не определен, разные вызовы этого метода get_available_units
могут создавть разный порядок элеметов.

Через веб можно увидеть результат работы этого метода на странице
/v2/call/frontend/get_available_resources (работате только на бетах и не работает в проде).

=cut

sub get_available_resources {    # TODO: remove if unused by frontend
    my ($self) = @_;

    return $self->resources->get_available_resources();
}

=head2 get_cur_user_protected_campaings

Возвращает arrayref с хешами в которых содержится информация про protected
площадки текущего пользователя.

Возвращает площадкит только по продуктам (для задачи для которой делалась
эта ручка другие продукты были не важны, если понадобятся другие продукты
то их моно просто добавить):

 * context_on_site
 * search_on_site

В том случае если у пользователя нет protected площадок, то возвращает
пустой arrayref.

=cut

sub get_cur_user_protected_campaings {    # TODO: remove if unused by frontend
    my ($self) = @_;

    my $user_id = $self->cur_user->get_cur_user_id();

    my $context = $self->context_on_site_campaign->get_all(
        fields => [qw(domain page_id)],
        filter => ['AND' => [{multistate => 'protected'}, {owner_id => $user_id},],],
    );

    my $search = $self->search_on_site_campaign->get_all(
        fields => [qw(domain page_id)],
        filter => ['AND' => [{multistate => 'protected'}, {owner_id => $user_id},],],
    );

    return [@$context, @$search];
}

sub get_db_timelog {return {}}

=head2 get_locales

Возвращает hashref вида:
{
    en => "English",
    ru => "Русский",
}

=cut

sub get_locales {
    my ($self) = @_;

    my $locals_init = $self->app->get_locales();

    return {map {$_ => $locals_init->{$_}{name}} keys %$locals_init};
}

=head2 get_simple_notification

Возвращает array с сообщениями, которые указали менедежеры. Например:

    {
        messages => [{
            "ntype" => "survey",
            "message" => "<a href=#>ссылка</a> на опрос",
            "id" => 2,
            "icon" => "icon",
            "link" => "http://link.to",
        }]
    }

Возвращается всегда hash с элементом messages со списком оповещений,
пустой список означает что сообщения нет.

Сообщение приходит на языке пользователя.

=cut

sub get_simple_notification {    # TODO: remove if unused by frontend
    my ($self) = @_;

    my $notes = $self->simple_notification->get_all(fields => [qw(id ntype message icon link show_message params)],);

    return {messages => [grep {$_->{show_message}} @$notes]};
}

=head1 rm_standard_report

    ->rm_standard_report( $report_id );

Метод удаляет стандартный отчет для пользователя. После того как этот метот
выполнен, метод standard_reports() для этого бользователя больше не
возвращает этот отчет с $report_id.

=cut

sub rm_standard_report {
    my ($self, $report_id) = @_;

    my $user_id = $self->get_option('cur_user', {})->{'id'};

    $self->partner_db->deleted_standard_reports->add(
        {
            id      => $report_id,
            user_id => $user_id,
        }
    );

    return FALSE;
}

=encoding UTF-8

=head1 simple_lang_detect

    ->simple_lang_detect(
        ip => '95.220.164.216',
        accept_language => 'en-us,en;q=0.5',
        host => 'dev-partner.yandex.ru',
        my_cookie => 'YycCAAM2AQEA', # необязательный параметр
    );

Возвращает скаляр с языком пользователя, например 'ru' или 'en'.

=cut

sub simple_lang_detect {
    my ($self, %opts) = @_;

    my $ip = delete($opts{ip}) // throw Exception::Validation::BadArguments gettext('No ip');
    my $accept_language = delete($opts{accept_language})
      // throw Exception::Validation::BadArguments gettext('No accept_language');
    my $host      = delete($opts{host}) // throw Exception::Validation::BadArguments gettext('No host');
    my $cookie    = delete($opts{my_cookie});
    my $region_id = delete($opts{yandex_gid_cookie});

    my $regions;
    if ($region_id && $region_id =~ /^\d+$/) {
        $regions = $self->api_geobase->get_cached_region_parents(id => $region_id);
    } else {
        $regions = $self->api_geobase->get_cached_region_parents(ip => $ip);
    }

    my $user_lang = $self->lang_detect->get_user_lang(
        regions         => $regions,
        supported       => [sort keys(%{$self->get_option('locales')})],
        accept_language => $accept_language,
        pass_language   => $self->get_option('cur_user', {})->{'language'},
        host            => $host,
        cookie          => $cookie,
        default         => 'ru',
    );

    $self->app->set_app_locale($user_lang);

    return $user_lang;
}

sub find_blocks {
    my ($self, $text) = @_;

    $text //= '';

    my $app = $self->app;

    my $parts = [
        [qr/^(\d+)$/,                   'id'],
        [qr/^(\d+)-/,                   'page_id'],
        [qr/^\d+-(\d+)$/,               'block_id'],
        [qr/^([a-zA-Z]+-[a-zA-Z]+-?)/,  'public_id_prefix'],
        [qr/^[A-Z]+-[A-Z]+-(\d+)-?/,    'page_id'],
        [qr/^[A-Z]+-[A-Z]+-\d+(-)/,     'page_id_exact'],
        [qr/^[A-Z]+-[A-Z]+-\d+-(\d+)$/, 'block_id'],
    ];

    my %filter_opts = ();
    foreach my $part (@$parts) {
        if ($text =~ /$part->[0]/) {
            $filter_opts{$part->[1]} = $1;
        }
    }

    my $filter = $app->partner_db->filter();

    my %page_models = ();
    if (exists($filter_opts{'public_id_prefix'})) {
        my $public_id_prefix = $filter_opts{'public_id_prefix'};
        $public_id_prefix .= '-' if $public_id_prefix !~ /\-$/;
        $public_id_prefix = uc($public_id_prefix);

        my @models =
          grep {$public_id_prefix eq $app->$_->public_id_prefix} @{$app->product_manager->get_block_model_accessors()};

        if (@models) {
            $filter->and(['model', 'IN', \\@models]);
            %page_models = map {$app->$_->get_campaign_model_name() => TRUE} @models;
        } else {
            # "RTB-Mobile" - нужно фильтровать по caption
            delete($filter_opts{'public_id_prefix'});
        }
    }

    if (!%filter_opts && $text ne '') {
        my $find_value = QBit::Application::Model::DBManager::Filter::text::__like_str($text);
        $filter->and(['caption', 'LIKE', \$find_value]);
    }

    if (exists($filter_opts{'id'})) {
        my $find_id_value = QBit::Application::Model::DBManager::Filter::text::__like_str($filter_opts{'id'});

        $filter->and(
            [
                'OR',
                [
                    ['page_id', 'LIKE', \$find_id_value],
                    ['id',      'LIKE', \$find_id_value],
                    ['caption', 'LIKE', \$find_id_value],
                ]
            ]
        );
    }

    if (exists($filter_opts{'page_id'})) {
        if (exists($filter_opts{'block_id'}) || exists($filter_opts{'page_id_exact'})) {
            $filter->and(['page_id', '=', \$filter_opts{'page_id'}]);
        } else {
            my $find_page_id_value =
              QBit::Application::Model::DBManager::Filter::text::__like_str($filter_opts{'page_id'});
            $filter->and(
                ['OR', [['page_id', 'LIKE', \$find_page_id_value], ['caption', 'LIKE', \$find_page_id_value],]]);
        }
    }

    if (exists($filter_opts{'block_id'})) {
        my $find_block_id_value =
          QBit::Application::Model::DBManager::Filter::text::__like_str($filter_opts{'block_id'});
        my $block_id_filter = $app->partner_db->filter(['id', 'LIKE', \$find_block_id_value]);

        $filter->and($block_id_filter);
    }

    #authorization filter
    my %all_page_accessors = map {$_ => TRUE} @{$app->product_manager->get_page_model_accessors()};
    my @page_accessors =
      map {$_->{'resource'}}
      grep {%page_models ? $page_models{$_->{'resource'}} : $all_page_accessors{$_->{'resource'}}}
      @{$app->resources->get_available_resources()};

    return () unless @page_accessors;

    my @all_access;
    my @restricted_access_by_page_ids = ();
    foreach my $page_accessor (@page_accessors) {
        if ($app->$page_accessor->check_short_rights('view_all')) {
            push(@all_access, $page_accessor);
        } else {
            push(@restricted_access_by_page_ids,
                map {$_->{'page_id'}} @{$app->$page_accessor->get_all(fields => [qw(page_id)])});
        }
    }

    return () if !@all_access && !@restricted_access_by_page_ids;

    my $auth_filter = [
        'OR',
        [
            (@all_access ? ['model', 'IN', \[map {@{$app->$_->get_block_model_names()}} @all_access]] : ()),
            (@restricted_access_by_page_ids ? ['page_id', 'IN', \\@restricted_access_by_page_ids] : ()),
        ]
    ];

    $filter->and($auth_filter);

    my $data = $app->partner_db->all_blocks->get_all(
        fields   => [qw(page_id id model caption)],
        filter   => $filter,
        order_by => %filter_opts ? [qw(page_id id)] : [qw(caption page_id)],
        limit    => 10,
    );

    foreach my $el (@$data) {
        my $model              = $el->{'model'};
        my $page_id_field_name = $app->$model->get_page_id_field_name();
        $el->{'public_id'} = $app->$model->public_id({$page_id_field_name => $el->{'page_id'}, id => $el->{'id'}});

        # cast numbers for bionic (PI-25984)
        $el->{id}      .= '';
        $el->{page_id} .= '';
    }

    return @$data;
}

TRUE;
