package Application::Model::GeoBase;

use qbit;

use base qw(Application::Model::DBManager::Base RestApi::SimpleModel);

use Tree::Simple;

use QBit::Validator;

use Exception;
use Exception::Validation::BadArguments;

our %GEOBASE_CACHE;

sub accessor      {'geo_base'}
sub db_table_name {'geo_base'}

__PACKAGE__->register_rights(
    [
        {
            name        => 'geo_base',
            description => d_gettext('Right to manage geo base'),
        }
    ]
);

__PACKAGE__->model_accessors(
    partner_db  => 'Application::Model::PartnerDB',
    users       => 'Application::Model::Users',
    api_balance => 'Application::Model::API::Yandex::Balance',
    all_pages   => 'Application::Model::AllPages',
    crimea      => 'Application::Model::Crimea',
);

__PACKAGE__->model_fields(
    id        => {db => TRUE, pk      => TRUE, default => TRUE,},
    parent_id => {db => TRUE, default => TRUE, type    => 'number', api => 1},
    name      => {db => TRUE, default => TRUE, i18n    => TRUE, type => 'string', api => 1},
    level      => {db => TRUE, default => TRUE,     type => 'number', api => 1},
    regularity => {db => TRUE, type    => 'number', api  => 1},
    public_id  => {
        db      => TRUE,
        db_expr => 'id',
        type    => 'string',
        api     => 1
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id        => {type => 'number', label => d_gettext('Region ID')},
        parent_id => {type => 'number', label => d_gettext('Parent ID')},
        name      => {type => 'text',   label => d_gettext('Name')},
        level     => {type => 'number', label => d_gettext('Level')},
    },
);

sub get_product_name {gettext('geo_base')}

#API

sub api_available_actions {
    return qw();
}

sub api_get_all {
    my ($self, $controller, $resource, $one_object) = @_;

    $controller->check_params(($one_object ? 'public_id' : ()), 'user_id', "fields[$resource]");

    my @fields = $controller->get_fields($resource);

    my $url_simple_model_get = $controller->get_abs_for_sprintf('simple_model__resource_get', qw(resource public_id));

    my $model_fields = $self->get_model_fields();

    my $user_id = $controller->param('user_id') // $self->get_option('cur_user', {})->{'id'};
    my $user = $self->users->get($user_id, fields => ['country_id'])
      // throw Exception gettext('User with id "%s" not found', $user_id);

    my $country_id = $user->{'country_id'} // -1;

    my @result = ();
    foreach my $row (@{$self->get_all(fields => ['public_id', @fields])}) {
        my $res = {
            id   => "$row->{'public_id'}",
            type => $resource,
            ($one_object ? () : (links => {self => sprintf($url_simple_model_get, $resource, $row->{'public_id'}),})),
        };

        if ($row->{'public_id'} == 977) {
            if (in_array($country_id, ['225', '149', '159'])) {    #Россия Беларусь Казахстан
                $row->{'parent_id'} = 225;                         #Россия
                $row->{'level'}     = 1;
            } else {
                $row->{'parent_id'} = 187;                         #Украина
                $row->{'level'}     = 2;
            }
        }

        foreach (@fields) {
            $res->{'attributes'}{$_} =
              $controller->get_typed_value(undef, $resource, $model_fields->{$_}, $row->{$_}, $_);
        }

        push(@result, $res);
    }

    return \@result;
}

=encoding UTF-8

=head1 get_geobase

Метод возвращает плоское дерево с регионами. Крым привязывается либо к России, либо к Украине в зависимости от пользователя.

=head2 Использование:

=over 4

=item *

$self->get_geobase(user_id => <USER_ID>)

=back

=head2 Параметры:

=over 4

=item *

Метод принимает хэш %opts с ключом user_id

=back

=head2 Результат:

=over 4

=item *

Метод возвращает ссылку на массив, элементами которого являются хэши

=back

=cut

sub get_geobase {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {user_id => {type => 'int_un'},},
        },
        throw => TRUE
    );

    my $tmp_rights = $self->app->add_tmp_rights('users_view_all');

    my $country_id = $self->crimea->get_country_id_with_crimea($opts{'user_id'});

    my $ukraine_id = 187;
    my $russia_id  = 225;
    my $crimea_id  = 977;

    my $level;
    my $parent_id;

    if ($country_id == $russia_id) {
        $level     = 2;
        $parent_id = 26;
    } elsif ($country_id == $ukraine_id) {
        $level     = 3;
        $parent_id = 20526;
    } else {
        throw "Error in code. This should never happen.";
    }

    my $geobase = $self->get_all(
        fields   => [qw(id parent_id name regularity level)],
        order_by => ['level', ['regularity', TRUE], 'id']
    );

    foreach my $entry (@$geobase) {
        if ($entry->{'id'} == $crimea_id) {
            $entry->{'parent_id'} = $parent_id;
            $entry->{'level'}     = $level;
        }
        $entry->{$_} = defined $entry->{$_} ? "$entry->{$_}" : $entry->{$_} for keys %$entry;
    }

    return [sort {$a->{'level'} <=> $b->{'level'}} @$geobase];
}

=encoding UTF-8

=head1 get_tree_with_cpm

Метод возвращает список регионов с заданными для них cpm. Проставляет cpm для дочерних регионов

=head2 Использование:

=over 4

=item *

$self->get_tree_with_cpm(<PAGE_ID>, <REGIONS_WITH_CPM>)

=back

=head2 Параметры:

=over 4

=item *

$page_id

=item *

$data - ссылка на массив хэшей (список регионов, с заданными для них cpm, без дочерних регионов)

=back

=head2 Результат:

=over 4

=item *

Метод возвращает массив, элементами которого являются хэши

=back

=cut

sub get_tree_with_cpm {
    my ($self, $page_id, $data) = @_;

    return () unless $data;

    QBit::Validator->new(
        data     => [$page_id, $data],
        template => {
            type     => 'array',
            contents => [{type => 'int_un'}, {type => 'array'}]
        },
        throw => TRUE
    );

    return () unless @$data;

    my %cpm = map {$_->{'id'} => $_->{'cpm'}} @$data;

    unless (exists($GEOBASE_CACHE{$page_id})) {
        my $page = $self->all_pages->get_all(
            fields    => [qw(owner_id)],
            filter    => {page_id => $page_id},
            from_view => TRUE
        )->[0];

        throw Exception::Validation::BadArguments gettext('Unknown page_id "%s"', $page_id)
          unless defined($page);

        my $user_id = $page->{'owner_id'} // throw gettext('Owner not found');
        $GEOBASE_CACHE{$page_id} = $self->get_geobase(user_id => $user_id);
    }

    # need clone here because unfortunately tree processing below adds extra fields to $geo_base
    my $geo_base = clone($GEOBASE_CACHE{$page_id});

    my %ids = ();

    my $tree = Tree::Simple->new({}, Tree::Simple->ROOT);

    foreach (@$geo_base) {
        my $element = Tree::Simple->new($_);

        $ids{$_->{'id'}} = $element;

        if ($_->{'parent_id'}) {
            $ids{$_->{'parent_id'}}->addChild($element);
        } else {
            $tree->addChild($element);
        }
    }

    $tree->traverse(
        sub {
            my ($element) = @_;

            my $value = $element->getNodeValue();

            if ($cpm{$value->{'id'}}) {
                $value->{'cpm'} = $cpm{$value->{'id'}};
            } elsif (my $parent_cpm = $element->getParent->getNodeValue()->{'cpm'}) {
                $value->{'cpm'} = $parent_cpm;
                $cpm{$value->{'id'}} = $parent_cpm;
            }
        }
    );

    return map {{id => $_, cpm => $cpm{$_}}} keys(%cpm);
}

TRUE;
