package Application::Model::IntAPI_ACL;

=encoding UTF-8

=cut

use qbit;

use base qw(QBit::Application::Model);

use LWP::Simple;
use Net::Patricia;
use Socket qw(getaddrinfo getnameinfo SOCK_RAW NI_NUMERICHOST AF_INET6);
use Utils::Logger qw(WARN);

use Exception::Denied;
use Exception::HBF::NotFound;

sub accessor {'intapi_acl'}

__PACKAGE__->model_accessors(
    partner_db => 'Application::Model::PartnerDB::IntAPI',
    memcached  => 'QBit::Application::Model::Memcached',
    api_hbf    => 'Application::Model::API::Yandex::HBF',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'intapi_acl',
            description => d_gettext('Rights to manage internal API ACLs'),
            rights      => {
                intapi_acl_view => d_gettext('Right to view internal API ACLs'),
                intapi_acl_edit => d_gettext('Right to edit internal API ACLs'),
            },
        }
    ]
);

my $IPv4 =
"((25[0-5]|2[0-4][0-9]|[0-1]?[0-9]{1,2})[.](25[0-5]|2[0-4][0-9]|[0-1]?[0-9]{1,2})[.](25[0-5]|2[0-4][0-9]|[0-1]?[0-9]{1,2})[.](25[0-5]|2[0-4][0-9]|[0-1]?[0-9]{1,2}))";

my $G = "[0-9a-fA-F]{1,4}";

my @tail = (
    ":",                                 "(:($G)?|$IPv4)",
    ":($IPv4|$G(:$G)?|)",                "(:$IPv4|:$G(:$IPv4|(:$G){0,2})|:)",
    "((:$G){0,2}(:$IPv4|(:$G){1,2})|:)", "((:$G){0,3}(:$IPv4|(:$G){1,2})|:)",
    "((:$G){0,4}(:$IPv4|(:$G){1,2})|:)"
);

my $IPv6_re = $G;
$IPv6_re = "$G:($IPv6_re|$_)" for @tail;
$IPv6_re = qq/:(:$G){0,5}((:$G){1,2}|:$IPv4)|$IPv6_re/;
$IPv6_re =~ s/\(/(?:/g;
$IPv6_re = qr/$IPv6_re/;

=head2 acl2subnets

=cut

sub acl2subnets {
    my ($self, $acl) = @_;

    return () unless defined($acl);

    my @rules = $self->get_elements_from_acl($acl);

    my $subnet_re = qr/^(?:$IPv4|$IPv6_re)\/\d{1,3}$/;

    # Expand macroses
    my @macroses = grep {$self->is_macros($_)} @rules;
    if (@macroses) {
        @rules = grep {!$self->is_macros($_)} @rules;
        @rules = (@rules, $self->_expanded_macros_with_cache(@macroses));
    }

    # Hosts -> IPs
    my @hosts = grep {!/^$IPv4|$IPv6_re|$subnet_re$/} @rules;
    @rules = grep {/^$IPv4|$IPv6_re|$subnet_re$/} @rules;
    foreach my $host (@hosts) {
        my ($err, @result) = getaddrinfo($host, '', {socktype => SOCK_RAW});
        push(@rules, [getnameinfo($_->{'addr'}, NI_NUMERICHOST)]->[1]) foreach @result;
    }

    # IP -> Subnets
    foreach (@rules) {
        $_ = "$_/32"  if /^$IPv4$/;
        $_ = "$_/128" if /^$IPv6_re$/;
    }

    return grep {/$subnet_re/} @rules;
}

=head2 check_acl

=cut

sub check_acl {
    my ($self, $acl, $ip) = @_;

    return TRUE if $self->get_option('check_intapi_acl', 1) == 0;

    my $pt    = Net::Patricia->new();
    my $pt_v6 = new Net::Patricia AF_INET6;

    foreach (@{$acl}) {
        $pt->add_string($_, TRUE) if /^$IPv4\/\d{1,3}$/;
        $pt_v6->add_string($_, TRUE) if /^$IPv6_re\/\d{1,3}$/;
    }

    my $result = FALSE;

    try {
        $result = $pt->match_string($ip);
    }
    catch {
        try {
            $result = $pt_v6->match_string($ip);
        }
        catch {};
    };

    return !!$result;
}

=head2 get_acl

=cut

sub get_acl {
    my ($self, $path, $method) = @_;

    my $acl = $self->partner_db->intapi_acl->get([$path, $method], fields => ['acl']) // return undef;

    return $acl->{'acl'};
}

=head2 get_acl_cached

    my @acl = $app->intapi_acl->get_acl_cached($path, $method);

=cut

sub get_acl_cached {
    my ($self, $path, $method) = @_;

    my @acl;

    my $acl = $self->partner_db->intapi_acl->get([$path, $method], fields => ['acl_cached']);

    if ($acl) {
        @acl = @{from_json($acl->{'acl_cached'})};
    }

    return @acl;
}

=head2 get_all

=cut

sub get_all {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('You have not enough rights') unless $self->check_short_rights('view');

    my @methods;
    foreach my $path (keys(%{$self->{'__METHODS__'}})) {
        foreach my $method (keys(%{$self->{'__METHODS__'}{$path}})) {

            # wiki.yandex-team.ru плохо работает если создать страницу в урле которого есть
            # подчеркивание.
            # Если создать страницу `a_b`, то на самом деле будет создана страница `ab`
            # Похоже что вики просто игнорирует подчеркивание в урле, например,
            # можно зайти на страницу https://wiki.yandex-team.ru/p__ar_tner/
            # (то же самое что и https://wiki.yandex-team.ru/partner/ )
            # Поэтому тут подчеркивания заменяются на минус.
            my $wiki_method_name = $method;
            $wiki_method_name =~ s/_/-/g;

            push(
                @methods,
                {
                    path   => $path,
                    method => $method,
                    attrs  => clone($self->{'__METHODS__'}{$path}{$method}{'attrs'}),
                    wiki_link =>
                      sprintf('https://wiki.yandex-team.ru/partner/w/partner2-intapi-%s-%s/', $path, $wiki_method_name),
                }
            );
        }
    }

    if ($opts{'with_acl'}) {
        my %acls;
        $acls{$_->{'path'}, $_->{'method'}} = $_->{'acl'} foreach @{$self->partner_db->intapi_acl->get_all()};

        $_->{'acl'} = $acls{$_->{'path'}, $_->{'method'}} foreach @methods;
    }

    return \@methods;
}

=head2 get_location

=cut

sub get_location {
    my ($self) = @_;

    return 'intapi';
}

=head2 init

=cut

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    require IntAPI;
    $self->{'__METHODS__'} = IntAPI->get_methods();
}

=head2 set_acl

=cut

sub set_acl {
    my ($self, $path, $method, $acl) = @_;

    throw Exception::Denied gettext('You have not enough rights') unless $self->check_short_rights('edit');

    $self->partner_db->transaction(
        sub {
            my @subnets = $self->acl2subnets($acl);

            $self->partner_db->intapi_acl->add(
                {
                    path       => $path,
                    method     => $method,
                    acl        => $acl,
                    acl_cached => to_json(\@subnets),
                },
                replace => TRUE
            );

            $self->partner_db->intapi_acl_changes->add(
                {
                    path    => $path,
                    method  => $method,
                    new_acl => $acl,
                    user_id => $self->get_option(cur_user => {})->{'id'},
                    dt      => curdate(oformat => 'db_time')
                }
            );
        }
    );

    return TRUE;
}

=head2 get_elements_from_acl

Разрезает строку на набор элементов. Разделителем являются:

 * пробельные символы
 * запятая
 * строка 'or'

    my @elements = $app->intapi_acl->is_macros('_C_MARKET_STATGATE_TESTING_ _SQLLOGSRV_, , _MSELINGTESTSRV_ or 2a02:6b8:0:408:7dae:ffc4:1cf9:a291');

=cut

sub get_elements_from_acl {
    my ($self, $acl) = @_;

    return grep {length($_)} split(/\s*(?:,|\sor\s|\s)\s*/, $acl);
}

=head2 is_macros

Возвращает true значение если параметр является валидным firewall макросом.

    my $bool = $app->intapi_acl->is_macros('_SQLLOGSRV_'); # true
    my $bool = $app->intapi_acl->is_macros('2a02:6b8:0:408:7dae:ffc4:1cf9:a291'); # false

=cut

sub is_macros {
    my ($self, $maybe_a_macros) = @_;

    my $macro_re = qr/^_[A-Z_]+_$/;

    return $maybe_a_macros =~ /$macro_re/;
}

sub _expanded_macros_with_cache {
    my ($self, @macroses) = @_;

    my @result;

    my $prefix = 'hbf_macros';

    foreach my $macros (@macroses) {
        my $expanded = $self->memcached->get($prefix => $macros);

        if (!defined($expanded)) {

            my @hosts;

            try {
                @hosts = $self->api_hbf->get_hosts_from_macros($macros);
            }
            catch Exception::HBF::NotFound with {
                WARN
"HBF can't resolve macros $macros. Maybe it was deleted from conductor and you should delete it from ACL?";
            };

            $expanded = \@hosts;
            $self->memcached->set($prefix => $macros, $expanded, 60);
        }

        push(@result, @$expanded);
    }

    return @result;
}

TRUE;
