package Application::Model::Inviter;

=encoding UTF-8

=cut

use qbit;

use base qw(Application::Model::Common Application::Model::ValidatableMixin RestApi::MultistateModel);

use Clone qw(clone);
use Digest::MD5 qw(md5_hex);

use PiConstants qw(
  @ADFOX_PRESET_DEFAULT
  );

eval {require Exception::DB::DuplicateEntry};

use constant SITE_ADVERTISING_NETWORK => 1;
# Don't use constant DISTRIBUTION      => 2;
use constant DSP => 3;
# Don't use constant TAXOPARK          => 4;
use constant VIDEO_ADVERTISING_NETWORK                  => 5;
use constant APP_ADVERTISING_NETWORK                    => 6;
use constant SITE_ADVERTISING_NETWORK_PARTNER_ASSISTANT => 7;

use Exception::Inviter;
use Exception::Inviter::NoUser;
use Exception::HTTPPI::NotFound;
use Exception::DB::DuplicateEntry;
use Exception::Denied;
use Exception::Multistate;
use Exception::Multistate::BadAction;
use Exception::Multistate::NotFound;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

my $TTL   = 2592000;
my @TYPES = (
    {
        id           => SITE_ADVERTISING_NETWORK,
        label        => d_gettext('Advertising network: Sites'),
        role_id      => 9,
        check_rights => 'inviter_an_partner',
    },
    {
        id           => DSP,
        label        => d_gettext('DSP'),
        role_id      => 7,
        check_rights => 'inviter_dsp_partner',
    },
    {
        id           => VIDEO_ADVERTISING_NETWORK,
        label        => d_gettext('Video advertising network'),
        role_id      => 20,
        check_rights => 'inviter_video_an_partner',
    },
    {
        id           => APP_ADVERTISING_NETWORK,
        label        => d_gettext('Advertising network: Applications'),
        role_id      => 27,
        check_rights => 'inviter_application_an_partner',
    },
    {
        id           => SITE_ADVERTISING_NETWORK_PARTNER_ASSISTANT,
        label        => d_gettext('Advertising network: Partner\'s assistant'),
        role_id      => 37,
        check_rights => 'inviter_an_partner_assistant',
        is_assistant => TRUE,
    },
);
my %TYPES = map {$_->{'id'} => $_} @TYPES;

my %MESSAGES = ();

my $INVITE_LINK_REGEXP = qr/\[%\s*invite_link\s*%\]/;

my $FIELDS_DEPENDS;

sub accessor             {'inviter'}
sub db_table_name        {'invites'}
sub get_opts_schema_name {'invites_opts'}
sub get_product_name     {gettext('inviter')}

sub is_assistant_invite {
    my ($self, $type) = @_;

    return $self->_get_type_by_id($type)->{'is_assistant'};
}    # Присвоение внизу

sub _get_type_by_id {
    my ($self, $type) = @_;
    return $TYPES{$type};
}

sub get_structure_model_accessors {
    return {
        partner_db    => 'Application::Model::PartnerDB::Inviter',
        users         => 'Application::Model::Users',
        mailer        => 'QBit::Application::Model::SendMail',
        rbac          => 'Application::Model::RBAC',
        dsp           => 'Application::Model::DSP',
        text_template => 'Application::Model::TextTemplate',
    };
}

sub get_structure_rights_to_register {
    return [
        {
            name        => 'inviter',
            description => d_gettext('Rights to use inviter'),
            rights      => {
                inviter_view             => d_gettext('Right to view invites'),
                inviter_view_all         => d_gettext('Right to view invites of all types'),
                inviter_add              => d_gettext('Right to add invite'),
                inviter_an_partner       => d_gettext('Right invite partner on role "Advertising network: Sites"'),
                inviter_dsp_partner      => d_gettext('Right invite dsp partner'),
                inviter_video_an_partner => d_gettext('Right invite video advertising network partner'),
                inviter_application_an_partner =>
                  d_gettext('Right invite partner on role "Advertising network: Applications"'),
                inviter_an_partner_assistant =>
                  d_gettext('Right invite assistant on role "Advertising network: Sites - Partner\'s assistant"'),
                inviter_view_field__manager_id => d_gettext('Right to view field manager\'s login for invite'),
                inviter_view_field__actions    => d_gettext('Right to view field "actions" for invite'),
                inviter_view_field__type       => d_gettext('Right to view field "type" for invite'),
            },
        }
    ];
}

sub get_structure_model_fields {
    return {
        id => {
            default => TRUE,
            db      => TRUE,
            pk      => TRUE,
        },
        public_id => {
            depends_on => ['id'],
            get        => sub {
                return $_[1]->{'id'};
            },
            api  => 1,
            type => 'string',
        },
        type  => {db => TRUE},
        email => {
            default => TRUE,
            db      => TRUE,
            api     => 1,
            type    => 'string',
        },
        manager_id => {
            db   => TRUE,
            type => 'number',
        },
        partner_id => {
            db   => TRUE,
            type => 'number',
        },
        create_date => {
            db   => TRUE,
            api  => 1,
            type => 'string',
        },
        multistate => {
            db   => TRUE,
            api  => 1,
            type => 'number',
        },
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            api  => 1,
            type => 'string',
        },
        actions => {
            depends_on => [qw(id multistate partner_id type)],
            get        => sub {
                $_[0]->model->get_actions($_[1]);
            },
            api  => 1,
            type => 'complex',
        },
        type_name => {
            depends_on => [qw(type)],
            get        => sub {
                defined($TYPES{$_[1]->{'type'}}) ? $TYPES{$_[1]->{'type'}}->{'label'}() : '';
            },
            api  => 1,
            type => 'string',
        },
        partner => {
            api        => 1,
            depends_on => ['partner_id'],
            get        => sub {
                $_[0]->{'__USERS__'}->{$_[1]->{'partner_id'} || ''};
            },
            type => 'users',
        },
        manager => {
            api        => 1,
            depends_on => ['manager_id'],
            get        => sub {
                $_[0]->{'__USERS__'}->{$_[1]->{'manager_id'}};
            },
            type => 'users',
        },
        update_time => {
            from_opts => 'from_hash',
            api       => 1,
            type      => 'string',
        },
        page_id => {
            need_check => {
                type     => 'int_un',
                optional => TRUE,
            },
            api  => 1,
            type => 'number',
        },
        can_edit => {
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            api  => 1,
            type => 'boolean',
        },
        is_expired => {
            depends_on => [qw(create_date update_time)],
            get        => sub {
                my $date = date_add(
                    (
                        defined($_[1]->{update_time})
                          && $_[1]->{update_time} !~ /^0000/ ? $_[1]->{update_time} : $_[1]->{create_date}
                    ),
                    second  => $TTL,
                    iformat => 'db_time',
                    oformat => 'db_time',
                );

                return !is_date_in_future($date, iformat => 'db_time');
            },
        },
        available_fields => {
            label => d_gettext('Available fields'),
            get   => sub {
                return $_[0]->model->get_available_fields();
              }
        },
        editable_fields => {
            get => sub {
                $_[0]->model->get_editable_fields();
              }
        },
        fields_depends => {
            get => sub {
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
        },
    };
}

sub get_structure_model_filter {
    return {
        db_accessor => 'partner_db',
        fields      => {
            id         => {type => 'text',       label => d_gettext('ID')},
            email      => {type => 'text',       label => d_gettext('E-Mail')},
            multistate => {type => 'multistate', label => d_gettext('Status')},
            partner_id => {type => 'number',     label => d_gettext('Partner ID')},
            partner    => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'partner_id',
                fk_field       => 'id',
                label          => d_gettext('Partner'),
            },
            manager => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'manager_id',
                fk_field       => 'id',
                label          => d_gettext('Manager'),
            },
            type => {
                type   => 'dictionary',
                label  => gettext('Type'),
                values => sub {
                    [map {{id => $_->{'id'}, label => $_->{'label'}()}} @TYPES];
                  }
            },
        }
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__manager_login')
            ? {
                name         => 'manager.login',
                label        => gettext('Manager\'s login'),
                login_filter => 'dsp_managers=1'
              }
            : ()
        ),
        (
            $self->check_rights('view_search_filters__partner_login')
            ? {
                name         => 'partner.login',
                label        => gettext('Partner\'s login'),
                login_filter => 'dsp=1'
              }
            : ()
        ),
        {name => 'id',    label => gettext('Invitation identifier')},
        {name => 'email', label => gettext('E-Mail')},
        (
            $self->check_rights('view_search_filters__type') ? {name => 'type', label => gettext('Type')}
            : ()
        ),
        {name => 'multistate', label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    return {
        empty_name  => 'New',
        multistates => [
            [sent       => d_gettext('Sent invite')],
            [registered => d_gettext('Got the role')],
            [filled     => d_gettext('Form filled')],
            [bound      => d_gettext('Got a login')],
        ],
        actions => {
            send     => d_gettext('Send'),
            bind     => d_gettext('Bind'),
            register => d_gettext('Register'),
            fill     => d_gettext('Fill form'),
        },
        multistate_actions => [
            {
                action    => 'send',
                from      => '__EMPTY__',
                set_flags => ['sent'],
            },
            {
                action    => 'bind',
                from      => 'sent and not registered and not bound',
                set_flags => ['bound'],
            },
            {
                action    => 'register',
                from      => 'sent and bound and not registered',
                set_flags => ['registered'],
            },
            {
                action    => 'fill',
                from      => 'registered and not filled',
                set_flags => ['filled'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('partner') || $fields->need('manager')) {
        my @ids        = grep {$_} map {$_->{'partner_id'}, $_->{'manager_id'}} @$result;
        my $tmp_rights = $self->app->add_tmp_rights('users_view_all');
        my $list       = $self->users->get_all(
            fields => [qw(id login)],
            filter => {id => array_uniq(\@ids)},
        );
        undef $tmp_rights;
        foreach my $user (@$list) {
            $fields->{'__USERS__'}{$user->{'id'}} = $user;
            delete $user->{id} unless ($self->check_rights('users_view_all'));
        }
    }
}

sub get_available_fields {
    my ($self) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    delete $fields{$_} for qw(can_edit page_id);

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s'      => 'actions',
            $accessor . '_view_field__type'    => [qw( type type_name )],
            $accessor . '_view_field__manager' => [qw( manager manager_id )],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = {
        type     => TRUE,
        lang     => TRUE,
        email    => TRUE,
        text     => TRUE,
        subject  => TRUE,
        page_id  => TRUE,
        can_edit => TRUE,
    };

    return $fields;
}

sub get_editable_fields {{}}

sub add {
    my ($self, %opts) = @_;

    # validation
    {
        throw Exception::Denied unless $self->check_short_rights('add');

        my $locals          = $self->app->get_locales();
        my $types           = $self->_get_types();
        my $default_message = $self->get_default_message($opts{'type'}, $opts{'lang'});
        my $is_assistant    = $opts{'type'} == Application::Model::Inviter::SITE_ADVERTISING_NETWORK_PARTNER_ASSISTANT;
        if ($is_assistant) {
            $opts{text}    //= $default_message->{text};
            $opts{subject} //= $default_message->{subject};
        }

        $self->app->validator->check(
            data     => \%opts,
            template => {
                type   => 'hash',
                fields => {
                    type => {
                        type => 'int_un',
                        in   => [map {$_->{id}} @$types],
                    },
                    lang  => {type => 'scalar', in => [sort keys %$locals]},
                    email => {type => 'email',},
                    text  => {
                        type  => 'scalar',
                        check => sub {
                            my ($qv, $text) = @_;

                            if ($text ne $default_message->{'text'} && $is_assistant) {
                                throw Exception::Validator::Fields gettext('You are not allowed to edit this field');
                            }

                            if ($text !~ $INVITE_LINK_REGEXP) {
                                throw Exception::Validator::Fields gettext(
                                    'Do not remove the link \'[% invite_link %]\'');
                            }
                          }
                    },
                    subject => {
                        type  => 'scalar',
                        check => sub {
                            my ($qv, $subject) = @_;

                            if (   $subject ne $default_message->{'subject'}
                                && $is_assistant)
                            {
                                throw Exception::Validator::Fields gettext('You are not allowed to edit this field');
                            }
                          }
                    },
                    page_id => {
                        type     => 'scalar',
                        optional => TRUE,
                        check    => sub {
                            my ($qv, $page_id) = @_;

                            throw Exception::Validator::Fields gettext('Incorrect Page ID: "%s"', $page_id)
                              unless ($page_id =~ /^[0-9]+\z/);

                            my $page = $self->app->all_pages->get_all(
                                fields    => [qw(model)],
                                filter    => {page_id => $page_id},
                                from_view => 1,
                            )->[0];
                            throw Exception::Validator::Fields gettext('Page not found for page_id "%s"', $page_id)
                              unless $page;

                            my $model = $page->{model};
                            throw Exception::Validator::Fields gettext('You can not add invite for this page')
                              unless $self->app->$model->is_can_per_page_invite();
                          }
                    },
                    can_edit => {
                        type     => 'boolean',
                        optional => TRUE,
                    }
                }
            },
            throw => TRUE,
        );
    }

    my $invitee_id = $self->get_option('cur_user', {})->{'id'};

    my $existing = $self->partner_db->invites->get_all(
        fields => [qw(id email opts)],
        filter =>
          [AND => [['email', '=', \$opts{email}], ['type', '=', \$opts{type}], ['manager_id', '=', \$invitee_id],]],
        limit => 1,
    )->[0];

    my $id;
    if ($existing) {
        $id = $existing->{id};

        $self->partner_db->invites->edit($id,
            {opts => {json_set => ['opts', \'$.update_time', \curdate(oformat => 'db_time')]}});
        # check page_id
        if ($opts{'page_id'}) {
            my $assistant = $self->partner_db->assistants->get_all(
                filter => ['AND', [[inviter_id => '=' => \$id], [page_id => '=' => \$opts{'page_id'}]]],)->[0];

            unless ($assistant) {
                $self->partner_db->assistants->add(
                    {
                        inviter_id => $id,
                        page_id    => $opts{'page_id'},
                        user_id    => 0,
                        can_edit   => $opts{'can_edit'} // 0,
                    }
                );
            }
        }
    } else {
        $self->partner_db->transaction(
            sub {
                my $count = 0;
                while (!$id) {
                    try {
                        $id = $self->partner_db->invites->add(
                            {
                                id          => $self->_generate_id(%opts),
                                type        => $opts{'type'},
                                lang        => $opts{'lang'},
                                email       => $opts{'email'},
                                manager_id  => $self->get_option('cur_user', {})->{'id'},
                                create_date => curdate(oformat => 'db_time'),
                                multistate  => 0,
                                opts        => to_json({update_time => curdate(oformat => 'db_time')})
                            }
                        );

                        if ($opts{'page_id'}) {
                            $self->partner_db->assistants->add(
                                {
                                    inviter_id => $id,
                                    page_id    => $opts{'page_id'},
                                    user_id    => 0,
                                    can_edit   => $opts{'can_edit'} // 0,
                                }
                            );
                        }
                    }
                    catch Exception::DB::DuplicateEntry with {
                        my ($exception) = @_;
                        throw $exception if ++$count > 3;

                        # NOTE! _generate_id зависит от времени
                        sleep 1;
                    };
                }

                $self->do_action($id, 'send', hash_transform(\%opts, [qw(subject text lang type)]));
            }
        );
    }
    return $id;

}

=head2 on_action_bind

В этот метод попадаем когда пользователь заходит по секретной ссылке
которую он получил в письме от менеджера.
Привязывает приглашение к пользователю.

=cut

sub on_action_bind {
    my ($self, $obj, %opts) = @_;

    my $uid = $opts{'partner_id'};
    throw Exception::Validation::BadArguments gettext('"partner_id" must be defined.') unless defined($uid);

    $obj = $self->_get_object_fields($obj, [qw(id manager_id type partner_id email)]);
    throw Exception::Multistate::BadAction
      if defined($obj->{partner_id});    # по этой ссылке уже кто-то получил инвайт

    my $cur_user = $self->get_option('cur_user', {});
    throw Exception::Validation::BadArguments gettext(
        '"partner_id" that is passed to the method is not the uid of the current user')
      if $uid != $cur_user->{'id'};

    if ($cur_user->{'blackbox'}) {
        my $tmp_rights = $self->app->add_tmp_rights('do_user_action_add');
        $self->users->add(hash_transform($cur_user, [qw(id login email lastname midname name)], {id => 'uid'}));
    }

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id));
    my $partner = $self->users->get($uid, fields => [qw(login client_id)]);
    undef($tmp_rights);

    unless ($partner) {
        throw gettext("No user '%s' in database", $opts{'partner_id'});
    }

    $self->partner_db->invites->edit($obj, {partner_id => $opts{'partner_id'}});
    $self->partner_db->assistants->edit($self->partner_db->filter({inviter_id => $obj->{id}}),
        {user_id => $opts{'partner_id'}});
}

sub on_action_fill {
    my ($self, $invite_obj) = @_;

    $invite_obj = $self->_get_object_fields($invite_obj, [qw(id type manager_id partner_id email)]);

    my $partner;
    {
        my $tmp_rights = $self->app->add_tmp_rights('users_view_field__client_id');
        $partner = $self->users->get($invite_obj->{'partner_id'}, fields => [qw(login client_id email)]);
    };

    $self->mailer->send(
        from    => 'default',
        to      => {user_id => $invite_obj->{'manager_id'}},
        subject => gettext("New partner has filled the registration form. Login: %s", $partner->{'login'}),
        body    => $self->app->text_template->process_content(
            $MESSAGES{$invite_obj->{'type'}}->{'mail_template'},
            {
                invite_id        => $invite_obj->{'id'},
                client_id        => $partner->{'client_id'},
                login            => $partner->{'login'},
                email            => $partner->{'email'},
                invitation_email => $invite_obj->{'email'},
                url_origin       => $self->app->get_url_origin(),
            },
            locale => 'ru',
        ),
    );
}

sub can_action_register {
    my ($self, $obj) = @_;

    $obj = $self->_get_object_fields($obj, [qw(id type partner_id)]);

    my $cur_user = $self->get_option('cur_user', {});

    return $obj->{'partner_id'} eq $cur_user->{'id'} && !$self->has_user_role_already($obj);
}

=head2 on_action_register

В этот метод попадаем когда пользователь заходит по секретной ссылке
которую он получил в письме от менеджера

=cut

sub on_action_register {
    my ($self, $obj, %opts) = @_;

    $obj = $self->_get_object_fields($obj, [qw(id manager_id type partner_id email opts)]);
    my $partner_id = $obj->{'partner_id'};

    my $cur_user = $self->get_option('cur_user', {});
    throw Exception::Validation::BadArguments gettext('Invitation "partner_id" is not the current user id')
      if $partner_id != $cur_user->{'id'};

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id));
    my $partner = $self->users->get($partner_id, fields => [qw(id login client_id email)])
      // throw Exception::Validation::BadArguments gettext('Unknown partner');
    undef($tmp_rights);

    my $roles = $self->rbac->get_cur_user_roles();

    my $type_data = $self->_get_type_by_id($obj->{'type'});
    my $role_id   = $type_data->{'role_id'};

    if ($roles && exists($roles->{$role_id})) {
        throw Exception::Validation::BadArguments gettext(
            "User '%s' is already registered in the partner database with the role %s",
            $partner->{login}, $roles->{$role_id}{'name'});
    } else {
        if ($obj->{'type'} == SITE_ADVERTISING_NETWORK_PARTNER_ASSISTANT) {
            my $adfox_user;
            try {
                $adfox_user = $self->app->api_adfox_graphql->create_user(
                    user_id => $cur_user->{'id'},
                    presets => \@ADFOX_PRESET_DEFAULT
                );
            };
            unless ($adfox_user) {
                throw Exception::Validation::BadArguments gettext('Can not create_user in Adfox');
            }
            # assistant doesnt have roles yet
            my $tmp_rights = $self->app->add_tmp_rights(qw(do_user_action_link_adfox_user));
            $self->users->do_action(
                $cur_user->{'id'},
                'link_adfox_user',
                adfox_id         => $adfox_user->{id},
                adfox_login      => $adfox_user->{login},
                mobile_mediation => TRUE,
            );
        }
        my $tmp_rights = $self->app->add_tmp_rights('rbac_user_role_set');
        $self->rbac->set_user_role($partner_id, $role_id);
    }

    if ($obj->{'type'} == DSP) {
        my $tmp_rights = $self->app->add_tmp_rights(qw(do_user_action_request_create_in_banner_store));
        $self->users->do_action($partner_id, 'request_create_in_banner_store');
    }

    $self->mailer->send(
        from    => 'default',
        to      => {user_id => $obj->{'manager_id'}},
        subject => $self->is_assistant_invite($obj->{'type'})
        ? gettext("Inviter: assistant %s has been registered", $partner->{'login'})
        : gettext("Inviter: partner %s has been registered",   $partner->{'login'}),
        body => $self->app->text_template->process_content(
            $MESSAGES{$obj->{'type'}}->{'mail_template'},
            {
                invite_id        => $obj->{'id'},
                client_id        => $partner->{'client_id'},
                login            => $partner->{'login'},
                invitation_email => $obj->{'email'},
                email            => $partner->{'email'},
                url_origin       => $self->app->get_url_origin(),
            },
            locale => 'ru'
        ),
    );
}

sub on_action_send {
    my ($self, $obj, %opts) = @_;

    my @missed_req_fields = grep {!exists($opts{$_})} qw(subject text lang type);
    throw Exception::Validation::BadArguments gettext('Missed required fields "%s"', join(@missed_req_fields, ', '))
      if @missed_req_fields;

    $obj = $self->_get_object_fields($obj, [qw(id email)]);

    my $tmp_locale = $self->app->set_tmp_app_locale($opts{'lang'});

    my %type_email_opts = $self->get_type_email_opts($opts{'type'});

    my $invite_link = $self->_get_invite_link($obj->{'id'}, hash_transform(\%opts, [qw(lang)]));
    $opts{'text'} =~ s/$INVITE_LINK_REGEXP/$invite_link/g;

    $self->mailer->send(
        from => $type_email_opts{'from'} // {
                $self->get_option('cur_user', {})->{'email'} => $self->get_option('cur_user', {})->{'name'} . ' '
              . $self->get_option('cur_user', {})->{'lastname'}
        },
        to => $obj->{'email'},
        (exists($type_email_opts{'cc'}) ? (cc => $type_email_opts{'cc'}) : ()),
        subject => $opts{'subject'},
        body    => $opts{'text'}
    );
}

sub query_filter {
    my ($self, $filter) = @_;

    throw Exception::Denied unless $self->check_short_rights('view');

    unless ($self->check_short_rights('view_all')) {
        my $type = [map {$_->{id}} @{$self->_get_types}];
        $filter->and({type => $type, manager_id => $self->get_option('cur_user', {})->{'id'}});
    }

    return $filter;
}

sub api_available_actions {
    return qw(
    );
}

sub api_can_add         {TRUE}
sub api_can_edit        {FALSE}
sub api_can_delete      {FALSE}
sub api_check_public_id {$_[1] =~ /^\w{1,32}$/i}

sub get_fields_depends {
    return {
        #если поменялось поле из ключа, то
        #нужно перезапросить поля из значения
        depends => {
            type => [qw(default_message)],
            lang => [qw(default_message)],
        },
        #для поля из ключа обязятельно нужно передать поля из значения
        required => {default_message => [qw(type lang)],},
    };
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{types}) {
        $result{types} = $self->get_types();
    }
    if ($need_fields->{default_message}) {
        $result{default_message} = $self->get_default_message($opts->{attributes}{type}, $opts->{attributes}{lang});
    }

    return \%result;
}

sub accept {
    my ($self, $invitation_id) = @_;

    $self->check_link_is_valid($invitation_id);

    my $cur_user = $self->get_option('cur_user');

    my $tmp_rights = $self->app->add_tmp_rights(qw(inviter_view inviter_view_all));

    if ($self->check_action($invitation_id, 'bind')) {
        $self->do_action($invitation_id, 'bind', partner_id => $cur_user->{'id'});
    } else {
        my $invite = $self->get($invitation_id, fields => ['partner_id']);

        throw Exception::Validation::BadArguments gettext('Invite link already belongs to another user')
          unless ($self->check_action($invitation_id, 'register') && $invite->{partner_id} eq $cur_user->{id});
    }

    return TRUE;
}

sub check_link_is_valid {
    my ($self, $invitation_id) = @_;

    my $invitation_link_can_be_bound;
    try {
        my $tmp_rights = $self->app->add_tmp_rights(qw(inviter_view inviter_view_all));

        $invitation_link_can_be_bound = $self->check_action($invitation_id, 'bind');
        my $invitation_link_can_be_registered = $self->check_action($invitation_id, 'register');

        throw Exception::Validation::BadArguments gettext('Incorrect invite link')
          unless $invitation_link_can_be_bound || $invitation_link_can_be_registered;

        throw Exception::Denied gettext("You don't have access to this page")
          if $self->check_rights('yndx_login');

        my $user_already_has_role = $self->has_user_role_already($invitation_id);
        throw Exception::Validation::BadArguments gettext('You already have this role')
          if $user_already_has_role

    }
    catch Exception::Multistate with {
        throw Exception::Validation::BadArguments gettext('Incorrect invite link');
    };

    return TRUE;
}

sub get_default_message {
    my ($self, $type, $lang) = @_;

    my $locals = $self->app->get_locales();
    my $types  = $self->_get_types();

    $self->app->validator->check(
        data => {
            'type' => $type,
            'lang' => $lang
        },
        template => {
            type   => 'hash',
            fields => {
                type => {
                    type => 'int_un',
                    in   => [map {$_->{id}} @$types],
                },
                lang => {type => 'scalar', in => [sort keys %$locals]},
            }
        },
        throw => TRUE,
    );

    my $result = {
        subject  => '',
        text     => '',
        disabled => 1
    };

    if (exists($MESSAGES{$type})) {
        my $tmp_local = $self->app->set_tmp_app_locale($lang);
        $result = {
            subject  => $MESSAGES{$type}->{'subject'}(),
            text     => $self->text_template->get_content($MESSAGES{$type}->{'text_caption'}),
            disabled => $type == SITE_ADVERTISING_NETWORK_PARTNER_ASSISTANT ? TRUE : FALSE
        };
    }

    return $result;
}

=head2 get_manager_ids

B<Параметры:> 1) $self 2) $user_id - uid партнреа

B<Возвращаемое значение:> 1) @manager_ids - список uid менеджеров, у которых
есть принадлежашие им инвайты, которыми воспользовался пользователь с uid
$user_id

Чаще всего будет возвращатся список из одного uid менеджера.

Метод бросает исключение Exception::Inviter::NoUser если пользователя с
uid $user_id не проходил через систему инвайтов.

=cut

sub get_manager_ids {
    my ($self, $user_id) = @_;

    my $results = $self->partner_db->invites->get_all(filter => {partner_id => $user_id,},);

    my @manager_ids = map {$_->{manager_id}} @{$results};
    my @uniq_manager_ids = @{array_uniq(@manager_ids)};

    if (scalar(@uniq_manager_ids) > 0) {
        return @uniq_manager_ids;
    } else {
        throw Exception::Inviter::NoUser gettext("User with uid %s has no invite manager", $user_id);
    }

}

sub get_type_email_opts {
    my ($self, $type) = @_;

    my %type_email_opts = %{(grep {$_->{'id'} eq $type} @{$self->_get_types()})[0]->{'email'} // {}};
    foreach (qw(from cc)) {
        $type_email_opts{$_} = $type_email_opts{$_}->($self)
          if exists($type_email_opts{$_}) && ref($type_email_opts{$_}) eq 'CODE';
    }

    return %type_email_opts;
}

sub get_types {
    my ($self) = @_;

    my $init_types = $self->_get_types();

    return [
        map +{
            id    => $_->{id},
            label => $_->{'label'}(),
        },
        @$init_types
    ];
}

sub has_user_role_already {
    my ($self, $obj) = @_;

    $obj = $self->_get_object_fields($obj, [qw(id type)]);
    throw Exception::Validation::BadArguments gettext('Object not found') unless defined($obj);

    my $type_data = $self->_get_type_by_id($obj->{'type'});
    throw Exception::Validation::BadArguments gettext('Unknown invitation type') unless $type_data;

    my $roles = $self->rbac->get_cur_user_roles();

    return $roles && exists($roles->{$type_data->{'role_id'}});
}

sub save {
    my ($self, $invitation_id, %opts) = @_;

    $self->check_link_is_valid($invitation_id);

    $self->app->validator->check(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                name => {
                    type    => 'scalar',
                    len_max => 255,
                },
                midname => {
                    type     => 'scalar',
                    len_max  => 255,
                    optional => TRUE
                },
                lastname => {
                    type    => 'scalar',
                    len_max => 255,
                },
                email => {
                    type    => 'scalar',
                    len_max => 255,
                    check   => sub {
                        my ($qv, $emails) = @_;

                        foreach my $email (split(/\s*,\s*/, $emails)) {
                            throw Exception::Validator::Fields gettext('Incorrect E-Mail') if !check_email($email);
                        }

                        return '';
                    },
                },
                phone => {
                    type    => 'scalar',
                    len_max => 255,
                },
                newsletter => {type => 'boolean',},
            }
        },
        throw => TRUE,
    );

    my $cur_user = $self->get_option('cur_user', {});

    if ($cur_user->{'blackbox'}) {
        my $tmp_rights = $self->app->add_tmp_rights('do_user_action_add');
        $self->users->add(
            provide_contacts => TRUE,
            uid              => $cur_user->{'id'},
            login            => $cur_user->{'login'},
            %opts,
        );
    } else {
        $self->users->edit($cur_user->{'id'}, %opts, provide_contacts => TRUE);
    }

    try {
        my $tmp_rights = $self->app->add_tmp_rights(qw(inviter_view inviter_view_all));
        $self->do_action($invitation_id, 'register');
    }
    catch Exception::Multistate::BadAction catch Exception::Multistate::NotFound with {
        throw Exception::Validation::BadArguments gettext('Incorrect invite link');
    };

    return $self->users->check_action($cur_user->{'id'}, 'created_partner_in_banner_store')
      ? {redirect => 'DSP'}
      : {};
}

sub _generate_id {
    my ($self, %opts) = @_;

    return md5_hex(md5_hex($self->get_option('salt', '')) . join(%opts) . md5_hex(time()));
}

sub _get_invite_link {
    my ($self, $id, %opts) = @_;

    $opts{'use_url_origin'} = TRUE;

    return $self->app->get_url_origin(%opts) . '/v2/invitation?id=' . $id;
}

sub _get_types {
    my ($self) = @_;

    return clone([grep {$self->check_rights($_->{check_rights})} @TYPES]);
}

sub get_email_list {
    my ($self) = @_;

    return $self->get_all(filter => {manager_id => $self->get_option('cur_user', {})->{'id'},});
}

$MESSAGES{SITE_ADVERTISING_NETWORK()} = {
    subject       => d_gettext('Invitation to Yandex Advertising Network Partner Program'),
    text_caption  => 'Invitation to Yandex Advertising Network Partner Program Body',
    mail_template => 'Body on partner invitation registration',
};

$MESSAGES{DSP()} = {
    subject       => d_gettext('Invitation to Yandex DSP Partner Program'),
    text_caption  => 'Invitation to Yandex DSP Partner Program Body',
    mail_template => 'Body on partner invitation registration',
};

$MESSAGES{VIDEO_ADVERTISING_NETWORK()} = {
    subject       => d_gettext('Invitation to Yandex Video Advertising Network Partner Program'),
    text_caption  => 'Invitation to Yandex Video Advertising Network Partner Program Body',
    mail_template => 'Body on partner invitation registration',
};

$MESSAGES{APP_ADVERTISING_NETWORK()} = {
    subject       => d_gettext('Invitation to Yandex Mobile apps Advertising Network Partner Program'),
    text_caption  => 'Invitation to Yandex Mobile apps Advertising Network Partner Program Body',
    mail_template => 'Body on partner invitation registration',
};

$MESSAGES{SITE_ADVERTISING_NETWORK_PARTNER_ASSISTANT()} = {
    subject => d_gettext('Invitation to Yandex Advertising Network Partner Program with role "Partner\'s assistant"'),
    text_caption  => 'Invitation to Yandex Advertising Network Partner Program Body with role "Partner\'s assistant"',
    mail_template => 'Body on assistant invitation registration',
};

TRUE;
