package Application::Model::Kladr;

=encoding UTF-8

=cut

use qbit;

use base qw(QBit::Application::Model::DB::mysql);

sub accessor {'kladr'}

sub address_has_street {
    my ($self, $data) = @_;

    my $geo_object;
    if ($data->{town}) {
        $geo_object = $data->{town};
    } elsif ($data->{city}) {
        $geo_object = $data->{city};
    } elsif ($data->{district}) {
        $geo_object = $data->{district};
    } elsif ($data->{region}) {
        $geo_object = $data->{region};
    } else {
        return 0;
    }

    my ($result) = $self->{dbh}->selectrow_array(
        q{
        select count(Name)
        from
            STREET
        where
            SUBSTR(Code, 1, 11) = ?
            and Code RLIKE "00$"
        },
        undef, $geo_object
    );

    return $result ? 1 : 0;
}

sub convert_addr {
    my $self = shift;
    my ($data, $prefix, $kladr_addr_data) = @_;
    my $addr = $self->validate_address(
        $self->extract_street_code($data, $prefix),
        $data->{$prefix . 'home'},
        $data->{$prefix . 'building'},
        $data->{$prefix . 'construction'},
        $data->{$prefix . 'flat'},
        $data->{$prefix . 'postcode'}
    );

    if (defined $addr) {
        if (ref $kladr_addr_data eq 'HASH') {
            $kladr_addr_data->{$_} = $addr->{$_} for (qw{code home building construction flat});
            # если задан хэш, в который надо возвращать результат, то из хэша с данными поля уничтожаются
            foreach my $field_name (qw{street town city district region home building construction flat postcode}) {
                my $name = $prefix . '_' . $field_name;
                delete $data->{$name};
            }
        }
        return $self->get_addr_str($addr);
    }
    return;
}

sub extract_street_code {
    my $self = shift;
    my ($data, $prefix) = @_;

    $prefix //= '';

    my @fields_street = qw{street town city district region};
    foreach my $field_name (@fields_street) {
        my $name = $prefix . $field_name;
        return $data->{$name} if $data->{$name};
    }
}

sub get_addr_hash {
    my $self = shift;
    my ($address, $full) = @_;

    # сфоримруем первые четыре уровня (регион, район, город, населённый пункт)

    my @code_parts = $self->_split_code($address->{code});
    my (@codes, $prev_code);
    foreach my $code_part (@code_parts[0 .. 3]) {
        $prev_code .= $code_part;
        push(@codes, $prev_code . '0' x (11 - length $prev_code)) if $code_part =~ /[^0]/;
    }
    my $in = join(', ', map(qq{"${_}00"}, @codes));

    my $codes = $self->_get_all(
        qq{
            select
                CONCAT(Name, ' ', Socr) name,
                code,
                status
            from
                KLADR
            where
                Code IN ($in)
            order by Code
        }
    );

    my %addr = (
        postcode     => $address->{postcode},
        gni          => $address->{gni},
        region       => undef,
        district     => undef,
        city         => undef,
        town         => undef,
        street       => undef,
        home         => undef,
        building     => undef,
        construction => undef,
        flat         => undef
    );

    foreach my $code (@$codes) {
        unless ($full) {
            if ($code->{status} == 1) {    # районный центр, убираем название района
                undef $addr{district};
            }
            if ($code->{status} == 2 || $code->{status} == 3)
            { # центр региона и, возможно, района - убираем название региона и района
                undef $addr{region};
                undef $addr{district};
            }
        }
        $addr{$self->_level_name_by_code($code->{code})} = $code->{name};
    }

    # добавляем название улицы
    my $street = $self->_get_all("select CONCAT(Name, ' ', Socr) name from STREET s where code=CONCAT(?, '00')",
        substr($address->{code}, 0, 15))->[0];
    $addr{street} = $street->{name} if defined $street;

    $address->{$_} and $addr{$_} = $address->{$_} for (qw{home building construction flat});

    return \%addr;
}

sub get_addr_str {
    my $self    = shift;
    my $address = $self->get_addr_hash(@_);

    my @addr;
    $address->{$_} and push(@addr, $address->{$_}) for (qw{postcode region district town city street});

    # построим каноническое название дома
    my $addr_str = "";
    $addr_str = $address->{home} if $address->{home};
    $addr_str .= "к" . $address->{building}         if $address->{building};
    $addr_str .= "стр" . $address->{construction} if $address->{construction};
    push(@addr, $addr_str)        if $addr_str;
    push(@addr, $address->{flat}) if $address->{flat};

    return join(', ', @addr);
}

sub get_doma_by_code {
    my $self = shift;
    my $code = shift;

    return $self->{dbh}->selectall_arrayref(
        q{select Code, Name, `Index` from DOMA where Code LIKE ?},
        {Slice => {}},
        substr($code, 0, 15) . '%'
    );
}

sub get_streets {
    my $self = shift;
    my ($code, $index) = @_;

    $index = $self->_fix_index($index);
    return [] unless $index;

    # выберем по индексу и паренту нужные улицы
    my $sth = $self->{dbh}->prepare(
        q{
            (
                select
                    SUBSTR(Code, 1, 15),
                    CONCAT(Name, ' ', Socr) name
                from
                    STREET s
                where
                        s.Index like ?
                    and SUBSTR(Code, 1, 11) = ?
                    and Code RLIKE "00$"
            )

            union

            (
                select
                    SUBSTR(s.Code, 1, 15),
                    CONCAT(s.Name, ' ', s.Socr) name
                from
                    DOMA d
                left join
                    STREET s
                    on s.CODE = CONCAT(substring(d.Code, 1, 15), "00")
                where
                        d.`Index` like ?
                    and SUBSTR(d.Code, 1, 11) = ?
            )

            union

            (
                select
                    SUBSTR(s.code, 1, 15),
                    CONCAT(s.Name, ' ', s.Socr) name
                from
                    KLADR k
                left join
                    STREET s
                    on SUBSTRING(s.code, 1, 11) = SUBSTRING(k.code, 1, 11)
                where
                    k.`Index` like ?
                    and SUBSTR(k.Code, 1, 11) = ?
                    and s.Code RLIKE "00$"
            )

            order by name
        }
    );

    $sth->execute($index, $code, $index, $code, $index, $code);
    my @result;
    while (my ($code, $name) = $sth->fetchrow_array) {
        push(@result, {code => _def($code), name => _def($name)});
    }

    unless (@result) {    # деревня без улиц
        return [{code => $code, name => gettext("no streets")}];
    }

    my ($w_or_wo) =
      $self->{dbh}
      ->selectrow_array('select Code from exceptions where Code=? and kind=?', undef, $code, 'w_or_w/o_streets');
    if ($w_or_wo)
    {    # для исключений, в которых адреса и с улицами есть, и без улиц
        unshift(@result, {code => '', name => ''});
    }

    return \@result;
}

sub get_streets_by_part {
    my $self = shift;
    my ($code, $part) = @_;

    my $streets = $self->_get_all(
        q{
        select
            SUBSTR(Code, 1, 15) code,
            CONCAT(Name, ' ', Socr) full_name
        from
            STREET
        where
            SUBSTR(Code, 1, 11) = ?
            and Code RLIKE "00$"
            and (
                `Name` LIKE ?
                or `Name` LIKE ?
            )
        order by full_name
        limit 20
        }, substr($code, 0, 11), $part . '%', "% $part%"
    );

    return [map {[$_->{'code'}, $_->{'full_name'}]} @$streets];
}

sub get_towns {
    my $self  = shift;
    my $index = shift;

    $index = $self->_fix_index($index);
    return [] unless $index;

    my $codes = $self->_get_all(
        q{
            select
                SUBSTR(Code, 1, 11) street
            from
                STREET
            where `Index` like ? and Code rlike "00$"

            union

            select
                SUBSTR(d.Code, 1, 11) street
            from
                DOMA d
            inner join KLADR k on CONCAT(SUBSTR(d.Code, 1, 11), '00') = k.Code
            where d.`Index` like ?

            union

            select
                SUBSTR(Code, 1, 11) street
            from
                KLADR
            where `Index` like ? and CODE rlike "00$"
        },
        $index, $index, $index
    );

    my $hash = {};
    my @codes;

    # построим временную структуру и найдём коды промежуточных элементов
    foreach my $code (map {$_->{'street'}} @$codes) {
        my $root = $hash;
        my $prev_subcode;

        my @code_parts = $self->_split_code($code);
        foreach my $code_part (@code_parts) {
            my $subcode = "";
            $prev_subcode .= $code_part;
            if ($code_part =~ /[1-9]/) {
                $subcode = $prev_subcode . "0" x (11 - length($prev_subcode));
            }
            if (not exists $root->{$subcode}) {
                $root->{$subcode} = {};
                push(@codes, $subcode) if $subcode;
            }
            $root = $root->{$subcode};
            last unless $code;
        }
    }

    return [] unless scalar @codes;

    # найдём названия
    my $in = join(', ', map(qq{"${_}00"}, @codes));
    my $code_names =
      $self->_get_all("select SUBSTR(Code, 1, 11) code, CONCAT(Name, ' ', Socr) name from KLADR where Code IN ($in)");
    my %code_names;
    $code_names{$_->{code}} = $_->{name} foreach @$code_names;
    $self->{code_names} = \%code_names;

    return $self->_create_list($hash, 0);
}

sub get_towns_by_code {
    my $self = shift;
    my $code = shift // '';
    my @result;

    $code =~ s/\D//g;

    # разобъём на уровни кодов
    $code .= '0' x (11 - length($code));
    my @code_parts = $code =~ /(..)(...)(...)(...)/;

    # найдём самую правую значащую часть
    my $sig_part = $#code_parts;
    $sig_part-- while $code_parts[$sig_part] =~ /^0+$/ && $sig_part >= 0;

    if ($sig_part < 3) {
        # построим условия для выбора дерева
        my $where;
        if ($sig_part >= 0) {
            my @or;
            for (my $i = $sig_part + 1; $i <= $#code_parts; $i++) {
                my $like;
                $like .= join('', @code_parts[0 .. $i - 1]);
                $like .= '___';
                $like .= '0' x (13 - length($like));
                push(@or, "Code like '$like'");
            }

            $where = join(' or ', @or);
            $where = "Code!='${code}00' and ($where)";
        } else {
            $where = "Code like '__00000000000'";
        }

        my $towns = $self->_get_all(
"select SUBSTR(Code, 1, 11) code, CONCAT(Name, ' ', Socr) full_name from KLADR where $where order by Code not in ('7700000000000', '7800000000000'), full_name"
        );

        foreach (@$towns) {
            my $level = $self->_level_by_code($_->{code}) - ($sig_part + 1);
            push(@{$result[$level]}, [$_->{code}, $_->{full_name}]);
        }
    }

    if ($sig_part >= 0) {
        my $street_count = $self->_get_all("select count(*) cnt from STREET where Code like '$code%'")->[0]{'cnt'};
        $result[3 - $sig_part] = [] if $street_count > 0;
    }

    return \@result;
}

sub index_exist {
    my ($self, $index) = @_;

    my ($result) = $self->{dbh}->selectrow_array(
        q{
            select 1 from STREET
            where `Index`= ? and Code rlike "00$"

            union

            select 1 from DOMA
            where `Index`=?

            union

            select 1 from KLADR
            where `Index`=? and CODE rlike "00$"
        },
        undef, $index, $index, $index
    );

    return $result;
}

sub name_by_code {
    my $self = shift;
    my $code = shift;

    if (length($code) == 15) {
        return $self->{dbh}
          ->selectrow_array('select CONCAT(Name, " ", Socr), `Index` from STREET where SUBSTR(Code, 1, 15)=?',
            undef, $code);
    } else {
        return $self->{dbh}
          ->selectrow_array('select CONCAT(Name, " ", Socr), `Index` from KLADR  where SUBSTR(Code, 1, 11)=?',
            undef, $code);
    }
}

sub populate_addr_selects {
    my ($self, $prefix, $data) = @_;
    if ($data->{$prefix . "code"}) {
        my $addr = $self->validate_address(
            substr($data->{$prefix . "code"}, 0, 15),
            $data->{$prefix . "home"},
            $data->{$prefix . "building"},
            $data->{$prefix . "construction"},
            $data->{$prefix . "flat"}
        );
        $data->{$prefix . "postcode"} = $addr->{postcode} if $addr->{postcode};
    }
    my $index = $data->{$prefix . "postcode"};
    return unless $index;

    my $towns = $self->get_towns($index);
    my $current = {children => $towns};
    my $last_code;

    my @code_parts = $data->{$prefix . "code"} ? $self->_split_code($data->{$prefix . "code"}) : ();
    my $temp_code = "";
    foreach my $field_name (qw{region district city town}) {
        my $code_part = shift @code_parts;
        $temp_code .= $code_part;
        my $flag_selected;
        foreach my $obj (@{$current->{children}}) {
            if (
                defined($data->{$prefix . $field_name}) && $obj->{code} eq $data->{$prefix . $field_name}
                || $data->{$prefix . "code"} && ($obj->{code} =~ /^$temp_code/
                    || $code_part =~ /^0+$/ && $obj->{code} eq '')
               )
            {
                $last_code           = $obj->{code} if $obj->{code};
                $current             = $obj;
                $current->{selected} = 1;
                $flag_selected       = 1;
                last;
            }
        }
        if (!$flag_selected && @{$current->{children}})
        { #если на этом уровне ничего не было выбрано, выбираем первое попавшееся
            my $obj = $current->{children}->[0];
            $last_code           = $obj->{code} if $obj->{code};
            $current             = $obj;
            $current->{selected} = 1;
        }
    }

    my $streets = $self->get_streets($last_code, $index);
    foreach (@$streets) {
        if (   $_->{code} eq $data->{$prefix . 'street'}
            || $data->{$prefix . 'code'} && $_->{code} eq substr($data->{$prefix . 'code'}, 0, 15))
        {
            $_->{selected} = 1;
            last;
        }
    }

    return ($towns, $streets);
}

sub populate_addr_selects2 {
    my ($self, $prefix, $data) = @_;

    my (@result, $parent_code, $towns, $last_code);
    foreach (qw(region district city town)) {
        $towns = $self->get_towns_by_code($parent_code) if $parent_code || $_ eq 'region';
        my $this_level = shift(@$towns);
        $parent_code = $self->_mark_selected_town($this_level, $data->{$prefix . $_});
        push(@result, $this_level);

        $last_code = $parent_code if $parent_code;
    }

    if ($data->{$prefix . 'street'}) {
        my ($street_name) =
          $self->{dbh}->selectrow_array('select concat(Name, " ", Socr) from STREET where SUBSTR(Code, 1, 15)=?',
            undef, $data->{$prefix . 'street'});
        push(@result, [$street_name]);
    } else {
        $last_code =~ s/\D//g;
        my ($street_count) = $self->{dbh}->selectrow_array("select count(*) from STREET where Code like '$last_code%'");
        push(@result, []) if $street_count > 0;
    }
    return \@result;
}

=head1 validate_address

Метод на входе получает данные адреса, а на выходе отдает hashref с
подкорректированными этим данными.

Использование:

    my $result = $app->kladr->validate_address(
        16, # номер дома
        "", # корпус
        "", # строение
        "", # квартира
        119034, # индекс
    );

это вернет hashref:

    {
        building     => "", # корпус
        code         => 770000000007095,
        construction => "", # строение
        flat         => "",
        gni          => 7704,
        home         => 16,
        postcode     => 119034,
    }

=cut

sub validate_address {
    my $self = shift;
    my ($street_code, @args) = @_;
    my $i = 0;
    for (@args) {
        next if $i == 3;    # skip flat
        if (defined $_) {
            $_ = _delspaces(_fixruen(_uc($_)));
        }
        $i++;
    }
    my ($number, $building, $construction, $flat, $entered_index) = map {$_ // ''} @args;

    return unless $street_code;
    return unless $number || $building || $construction;    # мой адрес не дом?

    # В $street_code обязательно должен быть либо город (ГГГ), либо населенный пункт (ППП), либо улица (УУУУ)
    # Иначе какой же это корректный адрес?
    $street_code =~ /^(\d{2})(\d{3})(\d{3})(\d{3})(\d{4})?/;
    my ($code_region, $code_district, $code_city, $code_town, $code_street) = map {$_ // 0} ($1, $2, $3, $4, $5);
    return unless (($code_city > 0) || ($code_town > 0) || ($code_street > 0));

    if (
        $street_code eq "77000002000"    # Зеленоград! Я люблю Зеленоград!
        && defined($number)
        && int($number) >
        100    # запихали номер корпуса в поле дом, глюпие зеленоградцы
       )
    {
        $building = $number;
        $number   = '';
    }

    $street_code .= "0" x (15 - length($street_code));

    my %address = (
        code         => $street_code,
        home         => $number,
        building     => $building,
        construction => $construction,
        flat         => $flat
    );

    # во-первых проверим, не имеет ли наша улица конкертного индекса и кода ИФНС
    my $row = $self->_get_all("select `index`, gninmb gni from STREET where code=?", $street_code . "00")->[0];
    $address{gni}      = $row->{gni}   if $row->{gni};
    $address{postcode} = $row->{index} if $row->{index};
    return \%address if $row->{index} && $row->{gni};    # ура, повезло!

    my $data = $self->_get_all(
        q{
        select CONCAT(Name, Korp) compact, `index`, gninmb gni, code from DOMA where code like ?
        union
        select CONCAT(Name, Korp) compact, `index`, gninmb gni, code from DOMA_ex where code like ?
    }, (substr($street_code, 0, 15) . '%') x 2
    );

    my $have_homes  = 0;
    my $best_result = 0;
    #while (my ($compact, $index, $gni, $code) = $sth->fetchrow_array) {
    foreach my $row (@$data) {
        my $result = $self->_address_fits_compact($row->{compact}, $number, $building, $construction);
        if ($result > $best_result) {
            $best_result = $result;
            $address{code} =
              substr($row->{code}, 0, 15)
              ; # похоже коды домов просто беззвучно меняются. иначе объяснить наличие в базе "немного неправильных" кодов не могу. а значит и хранить их не хочу.
            $address{postcode} = $row->{index};
            $address{gni}      = $row->{gni};
        }
        $have_homes = 1;
    }

    return \%address if $best_result;

    $row =
      $self->_get_all("select `index`, gninmb gni from KLADR where code=?", substr($street_code, 0, 11) . "00")->[0];

    $address{gni}      = $row->{gni}   if $row->{gni};
    $address{postcode} = $row->{index} if $row->{index};

    $address{postcode} = $entered_index . "0" x (6 - length($entered_index)) unless $address{postcode};

    return \%address;
}

sub _address_fits_compact {
    my $self = shift;

    my ($compact, $number, $building, $construction) = @_;
    my @ranges = split(/,/, _delspaces(_fixruen(_uc($compact))));
    my $max_result = 0;
    my @number_parts;
    foreach my $range (@ranges) {
        my $result = 0;
        if ($number) {
            #поробуем разфигачить номер дома на буквы и дроби
            $number =~ /^(\D*\d+)([А-Яа-я]+)?([\/\-]\d+)?([А-Яа-я]+)?$/i;
            @number_parts = ($1, $2, $3, $4);

            my $digits = $1;
            if ($digits !~ /\D/) {
                if ($range eq 'Ч') {    # чётный
                    $result = 1 if _even($digits);
                } elsif ($range eq 'Н') {    # нечётный
                    $result = 1 if _odd($digits);
                } elsif ($range =~ /Ч\((\d+)-(\d+)\)/) {    # чётный интервал
                    $result = 2 if _even($digits) && _range($digits, $1, $2);
                } elsif ($range =~ /Н\((\d+)-(\d+)\)/) {    # нечётный интервал
                    $result = 2 if _odd($digits) && _range($digits, $1, $2);
                } elsif ($range =~ /^(\d+)-(\d+)/) {         # простой интервал
                    $result = 3 if _range($digits, $1, $2);
                }
            }
        }

        push(@number_parts, "К" . $building)         if $building;
        push(@number_parts, "СТР" . $construction) if $construction;
        my ($accumulator, $i);
        foreach (@number_parts) {
            $i++;
            next unless $_;
            $accumulator .= $_;
            $result = 3 + $i if $range eq $accumulator;
        }

        $max_result = $result if $result > $max_result;
    }

    return $max_result;
}

sub _create_list {
    my ($self, $hash, $lvl) = @_;
    my @list;
    foreach my $code (keys %$hash) {
        my $subhash = $hash->{$code};
        push(@list, $self->_create_tree($code, $subhash, $lvl + 1));
    }
    @list = sort {$a->{name} cmp $b->{name}} @list;
    return \@list;
}

sub _create_tree {
    my ($self, $code, $hash, $lvl) = @_;
    my %node = (
        name => _def($self->{code_names}->{$code}),
        code => $code
    );
    $node{children} = $self->_create_list($hash, $lvl) if $lvl < 4;
    return \%node;
}

sub _def($) {defined $_[0] ? $_[0] : ""}
sub _delspaces($) {my $str = shift; $str =~ s/\s+//g; $str}
sub _even($) {!_odd($_[0])}

sub _fix_index {
    my ($self, $index) = @_;
    $index =~ s/0{1,2}$//;
    return unless $index =~ /^\d{3,6}$/;
    $index .= '_' x (6 - length $index);
}
sub _fixruen($) {my $str = shift; $str =~ tr/MOCKBAPEHT/МОСКВАРЕНТ/; $str}

sub _level_by_code {
    my $self = shift;
    my $code = substr(shift, 0, 11);
    return 0 if substr($code, 2, 9) eq '000000000';
    return 1 if substr($code, 5, 6) eq '000000';
    return 2 if substr($code, 8, 3) eq '000';
    return 3;
}

sub _level_name_by_code {
    my $self  = shift;
    my @names = qw{region district city town};
    return $names[$self->_level_by_code(shift)];
}

sub _mark_selected_town {
    my ($self, $arr, $code) = @_;
    foreach (@$arr) {
        if ($_->[0] eq $code) {
            $_->[2] = 1;
            return $code;
        }
    }
    return
}

sub _odd($) {$_[0] % 2}
sub _range($$$) {$_[0] >= $_[1] && $_[0] <= $_[2]}

sub _split_code {
    my $self = shift;
    my $code = shift;

    my @result;
    push(@result, substr($code, 0, $_, "")) for (2, 3, 3, 3, 4, 4);
    return @result;
}

sub _uc($) {
    my $str = shift;
    $str =~
tr/йцукенгшщзхъфывапролджэячсмитьбюё/ЙЦУКЕНГШЩЗХЪФЫВАПРОЛДЖЭЯЧСМИТЬБЮЁ/;
    $str
}

sub _zeropad($$) {$_[1] . "0" x ($_[0] - length $_[1])}

1;
