package Application::Model::MDSAvatars;

use qbit;

use base
  qw(Application::Model::DBManager::Base RestApi::DBModel Application::Model::DAC Application::Model::ValidatableMixin);

consume qw(
  Application::Model::Role::Has::AvailableFields
  );

use MIME::Base64 qw(decode_base64);

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

my $LENGTH_500_KB           = 5 * 100 * 1024;
my $USER_UPLOAD_LIMIT       = 12;
my $NAMESPACE_PARTNER_SIZES = {
    'big'         => 1,
    'card'        => 1,
    'card_retina' => 1,
    'list'        => 1,
    'list_retina' => 1,
    'orig'        => 1,
};
# number of hours we consider an image usable,
# should be less than $REMOVE_HOURS_OLD
my $KEEP_HOURS = 1;
# number of hours MDSAvatarsCleanup cron wont touch an image
my $REMOVE_HOURS_OLD = 2;

sub accessor         {'mds_avatars'}
sub db_table_name    {'mds_avatars'}
sub get_product_name {'mds_avatars'}

sub get_structure_model_accessors {
    return {
        all_pages                 => 'Application::Model::AllPages',
        assistants                => 'Application::Model::Assistants',
        api_media_storage_avatars => 'QBit::Application::Model::API::Yandex::MediaStorageAvatars',
        partner_db                => 'Application::Model::PartnerDB',
    };
}

sub get_structure_rights_to_register {
    return [
        {
            name        => 'mds_avatars',
            description => d_gettext('Right to manage mds_avatars'),
            rights      => {
                mds_avatars_add       => d_gettext('Right to add records to mds_avatars'),
                mds_avatars_add_other => d_gettext('Right to add records to mds_avatars for another user'),
                mds_avatars_view_all  => d_gettext('Right to view all records in mds_avatars'),
            }
        }
    ];
}

sub get_structure_model_fields {
    return {
        public_id => {
            db_expr => 'id',
            db      => TRUE,
            default => TRUE,
            label   => d_gettext('Public ID'),
            type    => 'number',
            api     => 1,
        },
        id => {
            default => TRUE,
            db      => TRUE,
            pk      => TRUE,
            label   => d_gettext('ID'),
            type    => 'number',
            api     => 1,
        },
        user_id => {
            db         => TRUE,
            label      => d_gettext('User ID'),
            type       => 'number',
            api        => 1,
            need_check => {
                type     => 'int_un',
                optional => TRUE,
                check    => sub {
                    my ($qv, $user_id) = @_;

                    my $user = $qv->app->app->users->get($user_id);
                    unless ($user) {
                        throw Exception::Validator::Fields gettext("The following users do not exist: %s", $user_id);
                    }
                  }
            },
        },
        group_id => {
            default => TRUE,
            db      => TRUE,
            label   => d_gettext('Group ID'),
            type    => 'number',
            api     => 1,
        },
        imagename => {
            default => TRUE,
            db      => TRUE,
            label   => d_gettext('Image name'),
            type    => 'string',
            api     => 1,
        },
        md5 => {
            default => TRUE,
            db      => TRUE,
            label   => d_gettext('MD5'),
            type    => 'string',
            api     => 1,
        },
        update_dt => {
            default => TRUE,
            db      => TRUE,
            label   => d_gettext('Update datetime'),
            type    => 'string',
            api     => 1,
        },
        links => {
            default => TRUE,
            db      => TRUE,
            label   => d_gettext('Links'),
            type    => 'complex',
            get     => sub {
                my ($self, $obj) = @_;
                $obj->{links} ? from_json($obj->{links}) : {};
            },
            api => 1,
        },
        block_cnt => {
            db    => TRUE,
            label => d_gettext('Count of blocks using this image'),
            type  => 'number',
        }
    };
}

sub get_structure_model_filter {
    return {
        db_accessor => 'partner_db',
        fields      => {
            id        => {type => 'number',},
            user_id   => {type => 'number',},
            md5       => {type => 'text',},
            update_dt => {type => 'date',},
            block_cnt => {
                db   => TRUE,
                type => 'number',
            }
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    my @f;
    if ($self->check_short_rights('add_other')) {
        push @f, {name => 'user_id', label => gettext('User ID')};
    }

    return \@f;
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my %fields = map {$_ => TRUE} keys(%{$self->get_model_fields});

    return \%fields;
}

sub get_available_fields_depends {[qw(id)]}

sub get_add_fields {
    my ($self) = @_;

    my $res = {};
    foreach (qw(img_content img_link page_id)) {
        $res->{$_} = TRUE;
    }

    return $res;
}

sub get_editable_fields {return {};}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied') unless $self->check_short_rights('add');

    $self->check_bad_fields_in_add(\%opts);

    my $page_id = $opts{page_id};

    my $page_all = $self->all_pages->get_all(fields => ['model'], filter => {page_id => $page_id})->[0];
    throw Exception::Validation::BadArguments gettext('Page not found') unless $page_all;
    my $model_accessor = $page_all->{model};
    my $page_model = $self->app->$model_accessor->get_all(fields => ['owner_id'], filter => {page_id => $page_id})->[0];
    throw Exception::Validation::BadArguments gettext('No access to this page') unless $page_model;

    $self->hook_fields_validation({%opts, (user_id => $page_model->{owner_id})});

    my $mds_response;
    if (exists $opts{img_link}) {

        get_url_or_throw($opts{img_link});

        $mds_response = $self->api_media_storage_avatars->upload_link($opts{img_link});
    } elsif (exists $opts{img_content}) {

        my $content = decode_base64($opts{img_content});
        throw Exception::Validation::BadArguments gettext("Can't decode base64 string") unless length($content);
        throw Exception::Validation::BadArguments gettext('File is too big') if $LENGTH_500_KB < length($content);

        $mds_response = $self->api_media_storage_avatars->upload_file($content);
    }

    throw Exception::Validation::BadArguments gettext('Can not save image') unless $mds_response;

    my $record_added = $self->_save_to_db($page_model->{owner_id}, $mds_response);
    return $record_added->{id};
}

sub query_filter {
    my ($self, $filter) = @_;

    unless ($self->check_short_rights('view_all')) {
        $filter = $self->limit_filter_assistant($filter);
    }

    return $filter;
}

sub api_available_actions {
    return qw();
}

sub api_can_add {TRUE}

sub check_bad_fields_in_add {
    my ($self, $data, $fields) = @_;

    $fields //= $self->get_add_fields();

    my @bad_fields = grep {!$fields->{$_}} keys(%$data);

    if (@bad_fields) {
        throw Exception::Validation::BadArguments gettext("You can't use following fields in add: %s",
            join(', ', @bad_fields));
    }

    return TRUE;
}

sub get_orphan_records_older_than {
    my ($self, $n_hours_old) = @_;

    $n_hours_old //= 1;

    my $dt_n_hours_ago =
      date_sub(curdate(oformat => 'db_time'), hour => $n_hours_old, iformat => 'db_time', oformat => 'db_time');
    return $self->partner_db_table()->get_all(
        fields => [qw/id group_id imagename/],
        filter => ['AND' => [['update_dt', '<', \$dt_n_hours_ago], ['block_cnt', '=', \0],]]
    );
}

sub hook_fields_validation {
    my ($self, $opts) = @_;

    $self->app->validator->check(
        data  => $opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(),
    );

    my $img_source_one_of = {img_link => 1, img_content => 1};
    if (1 < scalar grep {exists $img_source_one_of->{$_}} keys %$opts) {
        throw Exception::Validation::BadArguments gettext('Expected one key from: %s',
            join(', ', sort keys %{$img_source_one_of}));
    }

    throw Exception::Denied gettext('Too many images. Please try again later.')
      if $self->_is_uploads_limit_exceeded_for_user($opts->{user_id});
}

sub hours_keep {
    return $KEEP_HOURS;
}

sub hours_remove {
    return $REMOVE_HOURS_OLD;
}

sub _is_uploads_limit_exceeded_for_user {
    my ($self, $user_id) = @_;

    my $orphan_upload_count = @{
        $self->get_all(
            fields => ['id'],
            filter => {
                user_id   => $user_id,
                block_cnt => 0
            }
        )
      };

    return $USER_UPLOAD_LIMIT <= $orphan_upload_count;
}

sub _save_to_db {
    my ($self, $user_id, $uploaded_img_info) = @_;

    my $dt = curdate(oformat => 'db_time');

    my $links = $self->api_media_storage_avatars->get_property_sizes($uploaded_img_info);

    my $record = {
        user_id   => $user_id,
        group_id  => $self->api_media_storage_avatars->get_property_groupid($uploaded_img_info),
        imagename => $self->api_media_storage_avatars->get_property_imagename($uploaded_img_info),
        md5       => $self->api_media_storage_avatars->get_property_md5($uploaded_img_info),
        update_dt => $dt,
        links     => to_json($links),
    };

    my %r;

    my $user_image = $self->get_all(filter => {map {$_ => $record->{$_}} qw(user_id md5)});
    if (@$user_image) {
        $self->partner_db_table()->edit($user_image->[0]->{id}, {update_dt => $dt,});
        $r{id} = $user_image->[0]->{id};
    } else {
        $r{id} = $self->partner_db->mds_avatars->add($record);
    }

    $r{links} = {
        map {
            $links->{$_}->{link} = $self->api_media_storage_avatars->get_public_link($uploaded_img_info, $_);
            delete $links->{$_}->{path};
            $_ => $links->{$_};
          } keys %$links
    };
    return \%r;
}

TRUE;
