package Application::Model::MailNotification;

use qbit;

use base qw(Application::Model::Common Application::Model::ValidatableMixin);

consume qw(
  Application::Model::Role::Has::CreateDate
  );

use Exception::TooManyRequests;
use Exception::Validator::Fields;

use PiConstants qw(
  $COMMON_OFFER_STATISTICS
  $CONTRACT_OFFER_ERROR_MAIL
  $DOCS_PROJECT_MAIL
  $MAIL_NOTIFICATION_TYPES
  $MYSQL_MIN_DATETIME
  $PARTNER2_COMMON_OFFER_ERRORS
  $PARTNER2_CRON_MAIL
  $PARTNERS_INTERNAL_ACCESS_REQUESTS_MAIL
  );

my $TIMEOUT = 60;

our %TYPES = (
    $MAIL_NOTIFICATION_TYPES->{BASE} => {
        check_alive => 1,
        name        => 'base',
        values      => {},
        vars        => {plain_text_wrapper => 'plain_text_wrapper',},
    },
    $MAIL_NOTIFICATION_TYPES->{BLOCK_DOMAIN} => {
        check_alive     => 1,
        name            => 'block_domain',
        subject         => 'Ваш домен в Рекламной сети Яндекса заблокирован',
        vars            => {name_site => 'domain', accessor => 'accessor'},
        notification_id => 5,
        transport       => 'sender',
        templates       => {
            en  => '',
            ru  => 'GSTT6SA4-SID',
            def => 'ru',
        }
    },
    $MAIL_NOTIFICATION_TYPES->{BLOCK_PAGE} => {
        check_alive => 1,
        name        => 'block_page',
        subject => 'Ваша площадка в Рекламной сети Яндекса заблокирована',
        vars    => {name_page => 'caption', page_id => 'page_id', accessor => 'accessor'},
        notification_id => 4,
        transport       => 'sender',
        templates       => {
            en  => '',
            ru  => 'OYIQ6SA4-JWY',
            def => 'ru',
        }
    },
    $MAIL_NOTIFICATION_TYPES->{BLOCK_LOGIN} => {
        name      => 'block_login',
        subject   => 'Ваш аккаунт в Рекламной сети Яндекса заблокирован',
        vars      => {login => 'login',},
        transport => 'sender',
        templates => {
            en  => '',
            ru  => '08WC6SA4-0HH',
            def => 'ru',
        }
    },
    $MAIL_NOTIFICATION_TYPES->{APPROVE_PAGE} => {
        check_alive  => 1,
        name         => 'approve_page',
        pre_cooldown => 604_800,                                                    # 1 week
        preparing    => \&_is_it_need_to_keep_sending_message_while_approve_page,
        subject         => 'Начните получать доход от рекламы на вашем сайте',
        vars            => {name_page => 'caption', page_id => 'page_id', accessor => 'accessor'},
        notification_id => 7,
        transport       => 'sender',
        templates       => {
            en  => '',
            ru  => 'GGVR3SA4-M5E1',
            def => 'ru',
        }
    },
    $MAIL_NOTIFICATION_TYPES->{MONEY} => {
        check_alive => 1,
        cooldown    => 1_209_600,                                                     # 2 weeks
        name        => 'money',
        preparing   => \&_is_it_need_to_keep_sending_message_while_money_threshold,
        transport   => 'sender',
        templates   => {
            en  => 'KVMX7Y94-G381',
            ru  => '4FZ45Y94-9KM1',
            def => 'ru',
        },
        notification_id => 8,
    },
    $MAIL_NOTIFICATION_TYPES->{COMMON_OFFER_STATISTICS} => {
        name    => 'common_offer_statistics',
        subject => 'Статистика по единой оферте',
        to      => [$CONTRACT_OFFER_ERROR_MAIL, $PARTNER2_COMMON_OFFER_ERRORS],
        vars    => {map {$_ => $_} @{$COMMON_OFFER_STATISTICS}},
    },
    $MAIL_NOTIFICATION_TYPES->{ADFOX_PAID_OFFER_REQUEST} => {
        name => 'adfox_paid_offer_request',
        subject =>
          'РСЯ: заявка на создание договора оферты по таргетированию',
        to   => [$DOCS_PROJECT_MAIL, $PARTNER2_COMMON_OFFER_ERRORS],
        vars => {
            map {$_ => $_}
              qw(
              af_client_id
              af_contract_ext_id
              af_contract_id
              af_login
              af_user_id
              pi_client_id
              pi_contract_ext_id
              pi_contract_id
              pi_login
              pi_user_id
              )
        },
    },
    $MAIL_NOTIFICATION_TYPES->{DELETE_PAGE} => {
        check_alive => 1,
        name        => 'delete_page',
        subject =>
          'Ваша площадка в Рекламной сети Яндекса заархивирована',
        vars            => {caption => 'caption', page_id => 'page_id', accessor => 'accessor'},
        notification_id => 9,
    },
    $MAIL_NOTIFICATION_TYPES->{STOP_PAGE} => {
        check_alive => 1,
        name        => 'stop_page',
        subject     => 'Ваша площадка в Рекламной сети Яндекса остановлена',
        vars        => {caption => 'caption', page_id => 'page_id', accessor => 'accessor'},
        notification_id => 10,
        transport       => 'sender',
        templates       => {
            en  => '',
            ru  => 'W6E47SA4-DYI',
            def => 'ru'
        },
    },
    $MAIL_NOTIFICATION_TYPES->{EMAIL_DOCS_ABOUT_ORDERS} => {
        name    => 'email_docs_about_orders',
        subject => 'Просьба выставить счета по договорам',
        to      => [$DOCS_PROJECT_MAIL],
        vars    => {map {$_ => $_} qw(first_day last_day)},
    },
    $MAIL_NOTIFICATION_TYPES->{QUEUE_TASK} => {
        check_alive => 1,
        name        => 'queue_task',
        values      => {},
        vars        => {plain_text_wrapper => 'plain_text_wrapper',},
    },
    $MAIL_NOTIFICATION_TYPES->{SENDER_BASE} => {transport => 'sender'},

    $MAIL_NOTIFICATION_TYPES->{APPROVED} => {
        name      => 'letter_after_approve',
        vars      => {full_name => 'full_name', resource => 'resource'},
        transport => 'sender',
        templates => {
            en  => '',
            ru  => '0MHYH064-V32',
            def => 'ru'
        },
        notificaiton_id => 2
    },
    $MAIL_NOTIFICATION_TYPES->{FIRST_APPROVED} => {
        name      => 'letter_after_first_approve',
        vars      => {resource => 'resource'},
        transport => 'sender',
        templates => {
            en  => '',
            ru  => 'KTG9UR54-UDH',
            def => 'ru'
        },
        notificaiton_id => 1,
    },
    $MAIL_NOTIFICATION_TYPES->{MOBILE_MEDIATION_FIRST_APP} => {
        name      => 'mobile_mediation_first_app',
        vars      => {val1 => 'caption', user_id => 'user_id'},
        transport => 'sender',
        templates => {
            en  => 'GKDQ2P44-OVZ1',
            ru  => '8KP19SA4-1JE',
            def => 'ru',
        }
    },
    $MAIL_NOTIFICATION_TYPES->{REJECTED_BY_MODERATION} => {
        name      => 'letter_after_reject',
        vars      => {full_name => 'full_name', reasons => 'reasons', resource => 'resource'},
        transport => 'sender',
        templates => {
            en  => '',
            ru  => '8G5E5SA4-T3M1',
            def => 'ru'
        },
    },
    $MAIL_NOTIFICATION_TYPES->{ACCESS_GRANTED} => {
        name      => 'access_granted',
        vars      => {login => 'login'},
        transport => 'sender',
        templates => {
            en  => '',
            ru  => '8K7B7SA4-QIP',
            def => 'ru'
        },
    },
    $MAIL_NOTIFICATION_TYPES->{NEW_PARTNER_SITE} => {
        vars      => {map {($_ => $_)} qw/login/},
        transport => 'sender',
        templates => {
            en  => 'WSABTID4-JHF1',
            ru  => 'WME8TID4-F8X',
            def => 'ru',
        },
    },
    $MAIL_NOTIFICATION_TYPES->{NEW_PARTNER_MOBILE_APP} => {
        vars      => {map {($_ => $_)} qw/login/},
        transport => 'sender',
        templates => {
            en  => 'WMCFTID4-MB9',
            ru  => 'S9RCTID4-48G1',
            def => 'ru',
        },
    },
    $MAIL_NOTIFICATION_TYPES->{NEW_PARTNER_SITE_AND_APP} => {
        vars      => {map {($_ => $_)} qw/login/},
        transport => 'sender',
        templates => {
            en  => 'OQOHTID4-HHJ1',
            ru  => 'GSQGTID4-FTC',
            def => 'ru',
        },
    },
    $MAIL_NOTIFICATION_TYPES->{FIRST_APPROVED_SITE} => {
        vars      => {map {($_ => $_)} qw/resource/},
        transport => 'sender',
        templates => {
            en  => 'SN8KZJD4-T6P1',
            ru  => 'WOMITID4-ZK32',
            def => 'ru',
        },
    },
    $MAIL_NOTIFICATION_TYPES->{FIRST_APPROVED_MOBILE_APP} => {
        vars      => {map {($_ => $_)} qw/resource/},
        transport => 'sender',
        templates => {
            en  => '8EPMZJD4-14E1',
            ru  => 'CX8LZJD4-9IS',
            def => 'ru',
        },
    },
    $MAIL_NOTIFICATION_TYPES->{APPROVED_SITE} => {
        vars      => {map {($_ => $_)} qw/resource/},
        transport => 'sender',
        templates => {
            en  => 'SFHOZJD4-HBD1',
            ru  => 'WSLNZJD4-SL71',
            def => 'ru',
        },
    },
    $MAIL_NOTIFICATION_TYPES->{APPROVED_MOBILE_APP} => {
        vars      => {map {($_ => $_)} qw/resource/},
        transport => 'sender',
        templates => {
            en  => 'KDERZJD4-AN91',
            ru  => '4JBPZJD4-TMA',
            def => 'ru',
        },
    },
    $MAIL_NOTIFICATION_TYPES->{REJECTED_SITE} => {
        vars      => {map {($_ => $_)} qw/resource reasons/},
        transport => 'sender',
        templates => {
            en  => 'OIRUZJD4-BYB',
            ru  => 'KP8QZJD4-4R42',
            def => 'ru',
        },
    },
    $MAIL_NOTIFICATION_TYPES->{REJECTED_MOBILE_APP} => {
        vars      => {map {($_ => $_)} qw/resource reasons/},
        transport => 'sender',
        templates => {
            en  => 'CN5TZJD4-ZPN',
            ru  => '4NESZJD4-O7A1',
            def => 'ru',
        },
    },
);

sub accessor      {'mail_notification'}
sub db_table_name {'mail_notification'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        api_sender         => 'Application::Model::API::Yandex::Sender',
        mailer             => 'Application::Model::SendMail',
        partner_db         => 'Application::Model::PartnerDB',
        users              => 'Application::Model::Users',
        user_notifications => 'Application::Model::UserNotifications',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        actions => {
            depends_on => [qw(id multistate period)],
            get        => sub {
                $_[0]->model->get_actions($_[1]);
              }
        },
        date_x => {db => TRUE, need_check => {type => 'date', optional => TRUE, format => 'db_time'},},
        email  => {
            depends_on => [qw( user_id )],
            get        => sub {
                $_[0]->{'__USERS__'}{$_[1]->{'user_id'}}{'email'} // '';
            },
        },
        id         => {default => TRUE, db => TRUE, pk => TRUE,},
        multistate => {default => TRUE, db => TRUE,},
        opts   => {db => TRUE, need_check => {type => 'json', optional => TRUE,},},
        period => {db => TRUE, need_check => {type => 'date', optional => TRUE, format => 'sec',},},
        type    => {default => TRUE, db => TRUE, need_check => {type => 'int_un', in => [sort keys %TYPES],},},
        user_id => {
            default    => TRUE,
            db         => TRUE,
            need_check => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $user_id) = @_;

                    my $tmp_rights = $qv->app->app->add_tmp_rights(qw(users_view_all));
                    throw Exception::Validator::Fields gettext('User not found')
                      unless scalar @{$qv->app->users->get_all(fields => ['id'], filter => {id => $user_id,},)};

                    return TRUE;
                },
            },
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            id          => {type => 'number',     label => d_gettext('Job ID')},
            multistate  => {type => 'multistate', label => d_gettext('Status')},
            period      => {type => 'number',     label => d_gettext('Periodicity')},
            create_date => {type => 'date',       label => d_gettext('Creation date')},
            date_x      => {type => 'date',       label => d_gettext('Planned start date')},
            type        => {type => 'number',     label => d_gettext('Type')},
            user_id     => {type => 'number',     label => d_gettext('User ID')},
        },
    };
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => 'New',
        multistates => [
            [preparing   => d_gettext('Preparing to send')],
            [ready       => d_gettext('Ready to send')],
            [sent        => d_gettext('Message was sent')],
            [stopped     => d_gettext('Job stopped')],
            [periodicity => d_gettext('Periodical job')],
        ],
        actions => {
            add         => d_gettext('Prepare'),
            prepare     => d_gettext('Ready'),
            send        => d_gettext('Send'),
            stop        => d_gettext('Stop'),
            periodicity => d_gettext('Periodicity'),
        },
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['preparing'],
            },
            {
                action      => 'prepare',
                from        => 'preparing and not stopped',
                reset_flags => ['preparing'],
                set_flags   => ['ready'],
            },
            {
                action      => 'send',
                from        => 'ready and not stopped',
                reset_flags => ['ready'],
                set_flags   => ['sent'],
            },
            {
                action    => 'stop',
                from      => 'not stopped',
                set_flags => ['stopped'],
            },
            {
                action    => 'periodicity',
                from      => 'not stopped and not periodicity',
                set_flags => ['periodicity'],
            },
            {
                action      => 'prepare',
                from        => 'sent and periodicity and not stopped',
                reset_flags => ['sent'],
                set_flags   => ['ready'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    if ($fields->need('user_id')) {
        my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_all));
        my $users      = $self->users->get_all(
            fields => [qw( id email )],
            filter => {id => array_uniq(map {$_->{'user_id'}} @{$result})},
        );
        $fields->{'__USERS__'} = {map {$_->{'id'} => $_} @$users};
    }
}

sub get_add_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} qw(type user_id period date_x opts);

    return \%fields;
}

sub add {
    my ($self, %opts) = @_;

    my $allowed_fields = $self->get_add_fields();
    $self->check_bad_fields_in_add(\%opts, $allowed_fields);
    $self->_set_default_values(\%opts);

    $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => $allowed_fields),
    );

    my $id;
    $self->partner_db->transaction(
        sub {
            $id = $self->partner_db_table()->add(\%opts);
            $self->do_action($id, 'add');
            $self->do_action($id, 'periodicity') if defined $opts{period};
        }
    );

    return $id;
}

sub can_action_periodicity {
    my ($self, $obj) = @_;

    my $object = $self->_get_object_fields($obj, [qw(period)]);

    return defined $object->{period};
}

sub on_action_prepare {
    my ($self, $obj, %opts) = @_;

    my $object = $self->_get_object_fields($obj, [qw(type user_id opts)]);
    my $keep = 1;

    # Проверка, стоит ли продолжать рассылку, зависящая от типа задания
    if (my $preparing = $TYPES{$object->{type}}{preparing}) {
        $keep = $preparing->($self, $object);
    }

    # Проверка, жив ли пользователь (при необходимости)
    if ($keep) {
        my $opts        = from_json($object->{opts}) // {};
        my $check_alive = $opts->{check_alive}       // $TYPES{$object->{type}}{check_alive};
        if ($check_alive) {
            # Сейчас блокировка пользователя происходит снятием с него всех ролей
            # TODO Добавить вызов соответствующей проверки после реализации блокировки пользователей (PISUP-215)
            my $roles = $self->users->get($object->{user_id}, fields => ['roles'])->{roles};
            $keep = !!@$roles;
        }
    }

    $self->do_action($obj, 'stop') unless $keep;
}

sub _check_timeout {
    my ($self, $pk, $timeout, $timeouts) = @_;

    my $last_sent = $timeouts->{$pk} // $self->get_timeouts([$pk])->{$pk};

    throw Exception::TooManyRequests
      if ($last_sent gt date_sub(curdate(), second => $timeout, oformat => 'db_time',));

    return TRUE;
}

sub get_to {
    my ($self, $obj) = @_;

    my $data = from_json($obj->{opts}) // {};
    my $to = ($data->{to} // $TYPES{$obj->{type}}{to} // {user_id => $obj->{user_id}});

    return $to;
}

sub get_timeouts {
    my ($self, $recipients) = @_;

    my $data = $self->partner_db->mail_notification_timeout->get_all(
        fields => [qw(to dt)],
        filter => [to => IN => \$recipients],
    );

    my %timeouts = map {$_->{to} => $_->{dt}} @$data;
    $timeouts{$_} //= $MYSQL_MIN_DATETIME foreach (@$recipients);

    return \%timeouts;
}

sub on_action_send {
    my ($self, $obj, %opts) = @_;

    my $object = $self->_get_object_fields($obj, [qw(id type email period user_id opts)]);

    utf8::decode($object->{opts});
    my $data = from_json($object->{opts}) // {};

    my $curdate = curdate();
    my $to      = $self->get_to($object);
    my $to_pk   = to_json($to);

    my $timeouts = $opts{timeouts} // {};
    $self->_check_timeout($to_pk, ($TYPES{$object->{type}}{timeout} // $TIMEOUT), $timeouts);

    # Приоритет как для полей, так и для значений (в следующем абзаце):
    #   1) дополнительные параметры из задания
    #   2) значения из задания (только для значений)
    #   3) параметры из дефолтного конфига соответствующего типа
    #   4) дефолтное значение
    my %vars = (    # name_in_template => name_in_structure
        message_body => 'message_body',
        %{$TYPES{$object->{type}}{vars} // {}},
        %{$data->{vars} // {}},
    );

    my $cc  = ($data->{cc}  // $TYPES{$object->{type}}{cc});
    my $bcc = ($data->{bcc} // $TYPES{$object->{type}}{bcc});
    my $vars = {
        map {
            $_ => $data->{values}{$vars{$_}} // $object->{$vars{$_}} // $TYPES{$object->{type}}{values}{$vars{$_}}
              // '',
          } keys %vars
    };

    my $is_disabled = FALSE;

    if (defined $TYPES{$object->{type}}{transport} && $TYPES{$object->{type}}{transport} eq 'sender') {
        my $template =
             $data->{values}->{template}
          || $TYPES{$object->{type}}{templates}{$self->app->get_user_lang($obj->{user_id})}
          || $TYPES{$object->{type}}{templates}{$TYPES{$object->{type}}{templates}{'def'}};

        my $response = $self->api_sender->send(
            to       => $self->mailer->get_emails_hash($self->get_to($obj)),
            args     => $vars,
            template => $template,
        );
        $is_disabled = TRUE if $response->{result}{status} eq 'DISABLED';
    } else {
        $self->mailer->send(
            body => {
                content_type => 'text/html',
                type         => 'TT2',
                template =>
                  ($data->{template} // $self->accessor() . '/' . $TYPES{$object->{type}}{name} . '.html.tt2'),
                vars => $vars,
            },
            content_type => 'text/html',
            from         => ($data->{from} // $TYPES{$object->{type}}{from} // 'default'),
            subject => ($data->{subject} // $TYPES{$object->{type}}{subject} // gettext('Yandex Advertising Network')),
            to      => $to,
            ($cc  ? (cc  => $cc)  : ()),
            ($bcc ? (bcc => $bcc) : ()),
        );
    }

    $self->partner_db_table()
      ->edit($obj, {date_x => date_add($curdate, second => $object->{period}, oformat => 'db_time',),})
      if $object->{period};

    unless ($is_disabled) {
        $timeouts->{$to_pk} = trdate(norm => db_time => $curdate);
        $self->partner_db->mail_notification_timeout->add(
            {
                to => $to_pk,
                dt => $timeouts->{$to_pk},
            },
            duplicate_update => TRUE,
        );
    }

    if (my $notification_id = $TYPES{$object->{type}}{notification_id}) {
        my $tmp = $self->app->add_tmp_rights('do_user_notifications_add');
        $self->user_notifications->add(
            notification_id => $notification_id,
            user_id         => $object->{user_id},
            accessor        => $vars->{accessor},
            custom_data     => $vars,
        );
    }
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
    $self->multistates_graph($self->get_structure_multistates_graph());
}

sub add_common_offer_error {
    my ($self, %opts) = @_;

    $self->_check_obligatory_fields(\%opts, qw(to subject message));

    $self->add(
        type    => $MAIL_NOTIFICATION_TYPES->{BASE},
        user_id => 0,
        opts    => {
            check_alive => FALSE,
            subject     => $opts{subject},
            to          => $opts{to},
            values      => {
                message_body       => $opts{message},
                plain_text_wrapper => TRUE,
            },
        },
    );
}

sub add_email_docs_about_orders {
    my ($self, %opts) = @_;

    $self->_check_obligatory_fields(\%opts, qw(first_day last_day));

    $self->add(
        opts    => {values => \%opts,},
        user_id => 0,
        type    => $MAIL_NOTIFICATION_TYPES->{EMAIL_DOCS_ABOUT_ORDERS},
    );

    return TRUE;
}

sub add_common_offer_statistics {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, @{$COMMON_OFFER_STATISTICS});

    $self->add(
        opts => {
            values      => $obj,
            check_alive => FALSE,
        },
        user_id => 0,
        type    => $MAIL_NOTIFICATION_TYPES->{COMMON_OFFER_STATISTICS},
    );
}

sub _add_when_page_affected {
    my ($self, $type, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(page_id owner_id caption accessor));

    $self->add(
        opts => {values => {page_id => $obj->{page_id}, caption => $obj->{caption}, accessor => $obj->{accessor}}},
        type => $type,
        user_id => $obj->{owner_id},
    );
}

sub add_when_approve_page {
    my ($self, $obj) = @_;

    $self->_add_when_page_affected($MAIL_NOTIFICATION_TYPES->{APPROVE_PAGE}, $obj);
}

sub add_when_delete_page {
    my ($self, $obj) = @_;

    $self->_add_when_page_affected($MAIL_NOTIFICATION_TYPES->{DELETE_PAGE}, $obj);
}

sub add_when_stop_page {
    my ($self, $obj) = @_;

    $self->_add_when_page_affected($MAIL_NOTIFICATION_TYPES->{STOP_PAGE}, $obj);
}

sub add_when_blocking_page {
    my ($self, $obj) = @_;

    $self->_add_when_page_affected($MAIL_NOTIFICATION_TYPES->{BLOCK_PAGE}, $obj);
}

sub add_when_blocking_login {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(id login));

    $self->add(
        opts    => {values => {login => $obj->{login}}},
        type    => $MAIL_NOTIFICATION_TYPES->{BLOCK_LOGIN},
        user_id => $obj->{id},
    );
}

sub add_when_blocking_site {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(id domain));

    my $tmp_rights = $self->app->add_tmp_rights(qw(owner_site_view_all));
    foreach my $owner (
        @{
            $self->app->owner_site->get_all(
                fields => ['user_id'],
                filter => {domain_id => $obj->{id}},
            )
        }
      )
    {
        $self->add(
            opts    => {values => {domain => $obj->{domain}, accessor => 'site'}},
            type    => $MAIL_NOTIFICATION_TYPES->{BLOCK_DOMAIN},
            user_id => $owner->{user_id},
        );
    }
}

sub add_when_user_is_near_of_money_threshold {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(id));

    $self->add(
        type    => $MAIL_NOTIFICATION_TYPES->{MONEY},
        user_id => $obj->{id},
    );
}

sub add_when_first_app_added_to_mediation {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(owner_id caption));

    $self->add(
        opts => {values => {user_id => $obj->{owner_id}, caption => $obj->{caption}}},
        type    => $MAIL_NOTIFICATION_TYPES->{MOBILE_MEDIATION_FIRST_APP},
        user_id => $obj->{owner_id},
    );
}

sub add_when_approved {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(user_id full_name has_approved resource));

    $self->add(
        type => $MAIL_NOTIFICATION_TYPES->{$obj->{'has_approved'} ? 'APPROVED' : 'FIRST_APPROVED'},
        user_id => $obj->{'user_id'},
        opts    => {
            bcc    => $PARTNER2_CRON_MAIL,
            values => {
                resource  => $obj->{'resource'},
                full_name => $obj->{'full_name'},
            },
        },
    );
}

sub add_when_resource_rejected_by_moderation {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(user_id full_name resource reasons));

    $self->add(
        type    => $MAIL_NOTIFICATION_TYPES->{REJECTED_BY_MODERATION},
        user_id => $obj->{'user_id'},
        opts    => {
            bcc    => $PARTNER2_CRON_MAIL,
            values => {
                full_name => $obj->{full_name},
                reasons   => $obj->{reasons},
                resource  => $obj->{resource},
            },
        },
    );
}

sub add_when_access_granted {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(user_id login));

    $self->add(
        type    => $MAIL_NOTIFICATION_TYPES->{ACCESS_GRANTED},
        user_id => $obj->{'user_id'},
        opts    => {
            cc     => $PARTNERS_INTERNAL_ACCESS_REQUESTS_MAIL,
            values => {login => $obj->{'login'},},
        },
    );
}

sub add_when_partner_registered_w_site {
    my ($self, $obj) = @_;

    $self->_add_when_new_partner($obj, $MAIL_NOTIFICATION_TYPES->{NEW_PARTNER_SITE});
}

sub add_when_partner_registered_w_mobile_app {
    my ($self, $obj) = @_;

    $self->_add_when_new_partner($obj, $MAIL_NOTIFICATION_TYPES->{NEW_PARTNER_MOBILE_APP});
}

sub add_when_partner_registered_w_site_and_app {
    my ($self, $obj) = @_;

    $self->_add_when_new_partner($obj, $MAIL_NOTIFICATION_TYPES->{NEW_PARTNER_SITE_AND_APP});
}

sub _add_when_new_partner {
    my ($self, $obj, $mail_type) = @_;

    $self->_check_obligatory_fields($obj, qw(user_id login));

    $self->add(
        type    => $mail_type,
        user_id => $obj->{user_id},
        opts    => {values => {login => $obj->{login}},},
    );

}

sub add_when_approved_site {
    my ($self, $obj) = @_;

    $self->_add_when_moderation_pass($obj, $MAIL_NOTIFICATION_TYPES->{APPROVED_SITE});
}

sub add_when_first_approved_site {
    my ($self, $obj) = @_;

    $self->_add_when_moderation_pass($obj, $MAIL_NOTIFICATION_TYPES->{FIRST_APPROVED_SITE});
}

sub add_when_approved_mobile_app {
    my ($self, $obj) = @_;

    $self->_add_when_moderation_pass($obj, $MAIL_NOTIFICATION_TYPES->{APPROVED_MOBILE_APP});
}

sub add_when_first_approved_mobile_app {
    my ($self, $obj) = @_;

    $self->_add_when_moderation_pass($obj, $MAIL_NOTIFICATION_TYPES->{FIRST_APPROVED_MOBILE_APP});
}

sub add_when_rejected_site {
    my ($self, $obj) = @_;

    $self->_add_when_moderation_fail($obj, $MAIL_NOTIFICATION_TYPES->{REJECTED_SITE});
}

sub add_when_rejected_mobile_app {
    my ($self, $obj) = @_;
    $self->_add_when_moderation_fail($obj, $MAIL_NOTIFICATION_TYPES->{REJECTED_MOBILE_APP});
}

sub _add_when_moderation_pass {
    my ($self, $obj, $mail_type) = @_;

    $self->_check_obligatory_fields($obj, qw(user_id resource));

    $self->add(
        type    => $mail_type,
        user_id => $obj->{user_id},
        opts    => {values => {resource => $obj->{resource}},},
    );
}

sub _add_when_moderation_fail {
    my ($self, $obj, $mail_type) = @_;

    $self->_check_obligatory_fields($obj, qw(user_id resource));

    $self->add(
        type    => $mail_type,
        user_id => $obj->{user_id},
        opts    => {
            values => {
                reasons => $obj->{reasons} // [],
                resource => $obj->{resource},
            },
        },
    );
}

sub _check_obligatory_fields {
    my ($self, $data, @fields) = @_;

    my @missed = grep {!exists($data->{$_})} @fields;

    throw Exception::Validation::BadArguments gettext('Missing fields: %s', join(', ', @missed))
      if @missed;
}

sub _is_it_need_to_keep_sending_message_while_approve_page {
    my ($self, $obj) = @_;

    my $opts = from_json($obj->{opts})->{values};
    $self->_check_obligatory_fields($opts, qw(page_id));
    my $page_id = $opts->{page_id};

    # Проверим, что page ещё в нужном статусе
    my $page = $self->partner_db->all_pages->get_all(
        fields => [qw(model multistate)],
        filter => {page_id => $page_id},
    )->[0];
    my $keep = !!$page;
    if ($keep) {
        my $model = $page->{model};
        $keep =
          in_array($page->{multistate},
            $self->app->$model->get_multistates_by_filter('not (blocked or deleted or rejected)'));
    }

    # Проверим, что на ней ещё нет блоков
    if ($keep) {
        my $blocks = $self->partner_db->all_blocks->get_all(
            fields => ['id'],
            filter => {page_id => $page_id},
        );
        $keep = !@$blocks;
    }

    return $keep;
}

sub _is_it_need_to_keep_sending_message_while_money_threshold {
    my ($self, $obj) = @_;

    $self->_check_obligatory_fields($obj, qw(user_id));
    my $user_id = $obj->{user_id};

    my $done = $self->partner_db->users->get_all(
        fields => ['id'],
        filter => [
            AND => [
                [
                    {ifnull => [{json_unquote => [{json_extract => ['opts', \'$.is_form_done']}]}, \'null']},
                    IN => \["1", "true"]
                ],
                [id => '=' => \$user_id],
            ]
        ],
    );

    return !@$done;
}

sub _set_default_values {
    my ($self, $opts) = @_;

    $opts->{create_date} = curdate(oformat => 'db_time');
    $opts->{type} //= 0;
    $opts->{period} //= $TYPES{$opts->{type}}{cooldown} if $TYPES{$opts->{type}}{cooldown};
    if ($TYPES{$opts->{type}}{pre_cooldown}) {
        $opts->{date_x} //= date_add(curdate(), second => $TYPES{$opts->{type}}{pre_cooldown}, oformat => 'db_time',);
    } else {
        $opts->{date_x} //= $opts->{create_date};
    }
    $opts->{opts} //= '{}';
    $opts->{opts} = to_json($opts->{opts}) if (ref $opts->{opts});
}

TRUE;
