package Application::Model::Moderation;

use qbit;

use base qw(Application::Model::Common);

our %dict_caller_to_product = (
    context_on_site_mirrors => 'site',
    search_on_site_mirrors  => 'site',
);

our %dict_caller_to_type = (
    context_on_site_mirrors => 'mirrors',
    search_on_site_mirrors  => 'mirrors',
);

our $PRODUCT_TYPES = {site => {label => d_gettext('Site'),},};

our $MODERATION_TYPES = {mirrors => {label => d_gettext('Mirrors moderation'),},};

sub accessor      {'moderation'}
sub db_table_name {'moderation_queue'}

__PACKAGE__->model_accessors(
    partner_db               => 'Application::Model::PartnerDB::Moderation',
    users                    => 'Application::Model::Users',
    context_on_site_campaign => 'Application::Model::Product::AN::ContextOnSite::Campaign',
    search_on_site_campaign  => 'Application::Model::Product::AN::SearchOnSite::Campaign',
    site                     => 'Application::Model::Product::AN::Site',
    context_on_site_mirrors  => 'Application::Model::Product::AN::ContextOnSite::Campaign::Mirrors',
    search_on_site_mirrors   => 'Application::Model::Product::AN::SearchOnSite::Campaign::Mirrors',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'Moderation',
            description => d_gettext('Right to moderate'),
            rights      => {
                moderation_add  => d_gettext('Right to edit moderations'),
                moderation_view => d_gettext('Right to view moderations'),
            },
        }
    ]
);

__PACKAGE__->model_fields(
    id => {default => TRUE, db => TRUE, pk => TRUE, type => 'number', label => d_gettext('ID')},
    create_date => {default => TRUE, db => TRUE, label => d_gettext('Date')},
    page_id     => {default => TRUE, db => TRUE, label => d_gettext('Page ID')},
    page_type   => {
        label      => d_gettext('Page Type'),
        depends_on => ['page_id', 'id'],
        get        => sub {
            my ($self, $obj) = @_;

            return ($self->{'__PAGE_TYPES__'}{$obj->{'page_id'}} // {});
          }
    },
    page_caption => {
        label      => d_gettext('Page Caption'),
        depends_on => ['page_id', 'id'],
        get        => sub {
            my ($self, $obj) = @_;

            return ($self->{'__PAGE_CAPTIONS__'}{$obj->{'page_id'}} // {});
          }
    },
    domain_id => {default => TRUE, db => TRUE, label => d_gettext('Domain ID')},
    site      => {
        label    => d_gettext('Site'),
        submodel => {
            domain_id => sub {
                shift->model->partner_db->site;
            },
            fields => [qw(id domain)],
        },
    },
    partner => {
        label      => d_gettext('Partner'),
        depends_on => ['page_id', 'id'],
        get        => sub {
            my ($self, $obj) = @_;

            return ($self->{'__PARTNERS__'}{$obj->{'page_id'}}{owner} // {});
          }
    },
    mirrors => {
        label      => d_gettext('Mirrors'),
        depends_on => ['id'],
        get        => sub {
            $_[0]->{'__MIRRORS__'}->{$_[1]->{'id'}} // {};
          }
    },
    context_pages_amount => {
        label      => d_gettext('Context pages amount'),
        depends_on => ['partner', 'id'],
        get        => sub {
            %{$_[1]->{'partner'}} ? $_[0]->{'__CONTEXT_PAGES_AMOUNT__'}->{$_[1]->{'partner'}->{'id'}} // 0 : 0;
          }
    },
    search_pages_amount => {
        label      => d_gettext('Search pages amount'),
        depends_on => ['partner', 'id'],
        get        => sub {
            %{$_[1]->{'partner'}} ? $_[0]->{'__SEARCH_PAGES_AMOUNT__'}->{$_[1]->{'partner'}->{'id'}} // 0 : 0;
          }
    },
    moderation_history => {
        label      => d_gettext('Moderation history'),
        depends_on => ['id'],
        get        => sub {
            $_[0]->{'__MODERATION_HISTORY__'}->{$_[1]->{'id'}} // [];
          }
    },
    product       => {default => TRUE, db => TRUE, label => d_gettext('Product')},
    product_label => {
        depends_on => [qw(product)],
        get        => sub {
            my $product = $_[1]->{'product'};
            return $PRODUCT_TYPES->{$product}{'label'}();
        },
    },
    type       => {default => TRUE, db => TRUE, label => d_gettext('Request Type')},
    type_label => {
        depends_on => [qw(type)],
        get        => sub {
            my $type = $_[1]->{'type'};
            return $MODERATION_TYPES->{$type}{'label'}();
        },
    },
    multistate => {db => TRUE, label => d_gettext('Status')},
    state_name => {
        label      => d_gettext('State'),
        depends_on => [qw(multistate)],
        get        => sub {
            my ($self, $obj) = ($_[0]->model, $_[1]);
            if ($self->check_multistate_flag($obj->{'multistate'}, 'done')) {
                return pgettext('Moderation state name', 'Done');
            } else {
                return pgettext('Moderation state name', 'New');
            }
        },
    },
    actions => {
        label      => d_gettext('Actions'),
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
        },
    },
    comment => {
        db    => TRUE,
        label => d_gettext('Comment'),
    },
    editable_fields => {
        label      => d_gettext('Editable fields'),
        depends_on => [qw(multistate)],
        get        => sub {
            $_[0]->model->get_editable_fields($_[1]);
          }
    },
    available_fields => {
        label => d_gettext('Available fields'),
        get   => sub {
            return $_[0]->model->get_available_fields();
          }
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        # без этого не работает do_action (Exception :: BadArguments: Неизвестное поле "id")
        id => {
            type  => 'number',
            label => d_gettext('ID'),
        },
        domain => {
            type           => 'subfilter',
            model_accessor => 'site',
            field          => 'domain_id',
            fk_field       => 'id',
            label          => d_gettext('Domain'),
        },
        product => {
            type   => 'dictionary',
            label  => d_gettext('Product'),
            values => sub {
                [
                    map {+{id => $_, label => $PRODUCT_TYPES->{$_}{'label'}()}}
                    sort keys(%{$PRODUCT_TYPES})
                ];
              }
        },
        type => {
            type   => 'dictionary',
            label  => d_gettext('Type'),
            values => sub {
                [
                    map {+{id => $_, label => $MODERATION_TYPES->{$_}{'label'}()}}
                    sort keys(%{$MODERATION_TYPES})
                ];
              }
        },
        page_id     => {type => 'number',     label => d_gettext('Page ID')},
        create_date => {type => 'period',     label => d_gettext('Create date')},
        multistate  => {type => 'multistate', label => d_gettext('Status')},
    }
);

__PACKAGE__->multistates_graph(
    empty_name  => 'New',
    multistates => [[new => d_pgettext('Moderation state', 'New')], [done => d_pgettext('Moderation state', 'Done')],],
    actions     => {
        'edit' => d_pgettext('Moderation action', 'edit'),
        'add'  => d_pgettext('Moderation action', 'add'),
    },
    multistate_actions => [
        {
            action    => 'add',
            from      => '__EMPTY__',
            set_flags => [qw(new)],
        },
        {
            action      => 'edit',
            from        => 'new',
            set_flags   => [qw(done)],
            reset_flags => [qw(new)],
        },
    ],
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => 'domain.domain', label => gettext('Domain')},
        {name => 'product',       label => gettext('Product')},
        {name => 'page_id',       label => gettext('Page ID')},
        {name => 'type',          label => gettext('Type')},
        {name => 'multistate',    label => gettext('Status')},
        {name => 'create_date',   label => gettext('Date')},
    ];
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my ($request_ids, $page_ids, $relative_models, $relative_models_h, $fields_map, $tmp_rights);
    my %need_fields =
      map {$_ => TRUE}
      grep {$fields->need($_)}
      qw(
      page_type
      partner
      context_pages_amount
      search_pages_amount
      mirrors
      page_caption
      );

    if (keys %need_fields) {
        $request_ids = array_uniq(map {$_->{'id'}      // ()} @$result);
        $page_ids    = array_uniq(map {$_->{'page_id'} // ()} @$result);

        $tmp_rights =
          $self->app->add_tmp_rights('context_on_site_campaign_view_all', 'search_on_site_campaign_view_all');

        $relative_models = $self->partner_db->query->select(
            table  => $self->partner_db->moderation_rel_site_mirrors,
            fields => [qw(qid id relative_model moderation_status)],
            filter => [qid => 'IN' => \$request_ids]
        )->get_all();

        foreach my $row (@$relative_models) {
            $relative_models_h->{$row->{qid}}{relative_model} //= $row->{relative_model};
            push @{$relative_models_h->{$row->{qid}}{mirrors}},
              {id => $row->{id}, moderation_status => $row->{moderation_status}};
        }

        # Собираем единую структуру для записей, т.к. информация разбросана по нескольким табличкам
        $fields_map = {};

        my ($fields_map_cur, $mirrors_model);
        foreach my $r (@$result) {
            $fields_map_cur = $fields_map->{$r->{id}} //= {};

            $fields_map_cur->{page_id} = $r->{page_id};

            $mirrors_model = $fields_map_cur->{relative_model} = $relative_models_h->{$r->{id}}{relative_model};

            $fields_map_cur->{campaign_model} = $self->$mirrors_model->page_model_name();

            $fields_map_cur->{mirrors} = $relative_models_h->{$r->{id}}{mirrors};
        }
    }

    # Получаем тип площадки для того, чтобы на фронте можно было собрать ссылку
    if ($fields->need('page_type')) {
        $fields->{__PAGE_TYPES__} =
          {map {$fields_map->{$_}->{page_id} => $fields_map->{$_}->{campaign_model}} keys %$fields_map};
    }

    # Получаем информацию о партнёре
    if (
        (
               $fields->need('partner')
            or $fields->need('context_pages_amount')
            or $fields->need('search_pages_amount')
            or $fields->need('page_caption')
        )
        and !$fields->{'__PARTNERS__'}
       )
    {
        my (@context_page_ids, @search_page_ids);

        foreach (keys %$fields_map) {
            push @context_page_ids, $fields_map->{$_}->{page_id}
              if $fields_map->{$_}->{campaign_model} eq 'context_on_site_campaign';
            push @search_page_ids, $fields_map->{$_}->{page_id}
              if $fields_map->{$_}->{campaign_model} eq 'search_on_site_campaign';
        }

        my @owner_ids = ();
        if (@context_page_ids) {
            foreach (
                @{
                    $self->context_on_site_campaign->get_all(
                        fields => [qw(page_id owner_id caption)],
                        filter => {page_id => $page_ids},
                    )
                }
              )
            {
                push(@owner_ids, $_->{'owner_id'});
                $fields->{'__PARTNERS__'}{$_->{'page_id'}} = $_;
            }
        }

        if (@search_page_ids) {
            foreach (
                @{
                    $self->search_on_site_campaign->get_all(
                        fields => [qw(page_id owner_id caption)],
                        filter => {page_id => $page_ids},
                    )
                }
              )
            {
                push(@owner_ids, $_->{'owner_id'});
                $fields->{'__PARTNERS__'}{$_->{'page_id'}} = $_;
            }
        }

        my %users = map {$_->{'id'} => $_} @{
            $self->users->get_all(
                fields => [qw(id login client_id business_unit name midname lastname)],
                filter => {id => \@owner_ids}
            )
          };

        foreach my $row (values(%{$fields->{'__PARTNERS__'}})) {
            $row->{'owner'} = $users{delete($row->{'owner_id'})};
        }
    }

    # Получаем названия площадок
    if ($fields->need('page_caption')) {
        $fields->{__PAGE_CAPTIONS__} =
          {map {$_ => $fields->{'__PARTNERS__'}{$_}{caption}} (keys %{$fields->{'__PARTNERS__'}})};
    }

    # Получаем количество тематических площадок
    if ($fields->need('context_pages_amount')) {
        $fields->{'__CONTEXT_PAGES_AMOUNT__'} = {};
        map {$fields->{'__CONTEXT_PAGES_AMOUNT__'}->{$_->{owner_id}}++} @{
            $self->context_on_site_campaign->get_all(
                fields => [qw(owner_id)],
                filter => {
                    owner_id =>
                      array_uniq(map {$fields->{'__PARTNERS__'}{$_}{owner}{id}} (keys %{$fields->{'__PARTNERS__'}})),
                    multistate => 'not deleted'
                },
            );
          };
    }

    # Получаем количество поисковых площадок
    if ($fields->need('search_pages_amount')) {
        $fields->{'__SEARCH_PAGES_AMOUNT__'} = {};
        map {$fields->{'__SEARCH_PAGES_AMOUNT__'}->{$_->{owner_id}}++} @{
            $self->search_on_site_campaign->get_all(
                fields => [qw(owner_id)],
                filter => {
                    owner_id =>
                      array_uniq(map {$fields->{'__PARTNERS__'}{$_}{owner}{id}} (keys %{$fields->{'__PARTNERS__'}})),
                    multistate => 'not deleted'
                },
            );
          };
    }

    # Получаем зеркала с их статусами модерации
    if ($fields->need('mirrors')) {
        $fields->{'__MIRRORS__'} = {};
        my (@context_mirror_ids, @search_mirror_ids, @context_mirrors, @search_mirrors);

        foreach my $request_id (keys %$fields_map) {
            if ($fields_map->{$request_id}{campaign_model} eq 'context_on_site_campaign') {
                push @context_mirror_ids, map {$_->{id}} @{$fields_map->{$request_id}{mirrors}};
            } elsif ($fields_map->{$request_id}{campaign_model} eq 'search_on_site_campaign') {
                push @search_mirror_ids, map {$_->{id}} @{$fields_map->{$request_id}{mirrors}};
            }
        }

        # Получаем связки id:domain для зеркал контекстных площадок
        @context_mirrors = @{
            $self->context_on_site_mirrors->get_all(
                fields => [qw(id domain auto)],
                filter => {id => \@context_mirror_ids}
            );
          } if (@context_mirror_ids);

        # Получаем связки id:domain для зеркал посиковых площадок
        @search_mirrors = @{
            $self->search_on_site_mirrors->get_all(
                fields => [qw(id domain auto)],
                filter => {id => \@search_mirror_ids}
            );
          } if (@search_mirror_ids);

        my $mirrors_by_id = {map {$_->{id} => $_} @context_mirrors, @search_mirrors};

        foreach my $id (sort keys %$fields_map) {
            $fields->{'__MIRRORS__'}{$id} //= [];

            foreach my $mirror (@{$fields_map->{$id}{mirrors}}) {
                my $mirror_data = $mirrors_by_id->{$mirror->{id}};
                push(
                    @{$fields->{'__MIRRORS__'}{$id}},
                    {
                        domain            => $mirror_data->{domain},
                        id                => $mirror->{id},
                        moderation_status => $mirror->{moderation_status},
                        auto              => $mirror_data->{auto},
                    }
                );
            }
        }
    }

    if ($fields->need('moderation_history')) {
        $fields->{'__MODERATION_HISTORY__'} = {};
        my $request_ids = {};

        $request_ids->{id} //= [];
        map {push @{$request_ids->{id}}, $_->{id}} @$result;

        my $result_logs = $self->_get_moderation_history($request_ids);

        foreach my $log (@$result_logs) {
            my $moderation_req_id = $log->{elem_id};
            if ($moderation_req_id) {
                $fields->{'__MODERATION_HISTORY__'}->{$moderation_req_id} //= [];
                push @{$fields->{'__MODERATION_HISTORY__'}->{$moderation_req_id}}, $log;
            }
        }
    }
}

sub get_available_fields {
    my ($self) = @_;

    my $model_fields = $self->get_model_fields;

    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_editable_fields {
    my ($self, $data) = @_;
    my $editable_fields = {
        mirrors => 1,
        comment => 1,
    };

    return {} if $self->check_multistate_flag($data->{multistate}, 'done');

    return $editable_fields;
}

sub on_action_edit {
    my ($self, $obj, %data) = @_;

    my $fields_hs = $self->get_editable_fields();
    %data = map(exists($data{$_}) ? ($_ => $data{$_}) : (), keys(%$fields_hs));

    my $request_id     = $obj->{id};
    my $page_id        = ($self->get($request_id, fields => ['page_id']))->{page_id};
    my $relative_model = @{
        $self->partner_db->query->select(
            table  => $self->partner_db->moderation_rel_site_mirrors,
            fields => [qw(relative_model)],
            filter => [qid => '=' => \$request_id]
        )->get_all();
      }[0]->{relative_model};
    my $campaign_model =
      ($relative_model =~ /context_on_site/) ? 'context_on_site_campaign' : 'search_on_site_campaign';

    if (defined($data{mirrors})) {
        my $mirrors = delete($data{mirrors});

        my @approved_mirrors = map {$_->{id}} (grep {$_->{moderation_status} eq 'approved'} @$mirrors);
        my @rejected_mirrors = map {$_->{id}} (grep {$_->{moderation_status} eq 'rejected'} @$mirrors);

        # Проставляем статусы зеркал в промежуточной таблице для истории
        $self->partner_db->moderation_rel_site_mirrors->edit(
            $self->partner_db->filter({qid => $request_id, id => \@approved_mirrors}),
            {moderation_status => 'approved'});
        $self->partner_db->moderation_rel_site_mirrors->edit(
            $self->partner_db->filter({qid => $request_id, id => \@rejected_mirrors}),
            {moderation_status => 'rejected'});

        # Проставляем статусы зеркал в конечной таблице с зеркалами
        map {$self->$relative_model->do_action($_, 'approve')} @approved_mirrors;
        map {$self->$relative_model->do_action($_, 'reject')} @rejected_mirrors;
    }

    if (defined($data{comment})) {
        $self->partner_db_table()->edit($obj, \%data);
    }

    my $tmp_rights = $self->app->add_tmp_rights(
        'context_on_site_campaign_edit',     'search_on_site_campaign_edit',
        'context_on_site_campaign_view_all', 'search_on_site_campaign_view_all'
    );

    return TRUE;
}

sub add_request {
    my ($self, $caller, $opts) = @_;
    my $campaign_id     = $opts->{campaign_id};
    my $product         = $dict_caller_to_product{$caller};
    my $moderation_type = $dict_caller_to_type{$caller};
    my $rel_table_name  = "moderation_rel_${product}_${moderation_type}";

    # context_on_site_campaign.domain_id закрыто правом, добавляем
    my $tmp_rights = $self->app->add_tmp_rights('context_on_site_campaign_view_field__domain_id',
        'search_on_site_campaign_view_field__domain_id');

    my $campaign_model = ($caller =~ /context_on_site/) ? 'context_on_site_campaign' : 'search_on_site_campaign';
    my $campaign = $self->$campaign_model->get($campaign_id, fields => ['page_id', 'domain_id']);

    my $moderation_req_id = $self->partner_db_table()->add(
        {
            create_date => curdate(oformat => 'db_time'),
            page_id     => $campaign->{page_id},
            domain_id   => $campaign->{domain_id},
            product     => $product,
            type        => $moderation_type,
        }
    );

    if (exists($opts->{moderation_elements})) {
        foreach my $element_id (@{$opts->{moderation_elements}}) {
            $self->partner_db->$rel_table_name->add(
                {
                    qid            => $moderation_req_id,
                    id             => $element_id,
                    relative_model => $caller,
                }
            );
        }
    }

    $self->do_action($moderation_req_id, 'add');
}

# Получаем историю модерации
sub _get_moderation_history {
    my ($self, $obj) = @_;

    my $logs =
      $self->get_action_log_entries($obj, explain_actions => TRUE, explain_multistates => TRUE, expand_opts => TRUE);

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_all));
    rows_expand($logs, 'user_id' => $self->users => 'user');

    return $logs;
}

TRUE;
