package Application::Model::ModerationReason;

use qbit;
use Utils::Logger qw(INFOF);

use base qw(
  Application::Model::Common
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  );

use PiConstants qw(%MODERATION_REASON_TIMEOUTS);

sub accessor      {'moderation_reason'}
sub db_table_name {'moderation_reason'}

sub get_product_name {
    gettext('moderation_reason');
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        partner_db => 'Application::Model::PartnerDB',
        users      => 'Application::Model::Users',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;
    return [
        {
            name        => 'moderation_reason',
            description => d_gettext('Rights to use moderation reason'),
            rights      => {
                moderation_reason_view_action_log  => d_gettext('Right to view %s', 'moderation_reason log'),
                moderation_reason_manager_txt_view => d_gettext('Right to view %s', 'moderation_reason manager'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    my @locales = keys %{$self->get_option('locales')};

    return {
        id       => {default => TRUE, db => TRUE, pk => TRUE, type => 'number', api => 1},
        group_id => {
            default => TRUE,
            db      => TRUE,
            type    => 'number',
            api     => 1,
        },
        multistate      => {default => TRUE, db => TRUE, type => 'number', api => 1},
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1
        },
        reason_txt => {
            depends_on => [qw(manager_txt partner_txt)],
            default    => TRUE,
            type       => 'string',
            api        => 1,
            get        => sub {
                return $_[1]->{$_[0]->{__REASON_TXT_FIELD_NAME__}};
            },
        },
        manager_txt => {
            i18n       => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                type   => 'hash',
                fields => {
                    map {
                        $_ => {optional => TRUE, len_max => 512, len_min => 1}
                      } @locales
                },
            },
        },
        partner_txt => {
            i18n       => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                type   => 'hash',
                fields => {
                    map {
                        $_ => {optional => TRUE, len_max => 512, len_min => 1}
                      } @locales
                },
            },
        },
        timeout => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                type => 'scalar',
                in   => [
                    sort {
                        $a <=> $b
                      } keys %MODERATION_REASON_TIMEOUTS
                ],
            },
        },
        timeout_txt => {
            type       => 'string',
            api        => 1,
            depends_on => ['timeout'],
            get        => sub {
                defined($_[1]->{'timeout'})
                  && $MODERATION_REASON_TIMEOUTS{$_[1]->{'timeout'}}
                  ? $MODERATION_REASON_TIMEOUTS{$_[1]->{'timeout'}}->() // ''
                  : '';
            },
        },
        owner_id => {
            db         => TRUE,
            label      => d_gettext('Owner id'),
            need_check => {type => 'int_un',},
            type       => 'number',
            api        => 1,
        },
        owner => {
            label      => d_gettext('Owner'),
            depends_on => ['owner_id'],
            get        => sub {
                $_[0]->{'__OWNERS__'}->{$_[1]->{'owner_id'}} // {};
            },
            type => 'users',
        },
        fields_depends => {
            get => sub {
                return $_[0]->model->get_fields_depends();
            },
            type => 'complex',
            api  => 1,
        },
        public_id => {
            db      => TRUE,
            db_expr => 'id',
            type    => 'string',
            api     => 1,
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;
    return {
        db_accessor => 'partner_db',
        fields      => {
            id          => {type => 'number',     label => d_gettext('ID')},
            partner_txt => {type => 'text',       label => d_gettext('Partner text')},
            multistate  => {type => 'multistate', label => d_gettext('Status')},
            manager_txt => {type => 'text',       label => d_gettext('Manager text')},
            timeout     => {
                type   => 'dictionary',
                label  => d_gettext('Timeout'),
                values => sub {
                    return [
                        map {+{id => $_, label => $MODERATION_REASON_TIMEOUTS{$_}->(),}}
                        sort {$a <=> $b} keys %MODERATION_REASON_TIMEOUTS
                    ];
                },
            },
            owner_id => {type => 'number', label => d_gettext('User ID')},
        },
    };
}

sub get_available_fields_depends {[]}

sub get_editable_fields_depends {[qw(id)]}

sub get_actions_depends {
    [qw(id multistate)];
}

# число (0 - active; 1 - deleted)
sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name    => d_gettext('Active'),
        multistates   => [[deleted => d_gettext('Deleted')],],
        right_actions => {
            delete  => d_gettext('Delete'),
            add     => d_gettext('Add'),
            edit    => d_gettext('Edit'),
            restore => d_gettext('Restore'),
        },
        right_group        => [moderation_reason_action => d_gettext('Right to manage moderation_reason')],
        right_name_prefix  => 'moderation_reason_action_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action => 'edit',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    if ($fields->need('reason_txt')) {
        # заполняем имя поля из которого брать текст причины
        $fields->{__REASON_TXT_FIELD_NAME__} =
          $self->check_rights('moderation_reason_manager_txt_view') ? 'manager_txt' : 'partner_txt';
    }

    if ($fields->need('owner')) {
        my $tmp_rights = $self->app->add_tmp_rights('users_view_all');
        $fields->{'__OWNERS__'} = {
            map {$_->{id} => $_->{login}} @{
                $self->app->users->get_all(
                    fields => ['id', 'login'],
                    filter => {id => array_uniq(map {$_->{'owner_id'} // ()} @$result)},
                )
              }
        };
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();
    $self->app->delete_field_by_rights(\%fields, {$accessor . '_view_field__%s' => [qw(partner_txt manager_txt)],});

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %res = map {$_ => TRUE} qw(
      id
      partner_txt
      manager_txt
      timeout
      );

    return \%res;
}

sub get_editable_fields {
    my ($self, $object) = @_;

    my %res = map {$_ => TRUE} qw(
      partner_txt
      manager_txt
      timeout
      );

    return \%res;
}

sub api_available_actions {
    return qw(delete edit restore);
}

sub api_can_add {TRUE}

sub api_can_edit {TRUE}

sub get_fields_depends {return {}}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result;

    return \%result;
}

1;
