package Application::Model::Notification;

use Exception::Form;
use Exception::Validator::Fields;

use PiConstants qw($NOTIFICATION_TYPE $NOTIFICATION_VIEW_TYPE $NOTIFICATION_ICON $NOTIFICATION_FIELD_LIMIT);
use qbit;
use Utils::Logger qw(INFOF);
use Utils::MonitoringUtils;

use base qw(
  Application::Model::Common
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  );

sub accessor      {'notification'}
sub db_table_name {'notification'}

sub get_product_name {
    gettext('notification');
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        rbac               => 'Application::Model::RBAC',
        partner_db         => 'Application::Model::PartnerDB',
        user_notifications => 'Application::Model::UserNotifications',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;
    return [
        {
            name        => 'notification',
            description => d_gettext('Rights to use notification'),
            rights      => {
                notification_view_action_log => d_gettext('Right to view %s', 'notification log'),
                notification_view            => d_gettext('Right to view %s', 'notifications'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    my @locales = keys %{$self->get_option('locales')};

    return {
        id              => {default => TRUE, db => TRUE, pk   => TRUE,     type => 'number'},
        multistate      => {default => TRUE, db => TRUE, type => 'number', api  => 1},
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1
        },
        type => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            need_check => {in => $NOTIFICATION_TYPE,},
            api        => 1
        },
        short_name => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            need_check => {
                len_max => 16,
                len_min => 1,
                regexp  => qr/\A[a-z_\d]{1,16}\z/,
            },
            api => 1
        },
        view_type => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            need_check => {in => $NOTIFICATION_VIEW_TYPE,},
            api        => 1
        },
        ttl => {
            default    => TRUE,
            db         => TRUE,
            type       => 'number',
            depends_on => ['type'],
            need_check => {
                check => sub {
                    my ($qv, $value) = @_;

                    my $limit = $NOTIFICATION_FIELD_LIMIT->{ttl}{$qv->data->{type}};

                    throw Exception::Validator::Fields gettext('%s must between %s and %s', 'ttl', $limit->[0],
                        $limit->[1])
                      if $value < $limit->[0] || $value > $limit->[1];
                },
            },
            api => 1
        },
        caption => {
            i18n       => TRUE,
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                type   => 'hash',
                fields => {
                    map {
                        $_ => {
                            len_min => 1,
                            check   => sub {
                                my ($qv, $val) = @_;

                                my $value = length($val);
                                my $limit = $NOTIFICATION_FIELD_LIMIT->{caption}{$qv->data->{type}};

                                throw Exception::Validator::Fields gettext('%s must between %s and %s', 'caption', 1,
                                    $limit)
                                  if $value < 1 || $value > $limit;
                            },
                          }
                      } @locales
                },
            },
        },
        # TODO: справочник иконок
        icon_id => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {in => $NOTIFICATION_ICON},
        },
        message => {
            i18n       => TRUE,
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                type   => 'hash',
                fields => {
                    map {
                        my $locale = $_;
                        $locale => {
                            len_min => 1,
                            check   => sub {
                                my ($qv, $val) = @_;

                                my $value = length($val);
                                my $limit = $NOTIFICATION_FIELD_LIMIT->{message}{$qv->data->{type}};

                                throw Exception::Validator::Fields gettext('%s must between %s and %s',
                                    'message_' . $locale, 1, $limit)
                                  if $value < 1 || $value > $limit;
                            },
                          }
                      } @locales
                },
            },
        },
        button_caption => {
            i18n       => TRUE,
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'hash',
                fields   => {
                    map {
                        $_ => {
                            optional => TRUE,
                            len_min  => 1,
                            check    => sub {
                                my ($qv, $val) = @_;

                                my $value = length($val);
                                my $limit = $NOTIFICATION_FIELD_LIMIT->{button_caption}{$qv->data->{type}};

                                throw Exception::Validator::Fields gettext('%s must between %s and %s',
                                    'button_caption', 1, $limit)
                                  if $value < 1 || $value > $limit;
                            },
                          }
                      } @locales
                },
            },
        },
        url => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                len_max  => 255,
            },
        },
        expectation_caption => {
            i18n       => TRUE,
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'hash',
                fields   => {
                    map {
                        $_ => {optional => TRUE, len_max => 30, len_min => 1}
                      } @locales
                },
            },
        },
        expected_growth => {
            default    => TRUE,
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                max      => 99999,
                min      => 0,
            },
        },
        roles => {
            default    => TRUE,
            from_opts  => 'from_hash',
            type       => 'array',
            sub_type   => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'array',
                all      => {
                    type => 'int_un',
                    in   => [
                        map {
                            $_->{id}
                          } @{$self->rbac->get_roles()}
                    ],
                },
            },
        },
        logins => {
            default    => TRUE,
            from_opts  => 'from_hash',
            type       => 'array',
            sub_type   => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'array',
                all      => {len_max => 100, len_min => 1},
            },
        },
        fields_depends => {
            depends_on => ['id'],
            get        => sub {
                {};
            },
            type => 'complex'
        },
        public_id => {
            db      => TRUE,
            db_expr => 'id',
            type    => 'string',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;
    return {
        db_accessor => 'partner_db',
        fields      => {
            id         => {type => 'number',     label => d_gettext('ID')},
            caption    => {type => 'text',       label => d_gettext('Caption')},
            multistate => {type => 'multistate', label => d_gettext('Status')},
            short_name => {type => 'text',       label => d_gettext('Short name')},
            type       => {
                type   => 'dictionary',
                label  => gettext('Type'),
                values => [map {{id => $_, label => $_}} @$NOTIFICATION_TYPE],
            },
            view_type => {
                type   => 'dictionary',
                label  => gettext('View Type'),
                values => [map {{id => $_, label => $_}} @$NOTIFICATION_VIEW_TYPE],
            },
        },
    };
}

sub get_editable_fields_depends {
    [qw(id type)];
}

sub get_available_fields_depends {
    [qw(id type)];
}

sub get_actions_depends {
    [qw(id multistate)];
}

sub get_opts_schema_name {'notification_opts'}

# число (0 - new; 1 - working; 2 - expired; 3 - deleted; )
sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_gettext('Stopped'),
        multistates => [
            [working     => d_gettext('Working')],
            [deleted     => d_gettext('Deleted')],
            [need_update => d_gettext('Need update')],
            [updating    => d_gettext('Updating')],
        ],
        right_actions => {
            delete       => d_gettext('Delete'),
            add          => d_gettext('Add'),
            edit         => d_gettext('Edit'),
            start        => d_gettext('Start'),
            restore      => d_gettext('Restore'),
            start_update => d_gettext('Start update'),
            stop_update  => d_gettext('Stop update'),
        },
        right_group        => [notification_action => d_gettext('Right to manage notification')],
        right_name_prefix  => 'notification_action_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'start',
                from      => 'not (working or deleted)',
                set_flags => ['working', 'need_update']
            },
            {
                action => 'edit',
                from   => '__EMPTY__',
            },
            {
                action    => 'edit',
                from      => 'not (deleted or __EMPTY__)',
                set_flags => ['need_update']
            },
            {
                action      => 'delete',
                from        => 'not deleted',
                set_flags   => ['deleted'],
                reset_flags => ['working'],
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub api_can_add {TRUE}

sub get_available_fields {
    my ($self, $obj) = @_;

    unless ($self->check_rights("notification_view")) {
        return {};
    }

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %res = map {$_ => TRUE} qw(
      type
      view_type
      ttl
      caption
      icon_id
      message
      button_caption
      url
      expectation_caption
      expected_growth
      roles
      logins
      short_name
      );

    return \%res;
}

sub get_editable_fields {
    my ($self, $object) = @_;

    my %res = map {$_ => TRUE} qw(
      view_type
      ttl
      caption
      icon_id
      message
      button_caption
      url
      expectation_caption
      expected_growth
      roles
      logins
      short_name
      );

    if ($object->{type} eq 'auto') {
        delete @res{qw(roles logins)};
    }
    return \%res;
}

sub on_action_delete {
    my ($self, $obj, %opts) = @_;
    $obj = $self->_get_object_fields($obj, [qw(id short_name)]);

    my $ms = $self->user_notifications->get_multistates_by_filter('not deleted_by_user');
    my $count =
      $self->partner_db->user_notifications->delete(
        $self->partner_db->filter(['AND', [[notification_id => '=' => \$obj->{id}], [multistate => 'IN' => \$ms]]]));

    my $message = $opts{message};
    unless (defined $message) {
        my $cur_user = $self->app->get_option('cur_user', {id => 0, login => 'undef'});
        $message = sprintf "by user %s(%d)", $cur_user->{login}, $cur_user->{id};
    }
    INFOF("notification id=%d and %d user_notification was deleted because: '%s'", $obj->{id}, $count, $message);

    Utils::MonitoringUtils::send_to_graphite(
        interval => 'five_min',
        path     => 'user_notification.delete.' . $obj->{short_name},
        value    => $count,
        solomon  => {
            notification_name => $obj->{short_name},
            sensor            => 'user_notification.delete',
        }
    );
}

sub api_available_actions {
    return qw(delete edit restore start);
}

sub get_fields_depends { }

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'type'}) {
        $result{'type'} = $NOTIFICATION_TYPE;
    }

    if ($need_fields->{'view_type'}) {
        $result{'view_type'} = $NOTIFICATION_VIEW_TYPE;
    }

    if ($need_fields->{'icon_id'}) {
        $result{'icon_id'} = $NOTIFICATION_ICON;
    }

    return \%result;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter->and(['AND', [\undef]])
      unless $self->check_rights('notification_view');

    return $filter;
}

1;
