package Application::Model::PICategoriesDict;

use qbit;

use base qw(
  Application::Model::DBManager::Base
  RestApi::DBModel
  Utils::Dict
  );

use Tree::Simple 'use_weak_refs';

use Utils::JSON qw(fix_type_for_complex);

use Utils::Logger qw(ERRORF);
use PiConstants qw($CPM_MULTIPLIER);

sub accessor      {'picategories_dict'}
sub db_table_name {'picategories_dict'}

__PACKAGE__->model_accessors(
    partner_db       => 'Application::Model::PartnerDB::PICategories',
    tns_dict_article => 'Application::Model::TNSDict::Article',
);

__PACKAGE__->model_fields(
    category_id => {db => TRUE, pk      => TRUE, default => TRUE,     type => 'number', adjust_type => 'str',},
    parent_id   => {db => TRUE, default => TRUE, type    => 'string', api  => 1,        adjust_type => 'str',},
    level       => {db => TRUE, default => TRUE, type    => 'number', api  => 1,        adjust_type => 'str',},
    name        => {db => TRUE, default => TRUE, i18n    => TRUE,     type => 'string', api         => 1},
    public_id   => {
        db          => TRUE,
        db_expr     => 'category_id',
        type        => 'number',
        api         => 1,
        adjust_type => 'str',
    },
    available_fields => {
        get => sub {
            return $_[0]->model->get_available_fields();
        },
        api      => 1,
        type     => 'complex',
        fix_type => \&fix_type_for_complex,
    }
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        category_id => {type => 'number', label => d_gettext('Category ID')},
        parent_id   => {type => 'number', label => d_gettext('Parent ID')},
        level       => {type => 'number', label => d_gettext('Level')},
        name        => {type => 'text',   label => d_gettext('Category name')},
    }
);

sub get_product_name {gettext('picategories_dict')}
sub api_check_public_id {$_[1] =~ /^\d+$/}

sub get_category_tree {
    my ($self) = @_;

    if ($self->{tree}) {
        return $self->{tree};
    }

    # Получаем данные из базы
    my $categoryes_list = $self->partner_db->picategories_dict->get_all(
        fields   => ['category_id', 'parent_id', 'name', 'level'],
        order_by => ['level',       'name'],
    );

    throw gettext('Table "%s" is empty', 'picategories_dict') if @$categoryes_list == 0;

    # Создаем дерево
    # Тут хранятся все объексты дерева (ветви и листья).
    # Ключ - category_id (айдишник) из базы, значение - объект Tree::Simple с
    # элементом дерева
    my %ids;

    if ($categoryes_list->[0]{'level'} != 0) {
        throw gettext("First element should be on level 0");
    }

    # Корневой элемент дерева, category_id = 0, name = 'ОСНОВНАЯ ГРУППА'
    my $tree = Tree::Simple->new(
        {
            category_id => $categoryes_list->[0]{'category_id'},
            name        => $categoryes_list->[0]{'name'},
        }
    );

    $ids{$categoryes_list->[0]{'category_id'}} = $tree;

    foreach my $category (@$categoryes_list) {

        next if $category->{'level'} == 0;
        unless ($ids{$category->{'parent_id'}}) {
            throw "No parent '$category->{'parent_id'}' for entry '$category->{'category_id'}'";
        }

        my $element = Tree::Simple->new(
            {
                category_id => $category->{'category_id'},
                name        => $category->{'name'},
            }
        );

        $ids{$category->{'category_id'}} = $element;
        $ids{$category->{'parent_id'}}->addChild($element);

    }

    $self->{tree} = $tree;

    return $tree;
}

sub get_complete_category {
    my ($self, $list) = @_;
    return $self->get_cmp_from_tree($list, $self->get_category_tree);
}

sub get_available_fields {
    my ($self) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub api_available_actions {()}

# методы для маппинга

sub map_picategories_from_articles {
    my ($self, $orig_articles) = @_;

    my $category_map = $self->get_category_map();
    my $articles     = $self->tns_dict_article->get_complete_articles($orig_articles);
    my $picategories = {};
    foreach my $elem_tns (keys %$articles) {
        foreach my $elem_pi (@{$category_map->{TNS_PI}->{$elem_tns}}) {
            $self->update_category_cpm(
                {
                    cat           => $picategories,
                    id            => $elem_pi,
                    cpm           => $articles->{$elem_tns}->{value},
                    from_articles => TRUE
                }
            );
        }
    }
    return [sort {$a->{'id'} <=> $b->{'id'}} values %$picategories];
}

sub map_articles_from_picategories {
    my ($self, $orig_picategories) = @_;

    my $category_map = $self->get_category_map();
    my $picategories = $self->get_complete_category($orig_picategories);
    my $articles     = {};
    foreach my $elem_pi (keys %$picategories) {
        foreach my $elem_tns (@{$category_map->{PI_TNS}->{$elem_pi}}) {
            $self->update_category_cpm(
                {
                    cat               => $articles,
                    id                => $elem_tns,
                    cpm               => $picategories->{$elem_pi}->{value},
                    from_picategories => TRUE
                }
            );
        }
    }
    return [values %$articles];
}

sub update_category_cpm {
    my ($self, $opts) = @_;

    my $cpm = $opts->{cpm} / $CPM_MULTIPLIER;
    if (exists($opts->{cat}->{$opts->{id}})) {
        $cpm = $opts->{cat}->{$opts->{id}}->{cpm} if $opts->{cat}->{$opts->{id}}->{cpm} < $cpm;
    }
    $opts->{cat}->{$opts->{id}} = {id => $opts->{id}, cpm => $cpm};
}

sub get_category_map {
    my ($self, $opts) = @_;

    my $category_map = $self->app->api_yt_partner_picategory_mapping->get_picategory_articles_mapping();

    return $category_map;
}

1;
