package Application::Model::Page;

=encoding UTF-8

=cut

=head1 DESCRIPTION

Базовый класс для всех обычных пейджей.

=cut

use qbit;
use Exception::BK::Protected;

use base qw(
  Application::Model::PageMinimal
  );

use Utils::Logger qw( ERROR WARN INFO );
use Utils::InternalPagesInEurope qw(remove_european_domains_from_mirrors);

use PiConstants qw(
  $ADBLOCK_PARTNER_ROLE_ID
  $BLOCKS_WITH_BK_DATA_IN_JAVA
  $DIRECT_PLACE_ID
  $M_RU_TEXT_USER_ID
  );
use PiVariables;

use Exception::Denied;
use Exception::Validation::BadArguments;

__PACKAGE__->abstract_methods(qw( get_block_model_names));

sub get_block_models_for_available_levels {
    my ($self) = @_;

    return $self->get_block_models();
}

sub get_need_update_in_bk_block_models {@{$_[0]->get_block_model_names()}}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        %{$class->SUPER::get_structure_model_accessors()},
        api_balance       => 'QBit::Application::Model::API::Yandex::Balance',
        excluded_domains  => 'Application::Model::Excluded::Domains',
        excluded_phones   => 'Application::Model::Excluded::Phones',
        mail_notification => 'Application::Model::MailNotification',
        page_options      => 'Application::Model::PageOptions',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        view_field__statistics_exists => d_gettext('Right to view field "statistics_exists"'),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        #'available_blocks',
        statistics_exists => {
            depends_on   => [qw(multistate)],
            label        => d_gettext('Statistics exists'),
            check_rights => 'view_field__statistics_exists',
            get          => sub {
                return !!1;
            },
            type        => 'boolean',
            api         => 1,
            adjust_type => 'str',
        },
        available_levels => {
            depends_on => ['page_id'],
            get        => sub {
                my ($self, $obj) = @_;

                # TODO удалить следующую строчку в одном из следующих ПР, например по PI-15793
                return {} unless $obj->{'page_id'};

                return ($self->{'__AVAILABLE_LEVELS__'}{$obj->{'page_id'}} // {});
            },
            type        => 'complex',
            api         => 1,
            adjust_type => 'hash_int',
        },
        status => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'need_update')
                  || $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'updating') ? 'pending' : 'sync';
            },
            type => 'string',
            api  => 1,
        },
        mirrors => {
            depends_on => ['id', 'page_id'],
            get        => sub {
                [];
            },
            type     => 'array',
            sub_type => 'string',
        },
        excluded_domains => {
            label      => d_gettext('Excluded domains'),
            depends_on => ['page_id', 'excluded_domains.domain'],
            get        => sub {
                my ($self, $obj) = @_;

                # TODO удалить следующую строчку в одном из следующих ПР, например по PI-15793
                return [] unless $obj->{'page_id'};

                return ($self->{'excluded_domains'}{$obj->{'page_id'}} // []);
            },
            type       => 'array',
            sub_type   => 'string',
            need_check => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'domain',},
            },
            api => 1,
        },
        excluded_phones => {
            label      => d_gettext('Excluded phones'),
            depends_on => ['page_id', 'excluded_phones.phone'],
            get        => sub {
                my ($self, $obj) = @_;

                # TODO удалить следующую строчку в одном из следующих ПР, например по PI-15793
                return [] unless $obj->{'page_id'};

                return ($self->{'excluded_phones'}{$obj->{'page_id'}} // []);
            },
            type       => 'array',
            sub_type   => 'string',
            need_check => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'phone',},
            },
            api => 1,
        },

        #products
        map {
            my $field = "${_}_count";
            (
                $field => {
                    forced_depends_on => [qw(page_id is_deleted)],
                    get               => sub {
                        my ($fields, $row) = @_;

                        # TODO удалить следующую строчку в одном из следующих ПР, например по PI-15793
                        return 0 unless $row->{'page_id'};

                        return 0 if $row->{'is_deleted'};

                        return ($fields->{$field}{$row->{'page_id'}} // 0);
                    },
                    type => 'number',
                    api  => 1,
                }
            )
          } @{$self->get_block_model_names()}
    };
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    my $ids      = $opts{'ids'}      //= array_uniq(map {$_->{'id'}      // ()} @$result);
    my $page_ids = $opts{'page_ids'} //= array_uniq(map {$_->{'page_id'} // ()} @$result);

    {
        # fields [block_model]_count and available_fields are processed together
        my @block_model_names = @{$self->get_block_model_names()};

        my $block_models_map = {};
        my $resources;
        foreach my $block_model (@block_model_names) {
            my $field = "${block_model}_count";

            next unless $fields->need($field);

            $resources //=
              {map {$_->{'resource'} => $_->{'methods'}}
                  @{$self->resources->get_available_resources(\@block_model_names)}};

            unless ($resources->{$block_model}) {
                $fields->{$field} = {};

                next;
            }

            $block_models_map->{$block_model}{'field'} = $field;
        }

        if ($fields->need('available_levels')) {
            map {$block_models_map->{$_->accessor()}{'level'} = $_->get_statistics_level()}
              $self->get_block_models_for_available_levels();
        }

        foreach my $block_model (sort keys(%$block_models_map)) {
            my $cnt = $self->$block_model->get_cnt($page_ids) // {};

            if (exists($block_models_map->{$block_model}{'field'})) {
                my $field = $block_models_map->{$block_model}{'field'};
                $fields->{$field} = $cnt;
            }

            if (exists($block_models_map->{$block_model}{'level'})) {
                foreach my $page_id (keys(%$cnt)) {
                    next unless $cnt->{$page_id};
                    my $level = $block_models_map->{$block_model}{'level'};
                    $fields->{'__AVAILABLE_LEVELS__'}{$page_id}{$level} = TRUE;
                }
            }
        }
    }
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    if (exists($opts->{'mirrors'})) {
        $opts->{'mirrors'} =
          [map {get_domain($_, ascii => 1, with_port => 1) // "BAD_DOMAIN: '$_'"} @{$opts->{'mirrors'}}];

        my $id = $self->hook_stash->get('id')->{id};
        my $mirrors =
          [map {get_domain(ref($_) eq 'HASH' ? $_->{domain} : $_, ascii => 1, with_port => 1)}
              @{$self->get($id, fields => ['mirrors'])->{mirrors}}];

        for my $mirror (@{$opts->{'mirrors'}}) {
            next if in_array($mirror, $mirrors);
            $self->mirrors->check($self->hook_stash->get('id')->{id}, $mirror);
        }
    }
    $opts->{'excluded_domains'} = [map {get_domain($_) // "BAD_DOMAIN: '$_'"} @{$opts->{'excluded_domains'}}]
      if exists($opts->{'excluded_domains'});
    $opts->{'excluded_phones'} = [map {get_normalized_phone($_) // $_} @{$opts->{'excluded_phones'}}]
      if exists($opts->{'excluded_phones'});

    $self->SUPER::hook_fields_processing_before_validation($opts);
}

sub on_action_delete {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->get_view_edit_rights_for_all_block('stop');
    $self->do_all_block_action($obj, 'delete_with_page');
    undef $tmp_rights;

    my $object = $self->_get_object_fields($obj, [qw(caption owner_id page_id)]);
    $object->{accessor} = $self->accessor;
    $self->mail_notification->add_when_delete_page($object) unless ($opts{suppress_mail_notification});
}

sub on_action_restore {
    my ($self, $obj) = @_;

    $self->SUPER::on_action_restore($obj);

    $self->do_all_block_action($obj, 'restore_with_page');

    $self->maybe_do_action($obj, 'start_testing');
}

sub on_action_set_need_update {
    my ($self, $obj) = @_;
    $self->SUPER::on_action_set_need_update($obj);
}

sub has_active_agreement {
    my ($self, $obj) = @_;

    my $client_id = $self->_get_object_fields($obj, [qw(client_id)])->{'client_id'};

    return $self->agreement_checker->has_agreement_for_any_product_for_today(
        client_id => $client_id,
        products  => $self->get_block_model_names(),
    );
}

sub on_action_start {
    my ($self, $obj) = @_;

    if ($self->is_external_page()) {
        unless ($self->has_active_agreement($obj)) {
            throw Exception::Denied $self->get_agreement_error_message();
        }
    }

    $self->do_all_block_action($obj, 'start');

    $self->do_action($obj, 'set_need_update');
}

sub on_action_start_testing {
    my ($self, $obj) = @_;

    $self->do_all_block_action($obj, 'start');

    $self->do_action($obj, 'set_need_update');
}

sub on_action_stop {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->get_view_edit_rights_for_all_block('stop');
    $self->do_all_block_action($obj, 'stop');
    undef $tmp_rights;

    $self->do_action($obj, 'set_need_update');

    my $object = $self->_get_object_fields($obj, [qw(caption owner_id page_id)]);
    $object->{accessor} = $self->accessor;
    $self->mail_notification->add_when_stop_page($object) unless ($opts{suppress_mail_notification});
}

sub on_action_stop_testing {
    my ($self, $obj, %opts) = @_;

    $self->do_all_block_action($obj, 'stop');

    $self->do_action($obj, 'set_need_update');
}

sub on_action_stop_update {
    my ($self, $obj) = @_;
    $self->SUPER::on_action_stop_update($obj);
}

sub throw_error_by_action {
    my ($self, $object, $action) = @_;
    $self->SUPER::throw_error_by_action($object, $action);
    return FALSE;
}

sub cron_init_campaigns {
    my ($self) = @_;

    my $fields    = $self->get_model_fields();
    my $campaigns = $self->get_all(
        fields => [grep {$fields->{$_}} qw(client_id domain id multistate page_id)],
        # TODO изменить следующую строку в одном из следующих ПР, например по PI-15793
        # filter => {multistate => $self->get_multistate_by_action('register_in_balance')},
        filter => [
            AND => [
                {multistate => $self->get_multistate_by_action('register_in_balance')},
                [$self->get_page_id_field_name() => 'IS NOT' => undef],
            ]
        ],
    );

    foreach my $campaign (@$campaigns) {
        try {
            $self->partner_db->transaction(
                sub {
                    if ($self->check_multistate_flag($campaign->{'multistate'}, 'protected')) {
                        INFO {
                            extra   => {campaign => $campaign,},
                            message => 'Can\'t send protected page to BK',
                        };
                    } else {
                        $self->do_action($campaign, 'register_in_balance');
                        # Действия register_in_balance и set_need_update
                        # возможны почти из одних и тех же состояний.
                        #
                        # Теоретически возможна ситуация импортирования
                        # внутренних площадок из БК уже в состоянии protected,
                        # но при этом готовых к register_in_balance
                        if ($self->check_action($campaign, 'set_need_update')) {
                            $self->do_action($campaign, 'set_need_update');
                            $self->update_in_bk({id => $campaign->{'id'}});
                        } else {
                            WARN {
                                extra   => {campaign => $campaign,},
                                message => 'Can\'t update in BK after register in balance',
                            };
                        }
                    }
                }
            );
        }
        catch {
            my ($exception) = @_;
            ERROR $exception;
        };
    }
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = (
        %{$self->SUPER::get_need_update_in_bk_fields()},
        map {$_ => TRUE} (
            qw(
              excluded_domains
              excluded_phones
              mirrors
              performance_tgo_disable
              )
        )
    );

    return \%fields_need_update;
}

sub update_in_bk {
    my ($self, $filter, %opts) = @_;
    return $self->SUPER::update_in_bk($filter, %opts);
}

sub do_action {
    my ($self, $object, $action, %opts) = @_;
    return $self->SUPER::do_action($object, $action, %opts);
}

sub do_all_block_action {
    my ($self, $obj, $action, %opts) = @_;

    #надо что-то сделать с правами
    my $tmp_rights = $self->app->add_tmp_rights('view_technical_rtb_block',);

    my $page_id = $self->_get_object_fields($obj, [qw(page_id)])->{'page_id'};

    my @block_models = defined $opts{models} ? (map {$self->$_} @{$opts{models}}) : $self->get_block_models();

    foreach my $block_model (sort {$a->accessor() cmp $b->accessor()} @block_models) {
        $block_model->multi_do_action($page_id, $action, %opts);
    }
}

sub get_and_check_domain {
    my ($self, $domain) = @_;

    $domain = get_domain($domain);

    return $domain;
}

sub get_available_block_model_names {
    my ($self, $rec) = @_;

    my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('set_need_update'));

    my $is_page_updateable = $self->check_multistate_action($rec->{'multistate'}, 'set_need_update');

    return [] unless $is_page_updateable;

    my $is_assistant = $self->check_rights('is_assistant');
    my $accessor     = $self->accessor();

    my $can_edit_page = TRUE;    # in most cases the user sees this page because is is an owner or a manager
    my $assistant_role_only;
    my $can_add_adblock;

    if ($is_assistant) {
        my $cur_user = $self->app->get_option('cur_user', {});

        # assistant can get access to this page even if he is not an owner or a manager
        # we check for this situation below
        if ($rec->{'owner_id'} && $rec->{'assistants'}) {
            my $user_id = $cur_user->{'id'} // -1;

            $assistant_role_only = ($user_id != $rec->{'owner_id'});

            if ($assistant_role_only) {
                my @can_edit_as_assistant =
                  grep {$user_id eq $_->{'user_id'} && $_->{'can_edit'}} @{$rec->{'assistants'}};
                $can_edit_page = FALSE unless @can_edit_as_assistant;

                if ($can_edit_page && $accessor eq 'context_on_site_campaign') {
                    my $cur_user_roles = $cur_user->{'roles'} // $self->rbac->get_cur_user_roles() // {};

                    $can_add_adblock = TRUE
                      if grep {$_->{'id'} == $ADBLOCK_PARTNER_ROLE_ID} @{$rec->{'owner'}{'roles'}};
                }
            }
        }
    }

    return [] unless $can_edit_page;

    my @block_model_names = @{$self->get_block_model_names()};
    my %block_model_names = map {$_ => TRUE} @block_model_names;

    # Для Тематических площадок проверяем - можно ли добавлять блоки AdFox и AdBlock
    if ($accessor eq 'context_on_site_campaign') {
        if (!defined($rec->{'tag_id'}) || ($assistant_role_only && !$can_add_adblock)) {
            # Удаляем "context_on_site_adblock" из списка
            delete($block_model_names{'context_on_site_adblock'});
        }

        if ($rec->{'owner_id'} == $M_RU_TEXT_USER_ID) {
            #PI-8722
            delete($block_model_names{'context_on_site_rtb'});
        }
    }

    if ($accessor eq 'video_an_site') {
        my $block_types_on_page = $rec->{'block_types_on_page'};
        my $categories          = $rec->{'categories'};

        foreach my $block_accessor (keys(%block_model_names)) {
            my $incompatible_platforms = $self->$block_accessor->get_incompatible_platforms();

            if (grep {$rec->{'platform'} == $_} @$incompatible_platforms) {
                delete($block_model_names{$block_accessor});
                next;
            }

            my $types = $self->$block_accessor->get_types_ids();

            my $max_blocks_type = $self->$block_accessor->get_max_blocks_by_type();

            my $can_add_block = FALSE;
          CATEGORIES: foreach my $category (@{$rec->{'categories'}}) {
                next
                  if $category->{'archived'}
                      || (   $self->$block_accessor->without_yandex_video_search()
                          && $self->video_an_site_categories->is_yandex_video_search($category->{'id'}));

                foreach (@$types) {
                    if (   !$block_types_on_page->{$category->{'id'}}
                        || !$block_types_on_page->{$category->{'id'}}{$_}
                        || $block_types_on_page->{$category->{'id'}}{$_} < $max_blocks_type->{$_})
                    {
                        $can_add_block = TRUE;
                        last CATEGORIES;
                    }
                }
            }

            delete($block_model_names{$block_accessor}) unless $can_add_block;
        }
    }

    # Check "*_add" rights
    foreach my $block_accessor (keys(%block_model_names)) {
        delete($block_model_names{$block_accessor})
          unless $self->$block_accessor->check_rights($self->$block_accessor->get_rights_by_actions('add'));
    }

    return [grep {$block_model_names{$_}} @block_model_names];
}

sub get_block_models {
    my ($self) = @_;

    return map {$self->$_} @{$self->get_block_model_names()};
}

sub get_next_block_id {
    my ($self, $page_id) = @_;

    throw gettext('Must call in transaction') unless $self->partner_db->{'__SAVEPOINTS__'};

    my $block_seq_table = $self->block_seq_db_table;
    my $pk              = $block_seq_table->primary_key->[0];

    $block_seq_table->add(
        {
            $pk           => $page_id,
            next_block_id => 1,
        },
        ignore => TRUE,
    );

    my $row = $block_seq_table->get(
        $page_id,
        fields     => ['next_block_id'],
        for_update => TRUE
    );

    my $next_id_from_db = $row->{'next_block_id'};
    my $next_id         = $self->get_next_id($next_id_from_db);

    $block_seq_table->edit($page_id, {next_block_id => $next_id + 1});

    return $next_id;
}

sub get_next_id {
    my ($self, $id) = @_;

    return $id;
}

sub is_context_page {$_[0]->DOES('Application::Model::Role::Has::Context')}

sub is_external_page {$_[0]->isa('Application::Model::Page::MIXIN::External')}

sub is_internal_page {$_[0]->isa('Application::Model::Page::MIXIN::Internal')}

sub is_search_page {$_[0]->DOES('Application::Model::Role::Has::Search')}

sub is_can_per_page_invite {FALSE}

sub related_models {
    my ($self, $key_fields) = @_;

    return {
        %{$self->SUPER::related_models($key_fields)},
        excluded_domains => {
            accessor => 'excluded_domains',
            filter   => sub {
                my ($self, $results) = @_;
                return {page_id => array_uniq(map {$_->{'page_id'} // ()} @$results)};
            },
            key_fields => ['page_id'],
            value_type => 'array_domain',
        },
        excluded_phones => {
            accessor => 'excluded_phones',
            filter   => sub {
                my ($self, $results) = @_;
                return {page_id => array_uniq(map {$_->{'page_id'} // ()} @$results)};
            },
            key_fields => ['page_id'],
            value_type => 'array_phone',
        },
    };
}

sub get_view_edit_rights_for_all_block {
    my ($self, $action) = @_;

    my @rights;
    foreach my $block_model ($self->get_block_models()) {
        foreach my $right (qw(view edit)) {
            push @rights, $block_model->get_right($right);
        }
        push @rights, $block_model->get_rights_by_actions($action);
    }

    return $self->app->add_tmp_rights(@rights);
}

sub _data_for_bk {
    my ($self, $page, %opts) = @_;

    my %data = hash_transform(
        $page,
        [
            qw(
              business_unit
              client_id
              create_date
              domain
              fast_context
              is_yandex_page
              mobile_app_mode
              only_picture
              page_id
              product_type
              store
              update_time
              )
        ],
        {
            bk_state_name => 'state',
            view_images   => 'pictures_enabled',
            caption       => 'page_caption',
        }
    );

    $data{'product_id'} = $self->accessor();

    if (exists($data{'domain'})) {
        $data{$_} = $data{'domain'} foreach (qw(name description));
    }

    if ($page->{'banner_lang_name'}) {
        $data{'banner_language'} = $page->{'banner_lang_name'};
    }

    if ($page->{'page_lang_name'}) {
        $data{'page_language'} = $page->{'page_lang_name'};
    }

    my $page_has_direct = !$self->app->all_pages->is_only_rtb_product_page($page->{'page_id'});
    local $PiVariables::IGNORE_RTB_ON_DIRECT_SWITHER = $page_has_direct;

    _add_blocks_data_for_bk($self, $page, \%data, %opts) unless $opts{page_only};

    if ($self->is_context_page()) {
        if (!exists($data{'places'}) || !exists($data{'places'}{$DIRECT_PLACE_ID})) {
            $data{'places'} = {$DIRECT_PLACE_ID => {}};
        }
    }
    $data{'banners_count'} //= 9;
    $data{'is_pi2'} = 1;
    $data{'places'} = {} if !defined($data{'places'});
    $data{'login'}  = $self->app->users->get_login_by_client_id($page->{'client_id'});

    my $bk_data = {%data, $self->get_bk_data($page)};

    if ($self->can('apply_bk_data_patch')) {
        $bk_data = $self->apply_bk_data_patch($page, $bk_data);
    }

    #https://st.yandex-team.ru/PI-19415
    remove_european_domains_from_mirrors($bk_data);

    return $bk_data;
}

sub get_blocks_with_bk_data_in_java {
    my ($self) = @_;

    return from_json($self->kv_store->get($BLOCKS_WITH_BK_DATA_IN_JAVA) // '{}');
}

sub _add_blocks_data_for_bk {
    my ($self, $page, $data, %opts) = @_;

    my $blocks_in_java = $self->get_blocks_with_bk_data_in_java();

    my $has_blocks_from_java = FALSE;
    foreach my $model ($self->get_need_update_in_bk_block_models()) {
        if ($blocks_in_java->{$model}) {
            $has_blocks_from_java = TRUE;
            next if ($opts{is_from_java} || $opts{without_blocks});
        }

        my %m_params = $self->$model->get_bk_data($page->{'page_id'}, %opts);

        foreach my $key (keys(%m_params)) {
            if ($data->{$key} && ref($m_params{$key}) eq 'HASH') {
                $data->{$key}->{$_} = $m_params{$key}->{$_} foreach (keys(%{$m_params{$key}}));
            } else {
                $data->{$key} = $m_params{$key};
            }
        }
    }

    my ($is_send_to_bssoap, $is_send_to_logbroker, undef) =
      $self->api_bk->get_transports_to_send($self->accessor(), $page->{'page_id'});

    if ($is_send_to_bssoap || $is_send_to_logbroker) {
        # put `RtbBlocks` to `rtb_blocks` because java returns final bk_data with BK keys, we need our keys later
        if ($has_blocks_from_java && $opts{is_from_java}) {
            my $enriched_page =
              $self->api_java_bk_data->enrich_page(page_bk_data => $data, read_only => $opts{read_only});
            $data->{rtb_blocks} = $enriched_page->{RtbBlocks};
            if ($enriched_page->{RtbVideo}) {
                $data->{rtb_video} = $enriched_page->{RtbVideo};
            }
        }
    }

    return 1;
}

=head2 get_page_ids_hash_for_statistics

Возвращает хеш, ключи которого - это Page ID контекстных площадок РСЯ, по
которым нужно обновлять статистику.

    my %page_ids = $self->get_page_ids_hash_for_statistics();

После этого в %page_ids будет:

    (
        167933 => 1,
        167959 => 1,
        ...
    )

=cut

sub get_page_ids_hash_for_statistics {
    my ($self) = @_;

    my %page_ids =
      map {$_->{'page_id'} => TRUE} @{
        $self->get_all(
            fields => [qw(page_id)],
            filter => [
                AND => [
                    # TODO изменить фильтр (удалить следующую строчку) в одном из следующих ПР, например по PI-15793
                    [page_id => 'IS NOT' => undef],
                    (
                        $self->get_multistate_by_name('balance_registered')
                        ? {multistate => $self->get_multistates_by_filter('balance_registered')}
                        : ()
                    ),
                ]
            ],
        )
      };

    return %page_ids;
}

sub may_send_not_balance_registered {FALSE;}

TRUE;
