package Application::Model::Page::Dooh;
# Dooh == Digital out-of-home

use qbit;
use PiConstants qw(:ROLES $BALANCE_VIPTYPE_YANDEX $ADINSIDE_CLIENT_ID);

use Exception::Validator::Fields;

use base qw(
  Application::Model::Page
  Application::Model::Page::MIXIN::External
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::Filters
  Application::Model::Role::Has::Tier
  Application::Model::Role::Page::Has::BlocksLimit
  Application::Model::Role::Page::Has::FieldsModeration
  Application::Model::Role::Page::Has::AssessorWithTestMode
  Application::Model::Role::Page::Has::Tags
  Application::Model::Role::Has::Page::Patch
  );

use PiConstants qw(
  $BOOKMAKER_FLAG_ID
  $CONTEXT_TARGET_TYPE
  );

sub get_page_id_field_name {'page_id'}

sub get_dooh_dummy_domain {'dooh.ya'}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();
    $rights->[0]{'rights'} = {
        hash_transform(
            {%{$rights->[0]{'rights'}}, %{$self->get_external_structure_rights_to_register()}},
            [
                qw(
                  view_search_filters__login
                  ),
                map {$self->get_description_right($_)}
                  qw(
                  add_other
                  edit
                  edit_all
                  edit_field__assistants
                  edit_field__comment
                  view
                  view_action_log
                  view_all
                  view_field__comment
                  view_field__owner
                  view_full_multistate_name
                  view_field__assistants
                  edit_field__is_in_testmode
                  view_field__is_in_testmode
                  )
            ]
        ),
        map {$self->get_description_right($_)}
          qw(
          view_field__login
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $domain = $self->get_dooh_dummy_domain();

    my %model_fields = (
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->SUPER::get_external_structure_model_fields()},
        domain => {
            db      => TRUE,
            db_expr => \$domain,
            type    => 'string',
            api     => 1
        },
        product_type => {db => TRUE, db_expr => \'video'},
        # overwrite
        caption => {
            from_opts  => 'from_hash',
            default    => TRUE,
            need_trim  => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                len_min => 1,
                len_max => 255,
            },
        },
        # overwrite
        available_blocks => {
            depends_on => [qw(multistate owner_id)],
            get        => sub {
                return $_[0]->model->get_available_block_model_names($_[1]);
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1,
        },
        facility_type => {
            from_opts  => 'db_expr',
            type       => 'number',
            need_check => {
                type => 'int_un',
                in   => [sort keys %{$self->facility_type()}],
                msg  => d_gettext("unknown facility type"),
            },
            api => 1,
        },
        is_in_testmode => {
            from_opts  => 'from_hash',
            default    => TRUE,
            type       => 'boolean',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'boolean',
            },
        },
    );

    delete @model_fields{
        qw(
          is_tutby
          is_graysite
          mirrors
          statistics_exists
          cur_user_is_read_assistant
          )
      };

    return \%model_fields;
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();
    $filter->{'fields'} = {
        hash_transform(
            {%{$filter->{'fields'}}, %{$self->get_external_structure_model_filter}},
            [
                qw(
                  id
                  page_id
                  multistate
                  owner
                  owner_id
                  send_time
                  update_time
                  caption_text
                  )
            ]
        )
    };

    my $domain = $self->get_dooh_dummy_domain();
    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        domain => {
            type      => 'text',
            label     => d_gettext('Domain'),
            db_filter => sub {
                return [\$domain => $_[1]->[1] => \$_[1]->[2]];
              }
        },
        domain_text => {type => 'alias', path => [qw(domain)]},
        login       => {type => 'alias', path => [qw(owner login)]},
        caption     => {
            type       => 'json',
            value_type => 'text',
            label      => d_gettext('Caption'),
        }
    };

    return $filter;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => 'id',         label => gettext('ID')},
        {name => 'page_id',    label => gettext('Page ID')},
        {name => 'multistate', label => gettext('Multistate')},
        {name => 'caption',    label => gettext('Caption')},
        (
            $self->check_rights('view_search_filters__login')
            ? {name => 'owner.login', label => gettext('Login')}
            : ()
        )
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    my $accessor = $self->accessor();

    return {
        empty_name  => d_gettext('New'),
        multistates => [
            [working          => d_pgettext('Dooh status', 'Working')],
            [stopped          => d_pgettext('Dooh status', 'Stopped')],
            [deleted          => d_pgettext('Dooh status', 'Archived'), private => TRUE],
            [need_update      => d_pgettext('Dooh status', 'Need update')],
            [updating         => d_pgettext('Dooh status', 'Updating')],
            [read_only        => d_pgettext('Dooh status', 'Read only'), private => TRUE],
            [protected        => d_pgettext('Dooh status', 'Protected')],
            [check_statistics => d_pgettext('Dooh status', 'Check statistics'), private => TRUE, deprecated => TRUE],
            [balance_registered => d_pgettext('Dooh status', 'Added to Balance'), private => TRUE],
            # moderation
            [need_approve => d_pgettext('Page multistate', 'On moderation')],
            [approved     => d_pgettext('Page multistate', 'Approved')],
            [rejected     => d_pgettext('Page multistate', 'Rejected')],
            [blocked      => d_pgettext('Page multistate', 'Blocked')],
            [testing      => d_pgettext('Dooh status',     'Testing')],
        ],
        actions => {
            can_update_in_bk       => d_pgettext('Dooh action', 'Can update in bk'),
            reset_check_statistics => d_pgettext('Dooh action', 'Reset "check_statistics"'),
            restore_block          => d_pgettext('Dooh action', 'Restore block on campaign'),
            start_block            => d_pgettext('Dooh action', 'Start block on campaign'),
        },
        right_actions => {
            add                 => d_pgettext('Dooh action', 'Add'),
            delete              => d_pgettext('Dooh action', 'Archive'),
            edit                => d_pgettext('Dooh action', 'Edit'),
            register_in_balance => d_pgettext('Dooh action', 'Register in Balance'),
            reset_protected     => d_pgettext('Dooh action', 'Reset "protected"'),
            reset_read_only     => d_pgettext('Dooh action', 'Reset "read_only"'),
            restore             => d_pgettext('Dooh action', 'Restore'),
            set_need_update =>
              {label => d_pgettext('Dooh action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            set_protected => d_pgettext('Dooh action', 'Set "protected"'),
            set_read_only => d_pgettext('Dooh action', 'Set "read_only"'),
            start         => d_pgettext('Dooh action', 'Start'),
            start_update => {label => d_pgettext('Dooh action', 'Start update'), dont_write_to_action_log => TRUE},
            stop => d_pgettext('Indoor action', 'Stop'),
            stop_update => {label => d_pgettext('Dooh action', 'Stop update'), dont_write_to_action_log => TRUE},
            # moderation
            set_need_approve => d_pgettext('Page action', 'Return to moderation'),
            approve          => d_pgettext('Page action', 'Approve'),
            reject           => d_pgettext('Page action', 'Reject'),
            set_blocked      => d_pgettext('Page action', 'Set blocked'),
            reset_blocked    => d_pgettext('Page action', 'Reset blocked'),
            start_testing    => d_pgettext('Dooh action', 'Start testing'),
            stop_testing     => d_pgettext('Dooh action', 'Stop testing'),
        },
        right_group        => [$accessor => d_gettext('Right to manage %s campaigns', $accessor)],
        right_name_prefix  => $accessor . '_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'register_in_balance',
                from      => 'not (balance_registered or deleted)',
                set_flags => ['balance_registered'],
            },
            {
                action      => 'start',
                from        => 'approved and balance_registered and not (working or deleted or protected)',
                set_flags   => ['working'],
                reset_flags => ['stopped']
            },
            {
                action      => 'stop',
                from        => 'working and not (stopped or protected)',
                set_flags   => ['stopped'],
                reset_flags => ['working'],
            },
            {
                action    => 'delete',
                from      => 'not (working or deleted)',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
            },
            {
                action => 'edit',
                from   => 'not (deleted or blocked)'
            },
            {
                action    => 'set_read_only',
                from      => 'not read_only',
                set_flags => ['read_only'],
            },
            {
                action      => 'reset_read_only',
                from        => 'read_only',
                reset_flags => ['read_only'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'start_block',
                from   => 'working and not protected',
            },
            {
                action => 'restore_block',
                from   => 'not deleted',
            },
            {
                action => 'can_update_in_bk',
                from   => 'not (deleted or protected)',
            },
            {
                action    => 'set_protected',
                from      => 'not protected',
                set_flags => ['protected'],
            },
            {
                action      => 'reset_protected',
                from        => 'protected',
                reset_flags => ['protected'],
            },
            {
                action    => 'set_need_update',
                from      => 'not (deleted or protected) and (testing or approved)',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
            # moderation
            {
                action      => 'set_need_approve',
                from        => 'not blocked',
                set_flags   => ['need_approve'],
                reset_flags => ['rejected'],
            },
            {
                action      => 'approve',
                from        => 'need_approve or rejected',
                set_flags   => ['approved'],
                reset_flags => ['need_approve', 'rejected'],
            },
            {
                action      => 'reject',
                from        => 'need_approve and not approved',
                set_flags   => ['rejected'],
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reject',
                from        => 'need_approve and approved',
                reset_flags => ['need_approve'],
            },
            {
                action    => 'set_blocked',
                from      => 'not blocked',
                set_flags => ['blocked'],
            },
            {
                action      => 'reset_blocked',
                from        => 'blocked',
                reset_flags => ['blocked'],
            },
            {
                action    => 'start_testing',
                from      => 'not testing',
                set_flags => ['testing', 'need_update'],
            },
            {
                action      => 'stop_testing',
                from        => 'testing',
                reset_flags => ['testing', 'need_update'],
                set_flags   => ['need_approve'],
            },
        ]
    };
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    $self->SUPER::hook_processing_after_update($opts);

    my $obj = $self->_get_object_fields($self->hook_stash->get('current'), ['multistate']);
    my $multistate = $obj->{multistate};

    if ($self->check_multistate_flag($multistate, 'testing')
        && !$self->hook_stash->get('settings')->{is_in_testmode})
    {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('stop_testing'));
        $multistate = $self->do_action($self->hook_stash->get('id'), 'stop_testing');
    }

    if (
        $self->need_update_in_bk($self->hook_stash->get('fields'))
        && (   $self->check_multistate_flag($multistate, 'approved')
            || $self->check_multistate_flag($multistate, 'testing'))
       )
    {
        $multistate = $self->do_action($self->hook_stash->get('id'), 'set_need_update');
    }
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_tutby_and_assistant_and_robot_assistant($filter);

    return $filter;
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    $self->SUPER::pre_process_fields($fields, $result, %opts);
    $self->Application::Model::Page::MIXIN::External::pre_process_fields($fields, $result, %opts);
}

sub get_actions_depends {
    [qw(id multistate)];
}

sub get_available_fields_depends {
    [qw(id)];
}

sub get_editable_fields_depends {
    [qw(id multistate page_id is_read_only is_in_testmode owner_id)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my %fields = map {$_ => TRUE} keys(%{$self->get_model_fields});
    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s'    => [qw(login comment is_in_testmode)],
            $accessor . '_view_field__owner' => [qw(owner client_id business_unit is_yandex_page is_tutby)],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{$_} = TRUE foreach (qw(facility_type));

    foreach (qw(is_in_testmode)) {
        $fields->{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    if ($self->check_short_rights('add_other')) {
        $fields->{'login'}    = TRUE;
        $fields->{'owner_id'} = TRUE;
    }

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my $res = $self->_get_common_add_edit_fields();

    $res->{$_} = TRUE foreach (
        qw(
        excluded_domains
        excluded_phones)
    );

    $self->handle_field_assistants($data, $res, 'edit_field__assistants');

    # TODO : fix readonly fields
    if ($data->{'is_read_only'}) {
        foreach (
            qw(
            address
            gps
            )
          )
        {
            delete($res->{$_});
        }
    }

    if ($self->check_multistate_flag($data->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (keys(%{$self->get_need_update_in_bk_fields()}),) {
            delete($res->{$_});
        }
    }

    if ($self->check_short_rights("edit_field__is_in_testmode")) {
        # is_in_testmode can only be unset
        # for on_action_edit: $data is patched with opts object
        if ($self->hook_stash->inited) {
            $res->{is_in_testmode} = TRUE if $self->hook_stash->get('current')->{is_in_testmode};
        } else {    # for get_editable_fields: $data is an object in db
            $res->{is_in_testmode} = TRUE if $data->{is_in_testmode};
        }
    }

    return $res;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields      => $self->SUPER::_get_common_add_edit_fields(),
        no_right_fields => ['caption'],
    );

    return $fields;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{is_in_testmode} //= 1;
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    $self->SUPER::hook_save_fields_from_related_models($opts);

    my $settings = $self->hook_stash->get('settings');
    my $related  = $self->hook_stash->get('fields_from_related_models');
    my $mode     = $self->hook_stash->mode('add') ? 'add' : 'edit';

    my @to_update = qw(excluded_domains excluded_phones assistants);

    for my $model (@to_update) {
        if (defined(my $value = $related->{$model})) {
            $self->$model->replace($settings->{'page_id'}, $value, $mode);
        }
    }
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    $self->SUPER::hook_processing_after_insert($opts);

    if ($self->hook_stash->get('settings')->{is_in_testmode}) {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('start_testing'));
        $self->do_action($self->hook_stash->get('id'), 'start_testing');
    } else {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('set_need_approve'));
        $self->do_action($self->hook_stash->get('id'), 'set_need_approve');
    }
}

sub check_action {
    my ($self, $object, $action) = @_;

    return FALSE
      unless $self->check_short_rights('edit');

    $self->QBit::Application::Model::Multistate::DB::check_action($object, $action);
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = $self->get_defaults_thematic_filters($need_fields);

    if ($need_fields->{'facility_type'}) {
        $result{'facility_type'} = $self->get_facility_types();
    }

    return \%result;
}

sub get_facility_types {
    my ($self) = @_;

    my $facility_type = $self->facility_type();
    return [
        map {
            id      => $_,
              type  => $facility_type->{$_}->{'type'},
              code  => $facility_type->{$_}->{'code'},
              label => $facility_type->{$_}->{'label'}->()
        },
        (sort keys %$facility_type)
    ];
}

sub get_fields_depends {
    return {
        #если поменялось поле из ключ, то
        #нужно перезапросить поля из значения
        depends => {},
        #для поля из ключа обязятельно нужно передать поля из значения
        required => {},
    };
}

sub related_models {
    my ($self) = @_;

    return {%{$self->SUPER::related_models()}, %{$self->external_related_models()},};
}

sub can_action_approve {TRUE}

sub on_action_approve { }

sub on_action_register_in_balance {
    my ($self, $obj) = @_;

    my $campaign = $self->_get_object_fields($obj, [qw(id owner create_date client_id page_id)])
      // throw Exception::Validation::BadArguments gettext('Campaign with ID=%s does not exists', $obj->{'id'});

    $self->api_balance->create_or_update_place(
        operator_uid  => $self->get_option(cur_user => {})->{'id'},
        client_id     => $campaign->{'client_id'},
        page_id       => $campaign->{'page_id'},
        campaign_type => 3,                         # Контекстная
        ($campaign->{'client_id'} eq $ADINSIDE_CLIENT_ID ? (viptype => $BALANCE_VIPTYPE_YANDEX) : ()),
        is_tutby    => $campaign->{'owner'}{'is_tutby'},
        create_date => $campaign->{'create_date'},
    );

    my $can_start = $self->agreement_checker->has_agreement_for_any_product_for_today(
        client_id => $campaign->{client_id},
        products  => $self->get_block_model_names(),
    );

    if ($can_start && $self->check_action($campaign->{'id'}, 'start')) {
        $self->do_action($campaign, 'start');
    }
}

sub can_action_reject {TRUE}

sub on_action_reject { }

sub on_action_set_blocked { }

sub on_action_set_need_approve { }

sub on_action_start_testing {
    my ($self, $obj) = @_;

    return $self->_on_test_action($obj, 'start_testing');
}

sub on_action_stop_testing {
    my ($self, $obj) = @_;

    return $self->_on_test_action($obj, 'stop_testing');
}

sub _on_test_action {
    my ($self, $obj, $action) = @_;

    my $tmp_rights = $self->get_view_edit_rights_for_all_block($action);
    $self->do_all_block_action($obj, $action);

    return TRUE;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_need_update_in_bk_fields();

    $fields->{$_} = TRUE foreach (
        qw(
        caption
        facility_type
        is_in_testmode
        )
    );

    return $fields;
}

sub get_bk_data {
    my ($self, $page) = @_;

    return (
        target_type => $CONTEXT_TARGET_TYPE,
        is_in_testmode =>
          ($page->{is_in_testmode} // $self->check_multistate_flag($page->{multistate}, 'testing') || 0),
    );
}

sub api_available_actions {
    return qw(
      delete
      restore
      start
      stop
      edit
      set_read_only
      reset_read_only
      );
}

sub api_can_edit {TRUE}
sub api_can_add  {TRUE}

sub hook_owner_processing {
    my ($self, $opts) = @_;

    $self->SUPER::hook_owner_processing($opts);

    throw Exception::Validation::BadArguments gettext('Username must belong to the %s',
        $self->rbac->get_role($self->get_own_role)->{name})
      unless $self->rbac->get_roles_by_user_id($opts->{'owner_id'})->{$self->get_own_role};
}

TRUE;
