package Application::Model::Page::InDoor;

use qbit;
use base qw( Application::Model::Page::Dooh );

use PiConstants qw(:ROLES $INDOOR_DUMMY_DOMAIN $INDOOR_FACILITY_TYPE);

sub facility_type {$INDOOR_FACILITY_TYPE}

sub accessor      {'indoor'}
sub db_table_name {'indoor'}

sub get_product_name {gettext('indoor')}

sub get_block_model_names {
    ['indoor_block'];
}

sub get_testing_tag {'indoor-testing-page'}

sub get_opts_schema_name {'indoor_opts'}

sub get_dooh_dummy_domain {$INDOOR_DUMMY_DOMAIN}

sub get_own_role {$INDOOR_PARTNER_ROLE_ID}

sub block_seq_db_table {$_[0]->partner_db->indoor_block_seq}

sub get_structure_model_accessors {
    return {
        partner_db    => 'Application::Model::PartnerDB',
        indoor_block  => 'Application::Model::Block::InDoor',
        api_http_maps => 'Application::Model::API::Yandex::Maps',
        users         => 'Application::Model::Users',
    };
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        # overwrite
        product_type     => {db => TRUE, db_expr => \'indoor'},
        available_blocks => {
            depends_on => [qw(multistate owner_id)],
            get        => sub {
                return $_[0]->model->get_available_block_model_names($_[1]);
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1,
        },
        address => {
            from_opts  => 'from_hash',
            need_check => {
                len_min => 10,
                len_max => 512,
            },
            need_trim => TRUE,
            type      => 'string',
            api       => 1
        },
        gps => {
            from_opts  => 'db_expr',
            need_check => {type => 'gps'},
            need_trim  => TRUE,
            type       => 'string',
            api        => 1
        },
        business_oid => {
            from_opts  => 'db_expr',
            depends_on => [qw(facility_type gps)],
            need_check => {
                optional => 1,
                check    => sub {
                    my ($qv, $business_oid) = @_;
                    return if $qv->app->facility_type()->{$qv->data->{facility_type}}->{skip_business_oid};
                    my $result = $qv->app->api_http_maps->get_companies_by_ll($qv->data->{gps});
                    my %map = map {$_->{business_oid} => 1} @$result;
                    throw Exception::Validator::Fields gettext("Page address and organization address do not match")
                      unless $map{$business_oid};
                  }
            },
            need_trim => TRUE,
            type      => 'string',
            api       => 1
        }
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        address => {
            type       => 'json',
            value_type => 'text',
            label      => d_gettext('Address'),
        },
        gps => {
            type       => 'json',
            value_type => 'text',
            label      => d_gettext('GPS'),
        },
        facility_type => {
            type       => 'json',
            value_type => 'number',
            label      => d_gettext('Facility type'),
        },
        business_oid => {
            type       => 'json',
            value_type => 'number',
            label      => d_gettext('Organization id'),
        },
    };

    return $filter;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields      => $self->SUPER::_get_common_add_edit_fields(),
        no_right_fields => [
            qw(
              address
              gps
              business_oid
              )
        ],
    );

    return $fields;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_need_update_in_bk_fields();

    $fields->{$_} = TRUE foreach (
        qw(
        address
        gps
        business_oid
        )
    );

    return $fields;
}

sub get_facility_types {
    my ($self) = @_;

    my $result = $self->SUPER::get_facility_types();

    my $facility_type = $self->facility_type();

    for my $type (@$result) {
        $type->{business_oid_required} = 0 + (!$facility_type->{$type->{id}}->{skip_business_oid});
    }

    return $result;
}

sub get_fields_moderated {
    my ($self) = @_;

    return {
        caption => TRUE,
        address => TRUE,
        gps     => TRUE,
    };
}

sub fix_template {
    my ($self, $qv) = @_;

    my $fields = $qv->template->{'fields'};

    if (defined $qv->data->{facility_type} && $self->facility_type()->{$qv->data->{facility_type}}->{skip_business_oid})
    {
        $fields->{business_oid} = {eq => undef, optional => TRUE};
    } else {
        # Если работает крон валидатора, то нужно оставить optional
        # Определяем по типу, который выставляется в QBitValidatorChecker
        my $type = exists($fields->{business_oid}{type}) ? $fields->{business_oid}{type} : undef;
        unless (defined($type) && (ref($type) eq 'ARRAY' && $type->[0] eq 'int_un' || $type eq 'int_un')) {
            $fields->{business_oid}{optional} = FALSE;
        }
    }
}

TRUE;
