package Application::Model::Page::MIXIN::External;

use qbit;

use base qw(QBit::Application::Model);

use PiConstants qw($ADINSIDE_CLIENT_ID $OWNERS_USER_FIELDS $SYSTEM_CRON_USER_ID);

use Exception::Validation::BadArguments;

sub get_testing_tag { }

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        assistants                   => 'Application::Model::Assistants',
        user_global_excluded_phones  => 'Application::Model::Users::GlobalExcludedPhones',
        user_global_excluded_domains => 'Application::Model::Users::GlobalExcludedDomains',
        agreement_checker            => 'Application::Model::AgreementChecker',
        rbac                         => 'Application::Model::RBAC',
    };
}

sub check_action {
    my ($self, $object, $action) = @_;

    return FALSE
      unless $self->check_short_rights('edit');

    my $fields = $self->_get_object_fields($object, [qw(cur_user_is_read_assistant multistate)]);

    my $is_protected = $self->check_multistate_flag($fields->{'multistate'}, 'protected');
    if ($is_protected) {
        return FALSE unless $self->check_rights('edit_protected_pages');
    } else {
        return FALSE if $fields->{'cur_user_is_read_assistant'};
    }

    $self->QBit::Application::Model::Multistate::DB::check_action($object, $action);
}

sub external_related_models {
    my ($self) = @_;

    return {
        global_excluded_domains => {
            accessor => 'user_global_excluded_domains',
            filter   => sub {
                return {user_id => array_uniq(map {$_->{'owner_id'} // ()} @{$_[1]})};
            },
            key_fields => ['user_id'],
            value_type => 'array_domain',
        },
        global_excluded_phones => {
            accessor => 'user_global_excluded_phones',
            filter   => sub {
                return {user_id => array_uniq(map {$_->{'owner_id'} // ()} @{$_[1]})};
            },
            key_fields => ['user_id'],
            value_type => 'array_phone',
        },
    };
}

sub get_external_structure_model_fields {
    my ($self) = @_;

    return {
        owner => {
            label      => d_gettext('Owner'),
            depends_on => ['owner_id'],
            get        => sub {
                $_[0]->{'__OWNERS__'}->{$_[1]->{'owner_id'}} // {};
            },
            type => 'users',
            api  => 1,
        },
        client_id => {
            depends_on => [qw(owner)],
            label      => d_gettext('Client ID'),
            get        => sub {
                return $_[1]->{'owner'}{'client_id'};
            },
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        login => {
            depends_on => [qw(owner)],
            label      => d_gettext('Owner login'),
            get        => sub {
                return $_[1]->{'owner'}{'login'};
            },
            type => 'string',
            api  => 1,
        },
        assistants => {
            label      => d_gettext('Assistants'),
            depends_on => ['page_id'],
            get        => sub {
                my ($self, $obj) = @_;

                # TODO удалить следующую строчку в одном из следующих ПР, например по PI-15793
                return [] unless $obj->{'page_id'};

                return [sort {($a->{user_id} // 0) <=> ($b->{user_id} // 0)}
                      @{$self->{'__ASSISTANTS__'}{$obj->{'page_id'}} // []}];
            },
            need_check => {
                optional => TRUE,
                type     => 'array',
                all      => {
                    type   => 'hash',
                    fields => {
                        inviter_id => {type => 'scalar', optional => 1},
                        can_edit   => {type => 'boolean',},
                        user_id    => {type => 'int_un',},
                    },
                    extra => TRUE,
                },
            },
            type        => 'array',
            sub_type    => 'complex',
            api         => 1,
            adjust_type => 'array_hash_str',
        },
        cur_user_is_read_assistant => {
            label      => d_gettext('Assistants'),
            depends_on => ['assistants'],
            get        => sub {

                my %assistants_rights = map {$_->{'user_id'} => $_->{'can_edit'}} @{$_[1]->{'assistants'}};

                my $cur_user_id = $_[0]->model->app->get_option('cur_user', {})->{'id'} // '';
                if ($cur_user_id eq $SYSTEM_CRON_USER_ID) {
                    return FALSE;
                }
                if (exists($assistants_rights{$cur_user_id})) {
                    return TRUE unless $assistants_rights{$cur_user_id};
                }

                return FALSE;
            },
            type        => 'boolean',
            api         => 1,
            adjust_type => 'str',
        },
        is_yandex_page => {
            forced_depends_on => ['owner'],
            get               => sub {
                return $_[1]->{'owner'}{'client_id'} == $ADINSIDE_CLIENT_ID ? 1 : 0;
            },
            type        => 'boolean',
            adjust_type => 'int',
        },
        global_excluded_domains => {
            label      => d_gettext('Global excluded domains'),
            depends_on => ['owner_id', 'global_excluded_domains.domain'],
            get        => sub {
                $_[0]->{'global_excluded_domains'}{$_[1]->{'owner_id'}} // [];
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1,
        },
        global_excluded_phones => {
            label      => d_gettext('Global excluded phones'),
            depends_on => ['owner_id', 'global_excluded_phones.phone'],
            get        => sub {
                $_[0]->{'global_excluded_phones'}{$_[1]->{'owner_id'}} // [];
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1,
        },
        business_unit => {
            depends_on => [qw(owner)],
            label      => d_gettext('Business unit'),
            get        => sub {
                return $_[1]->{'owner'}{'business_unit'};
            },
            type        => 'boolean',
            adjust_type => 'str',
        },
        available_blocks => {
            depends_on => [qw(multistate owner_id assistants)],
            get        => sub {
                return $_[0]->model->get_available_block_model_names($_[1]);
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1,
        },
        is_tutby => {
            depends_on => [qw(owner)],
            label      => d_gettext('Partner TUT.BY'),
            get        => sub {
                return $_[1]->{'owner'}{'is_tutby'};
            },
            type        => 'boolean',
            adjust_type => 'str',
        },
        # redefined as a real DB value in external site pages
        is_graysite => {
            label       => d_gettext('Gray site'),
            db          => TRUE,
            db_expr     => \0,
            type        => 'boolean',
            adjust_type => 'str',
        },
    };
}

sub get_external_structure_model_filter {
    my ($self) = @_;

    my $page_id_field_name = $self->get_page_id_field_name();

    return {
        owner_id => {type => 'number', label => d_gettext('Owner id')},
        owner    => {
            type           => 'subfilter',
            model_accessor => 'users',
            field          => 'owner_id',
            fk_field       => 'id',
            label          => d_gettext('Owners'),
        },
        assistants => {
            type           => 'subfilter',
            model_accessor => 'assistants',
            field          => $page_id_field_name,
            fk_field       => 'page_id',
            label          => d_gettext('Assistants'),
        },
        user_type => {type => 'alias', path => [qw(owner user_type)]},
    };
}

sub get_external_structure_rights_to_register {
    my ($self) = @_;

    return {
        view_search_filters__login     => d_gettext('Right to view search filter "login"'),
        view_search_filters__client_id => d_gettext('Right to view search filter "client_id"'),

        $self->get_description_right(
            'add_other', d_gettext('Right to add %s for other users', $self->get_product_name())
        ),

        map {$self->get_description_right($_)} qw(
          view_field__owner
          view_field__assistants
          edit_field__assistants
          view_full_multistate_name
          view_field__is_tutby
          ),

    };
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    if ($fields->need('owner')) {
        my $tmp_rights = $self->app->add_tmp_rights('users_view_all', 'users_view_field__client_id');
        $fields->{'__OWNERS__'} = {
            map {$_->{'id'} => $_} @{
                $self->users->get_all(
                    fields => [@$OWNERS_USER_FIELDS, 'adfox_info'],
                    filter => {id => array_uniq(map {$_->{'owner_id'} // ()} @$result)},
                )
              }
        };
    }

    if ($fields->need('assistants')) {
        my $page_ids = $opts{'page_ids'} //= array_uniq(map {$_->{'page_id'} // ()} @$result);

        $fields->{'__ASSISTANTS__'} = {};

        foreach (@{$self->assistants->get_assistants(filter => {page_id => $page_ids})}) {
            my $page_id = delete($_->{'page_id'});

            $fields->{'__ASSISTANTS__'}{$page_id} //= [];
            push(@{$fields->{'__ASSISTANTS__'}{$page_id}}, $_);
        }
    }
}

sub handle_field_assistants {
    my ($self, $data, $fields, $right) = @_;

    if (
        $self->check_short_rights($right)
        && (
            $self->check_short_rights('view_all')
            || (   exists $data->{owner_id}
                && defined $data->{owner_id}
                && $self->get_option('cur_user', {id => -1})->{'id'} == $data->{owner_id})
           )
       )
    {
        $fields->{assistants} = 1;
    } else {
        delete $fields->{assistants};
    }
}

sub get_default_owner_login { }

TRUE;
